/*******************************************************************************
*                                                                              *
*                                   Viewmol                                    *
*                                                                              *
*                               M A R C U B . C                                *
*                                                                              *
*                 Copyright (c) Joerg-R. Hill, December 1996                   *
*                                                                              *
********************************************************************************
*
* $Id: marcub.c,v 1.1 1996/12/10 18:41:59 jrh Exp $
* $Log: marcub.c,v $
 * Revision 1.1  1996/12/10  18:41:59  jrh
 * Initial revision
 *
*
*
* This function implements the drawing of the basic patterns of the improved
* marching cube algorithm as described in W. Heiden, T. Goetze, and J. Brick-
* mann: J. Comp. Chem. 14 (1993), 246-250. The calculation of the grid has to be
* done in a separate function. The parameters passed to this function are:
* xstart, ystart, and zstart - the cartesian coordinates of the lower left front
* corner of the volume which is used to draw the isosurface (the origin of the
* grid); grid - a pointer to an array which contains the values of the function
* at each grid point, three dimensional, stored row by row, step - the distance
* between two grid points; xres, yres, and zres - the number of times step has
* to be added to xstart, ystart, or zstart to walk through the whole grid, is
* equal to the number of grid points minus one in each direction; level - a
* floating point number which determines which isosurface to select;
* begin, end, vert, and normal - pointers to the drawing functions.
* The numbering of the cube's corners is as follows:
*
*                              7+------+6
*          y                   /|     /|
*          ^  z              3+------+2|
*          | /                |4+----|-+5
*          |/                 |/     |/
*          +----> x          0+------+1 
*                                      
*
*/
#include<math.h>
#include<stdio.h>
#include<GL/gl.h>
#include "marcub.h"

#define FALSE 0
#define TRUE  1
#define index(i, j, k) ((ny+1)*(nz+1)*(k)+(nz+1)*(j)+(i))

/* Prototypes */
void single_triangle(double, double, double, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void single_rectangle(double, double, double, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void three_triangles(double, double, double, int, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void pattern8(double, double, double, int, int, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void pattern9(double, double, double, int, int, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void pattern11(double, double, double, int, int, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void pattern3a(double, double, double, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void pattern6a(double, double, double, int, int, int, double *, int, int, int, int, void (*)(int), void (*)(), void (*)(), void (*)());
void set_vertex(double, double, double, int, int, double *, int, int, int, double *, double *, int, void (*)(), void (*)());

/* Global variables */
static int edges[8][3]={{1, 3, 4}, {0, 2, 5}, {3, 1, 6}, {2, 0, 7},
                        {5, 7, 0}, {4, 6, 1}, {7, 5, 2}, {6, 4, 3}};
static int dist[8][8]={{0, 1, 2, 1, 1, 2, 3, 2}, {1, 0, 1, 2, 2, 1, 2, 3},
                       {2, 1, 0, 1, 3, 2, 1, 2}, {1, 2, 1, 0, 2, 3, 2, 1},
                       {1, 2, 3, 2, 0, 1, 2, 1}, {2, 1, 2, 3, 1, 0, 1, 2},
                       {3, 2, 1, 2, 2, 1, 0, 1}, {2, 3, 2, 1, 1, 2, 1, 0}};
static double cube[8], level, step;
static int interpol, ny, nz;

void marching_cube(double xstart, double ystart, double zstart, double grid[],
                   double gridstep, int xres, int yres, int zres, double surface,
                   int primitive, void (*begin)(int), void (*end)(void),
			 void (*vert)(double, double, double),
			 void (*norm)(double, double, double), int ip, int debug)
{
  int setcorners[8], notsetcorners[8], d[6], one, two;
  int first, last;
  int c0=0, c1=0, c2=0, c3=0, c3a=0, c5=0, c6=0, c6a=0, c7=0, c7a=0, c8=0;
  int c9=0, c10=0, c11=0, c12=0, c13=0, invert;
  register double x, y, z;
  register int i, j, k, l, m, set, notset;

  level=surface;
  interpol=ip;
  step=gridstep;
  ny=yres;
  nz=zres;

  z=zstart;
  for (k=0; k<zres; k++)
  {
    y=ystart; 
    for (j=0; j<yres; j++)
    {
      x=xstart;
      for (i=0; i<xres; i++)
      {
        set=0;
        notset=0;
        invert=FALSE;
        if (grid[index(i,j,k)]       < level) setcorners[set++]=0;
        else                                  notsetcorners[notset++]=0;
        if (grid[index(i+1,j,k)]     < level) setcorners[set++]=1;
        else                                  notsetcorners[notset++]=1;
        if (grid[index(i+1,j+1,k)]   < level) setcorners[set++]=2;
        else                                  notsetcorners[notset++]=2;
        if (grid[index(i,j+1,k)]     < level) setcorners[set++]=3;
        else                                  notsetcorners[notset++]=3;
        if (grid[index(i,j,k+1)]     < level) setcorners[set++]=4;
        else                                  notsetcorners[notset++]=4;
        if (grid[index(i+1,j,k+1)]   < level) setcorners[set++]=5;
        else                                  notsetcorners[notset++]=5;
        if (grid[index(i+1,j+1,k+1)] < level) setcorners[set++]=6;
        else                                  notsetcorners[notset++]=6;
        if (grid[index(i,j+1,k+1)]   < level) setcorners[set++]=7;
        else                                  notsetcorners[notset++]=7;

        cube[0]=grid[index(i,j,k)];
        cube[1]=grid[index(i+1,j,k)];
        cube[2]=grid[index(i+1,j+1,k)];
        cube[3]=grid[index(i,j+1,k)];
        cube[4]=grid[index(i,j,k+1)];
        cube[5]=grid[index(i+1,j,k+1)];
        cube[6]=grid[index(i+1,j+1,k+1)];
        cube[7]=grid[index(i,j+1,k+1)];

        switch (set)
        {
          case 0:                                                      /* Pattern 0 */
          case 8: c0++;
                  break;

          case 7: setcorners[0]=notsetcorners[0];                      /* Pattern 1 */
          case 1: single_triangle(x, y, z, setcorners[0], grid, i, j, k,
					    primitive, begin, end, vert, norm);
                  c1++;
                  break;

          case 6: setcorners[0]=notsetcorners[0];                   /* Patterns 2-4 */
                  setcorners[1]=notsetcorners[1];
                  invert=TRUE;
          case 2: l=dist[setcorners[0]][setcorners[1]];
                  switch (l)
                  {                                                    /* Pattern 2 */
                    case 1: single_rectangle(x, y, z, setcorners[0], setcorners[1],
							   grid, i, j, k, primitive, begin,
							   end, vert, norm);
                            c2++;
                            break;
                    case 2: if (invert)                     /* Patterns 3, 3a and 4 */
                            {
                              pattern3a(x, y, z, setcorners[0], setcorners[1],
						    grid, i, j, k, primitive, begin, end,
						    vert, norm);
                              c3a++;
                              break;
                            }
                    case 3: single_triangle(x, y, z, setcorners[0], grid, i, j,
							  k, primitive, begin, end, vert, norm);
                            single_triangle(x, y, z, setcorners[1], grid, i, j,
							  k, primitive, begin, end, vert, norm);
                            c3++;
                            break;
                  }
                  break;

          case 5: setcorners[0]=notsetcorners[0];                 /* Patterns 5 - 7 */
                  setcorners[1]=notsetcorners[1];
                  setcorners[2]=notsetcorners[2];
                  invert=TRUE;
          case 3: d[0]=dist[setcorners[0]][setcorners[1]];
                  d[1]=dist[setcorners[0]][setcorners[2]];
                  d[2]=dist[setcorners[1]][setcorners[2]];
                  one=0;
                  two=0;
                  for (l=0; l<3; l++)
                  {
                    switch (d[l])
                    {
                      case 1: one++;
                              break;
                      case 2: two++;
                              break;
                    }
                  }
                  if (one == 2 && two == 1)                            /* Pattern 5 */
                  {
                    three_triangles(x, y, z, setcorners[0], setcorners[1],
                                    setcorners[2], grid, i, j, k, primitive,
						begin, end, vert, norm);
                    c5++;
                  }
                  else if (one == 1 && two == 1)                 /* Pattern 6 or 6a */
                  {
                    if (d[0] == 1)
                    {
                      if (invert)
                      {
                        pattern6a(x, y, z, setcorners[0], setcorners[1],
                                  setcorners[2], grid, i, j, k, primitive,
					    begin, end, vert, norm);
                        c6a++;
                      }
                      else
                      {
                        single_rectangle(x, y, z, setcorners[0], setcorners[1],
						     grid, i, j, k, primitive, begin, end,
						     vert, norm);
                        single_triangle(x, y, z, setcorners[2], grid, i, j, k,
						    primitive, begin, end, vert, norm);
                        c6++;
                      }
                    }
                    else if (d[1] == 1)
                    {
                      if (invert)
                      {
                        pattern6a(x, y, z, setcorners[0], setcorners[2],
                                  setcorners[1], grid, i, j, k, primitive,
					    begin, end, vert, norm);
                        c6a++;
                      }
                      else
                      {
                        single_rectangle(x, y, z, setcorners[0], setcorners[2],
						     grid, i, j, k, primitive, begin, end,
						     vert, norm);
                        single_triangle(x, y, z, setcorners[1], grid, i, j, k,
						    primitive, begin, end, vert, norm);
                        c6++;
                      }
                    }
                    else if (d[2] == 1)
                    {
                      if (invert)
                      {
                        pattern6a(x, y, z, setcorners[1], setcorners[2],
                                  setcorners[0], grid, i, j, k, primitive,
					    begin, end, vert, norm);
                        c6a++;
                      }
                      else
                      {
                        single_rectangle(x, y, z, setcorners[1], setcorners[2],
						     grid, i, j, k, primitive, begin, end,
						     vert, norm);
                        single_triangle(x, y, z, setcorners[0], grid, i, j, k,
						    primitive, begin, end, vert, norm);
                        c6++;
                      }
                    }
                  }
                  else if (two == 3)                             /* Pattern 7 or 7a */
                  {
                    if (invert)
                    {
                      pattern3a(x, y, z, setcorners[0], setcorners[1], grid, i,
					  j, k, primitive, begin, end, vert, norm);
                      c7a++;
                    }
                    else
                    {
                      single_triangle(x, y, z, setcorners[0], grid, i, j, k,
						  primitive, begin, end, vert, norm);
                      single_triangle(x, y, z, setcorners[1], grid, i, j, k,
						  primitive, begin, end, vert, norm);
                      c7++;
                    }
                    single_triangle(x, y, z, setcorners[2], grid, i, j, k,
						primitive, begin, end, vert, norm);
                  }
                  break;

          case 4: d[0]=dist[setcorners[0]][setcorners[1]];       /* Patterns 8 - 14 */
                  d[1]=dist[setcorners[0]][setcorners[2]];
                  d[2]=dist[setcorners[0]][setcorners[3]];
                  d[3]=dist[setcorners[1]][setcorners[2]];
                  d[4]=dist[setcorners[1]][setcorners[3]];
                  d[5]=dist[setcorners[2]][setcorners[3]];
                  one=0;
                  two=0;
                  for (l=0; l<6; l++)
                  {
                    switch (d[l])
                    {
                      case 1: one++;
                              break;
                      case 2: two++;
                              break;
                    }
                  }
                  if (one == 4 && two == 2)                            /* Pattern 8 */
                  {
                    notsetcorners[0]=setcorners[0];
                    m=1;
                    for (l=0; l<3; l++)
                    {
                      if (d[l] == 1)
                      {
                        notsetcorners[m]=setcorners[l+1];
                        setcorners[l+1]=(-1);
                        m+=2;
                      }
                    }
                    for (l=1; l<4; l++)
                    {
                      if (setcorners[l] != -1)
                      {
                        notsetcorners[2]=setcorners[l];
                        break;
                      }
                    }
                    pattern8(x, y, z, notsetcorners[0], notsetcorners[1],
                             notsetcorners[2], notsetcorners[3], grid, i, j, k,
				     primitive, begin, end, vert, norm);
                    c8++;
                  }
                  else if (one == 3 && two == 3)                        /* Pattern 9 */
                  {
                    if (d[0]+d[1]+d[2] == 3) pattern9(x, y, z, setcorners[0],
                                                      setcorners[1], setcorners[2],
                                                      setcorners[3], grid, i, j,
									k, primitive, begin, end,
									vert, norm);
                    else if (d[0]+d[3]+d[4] == 3) pattern9(x, y, z, setcorners[1],
                                                           setcorners[0], setcorners[2],
                                                           setcorners[3], grid,
									     i, j, k, primitive,
									     begin, end, vert, norm);
                    else if (d[1]+d[3]+d[5] == 3) pattern9(x, y, z, setcorners[2],
                                                           setcorners[0], setcorners[1],
                                                           setcorners[3], grid,
									     i, j, k, primitive,
									     begin, end, vert, norm);
                    else pattern9(x, y, z, setcorners[3], setcorners[0], setcorners[1],
                                  setcorners[2], grid, i, j, k, primitive,
					    begin, end, vert, norm);
                    c9++;
                  }
                  else if (one == 2 && two == 2)                      /* Pattern 10 */
                  {
                    if (d[0] == 1) single_rectangle(x, y, z, setcorners[0], setcorners[1],
								    grid, i, j, k, primitive, begin, end, vert, norm);
                    if (d[1] == 1) single_rectangle(x, y, z, setcorners[0], setcorners[2],
								    grid, i, j, k, primitive, begin, end, vert, norm);
                    if (d[2] == 1) single_rectangle(x, y, z, setcorners[0], setcorners[3],
								    grid, i, j, k, primitive, begin, end, vert, norm);
                    if (d[3] == 1) single_rectangle(x, y, z, setcorners[1], setcorners[2],
								    grid, i, j, k, primitive, begin, end, vert, norm);
                    if (d[4] == 1) single_rectangle(x, y, z, setcorners[1], setcorners[3],
								    grid, i, j, k, primitive, begin, end, vert, norm);
                    if (d[5] == 1) single_rectangle(x, y, z, setcorners[2], setcorners[3],
								    grid, i, j, k, primitive, begin, end, vert, norm);
                    c10++;
                  }
                  else if (one == 3 && two == 2)              /* Patterns 11 and 14 */
                  {
                    first=(-1);
                    last=(-1);
                    if (d[0]+d[1]+d[2] == 6) first=setcorners[0];
                    if (d[0]+d[3]+d[4] == 6)
                    {
                      if (first != -1) last=setcorners[1];
                      else             first=setcorners[1];
                    }
                    if (d[1]+d[3]+d[5] == 6)
                    {
                      if (first != -1) last=setcorners[2];
                      else             first=setcorners[2];
                    }
                    if (d[2]+d[4]+d[5] == 6)
                    {
                      if (first != -1) last=setcorners[3];
                    }
                    for (l=0; l<4; l++)
                      if (dist[first][setcorners[l]] == 1) break;
                    for (m=0; m<4; m++)
                      if (dist[last][setcorners[m]] == 1) break;

                    pattern11(x, y, z, first, setcorners[l],
                              setcorners[m], last, grid, i, j, k, primitive,
					begin, end, vert, norm);
                    c11++;
                  }
                  else if (one == 2 && two == 3)                      /* Pattern 12 */
                  {
                    if (d[0]+d[1]+d[2] == 7)
                    {
                      single_triangle(x, y, z, setcorners[0], grid, i, j, k,
						  primitive, begin, end, vert, norm);
                      three_triangles(x, y, z, setcorners[1], setcorners[2],
                                      setcorners[3], grid, i, j, k, primitive,
						  begin, end, vert, norm);
                    }
                    if (d[0]+d[3]+d[4] == 7)
                    {
                      single_triangle(x, y, z, setcorners[1], grid, i, j, k,
						  primitive, begin, end, vert, norm);
                      three_triangles(x, y, z, setcorners[0], setcorners[2],
                                      setcorners[3], grid, i, j, k, primitive,
						  begin, end, vert, norm);
                    }
                    if (d[1]+d[3]+d[5] == 7)
                    {
                      single_triangle(x, y, z, setcorners[2], grid, i, j, k,
						  primitive, begin, end, vert, norm);
                      three_triangles(x, y, z, setcorners[0], setcorners[1],
                                      setcorners[3], grid, i, j, k, primitive,
						  begin, end, vert, norm);
                    }
                    if (d[2]+d[4]+d[5] == 7)
                    {
                      single_triangle(x, y, z, setcorners[3], grid, i, j, k,
						  primitive, begin, end, vert, norm);
                      three_triangles(x, y, z, setcorners[0], setcorners[1],
                                      setcorners[2], grid, i, j, k, primitive,
						  begin, end, vert, norm);
                    }
                    c12++;
                  }
                  else if (two == 6)                                 /* Pattern 13 */
                  {
                    single_triangle(x, y, z, setcorners[0], grid, i, j, k,
						primitive, begin, end, vert, norm);
                    single_triangle(x, y, z, setcorners[1], grid, i, j, k,
						primitive, begin, end, vert, norm);
                    single_triangle(x, y, z, setcorners[2], grid, i, j, k,
						primitive, begin, end, vert, norm);
                    single_triangle(x, y, z, setcorners[3], grid, i, j, k,
						primitive, begin, end, vert, norm);
                  }
                  c13++;
                  break;
        }
        x+=step;
      }
      y+=step;
    }
    z+=step;
  }
  if (debug)
    printf("Number of patterns: %d x 0, %d x 1, %d x 2, %d x 3, %d x 3a, %d x 5, %d x 6, %d x 6a, %d x 7, %d x 7a, %d x 8, %d x 9, %d x 10, %d x 11, %d x 12, %d x 13\n",
           c0, c1, c2, c3, c3a, c5, c6, c6a, c7, c7a, c8, c9, c10, c11, c12, c13);
}

void single_triangle(double x, double y, double z, int corner, double grid[],
			   int ig, int jg, int kg, int primitive, void (*begin)(int),
			   void (*end)(), void (*vert)(double, double, double),
			   void (*norm)(double, double, double))
{
  double v[3], n[3];

  (*begin)(primitive);
  set_vertex(x, y, z, corner, edges[corner][0], grid, ig, jg, kg, v, n, primitive, vert, norm);
  set_vertex(x, y, z, corner, edges[corner][1], grid, ig, jg, kg, v, n, primitive, vert, norm);
  set_vertex(x, y, z, corner, edges[corner][2], grid, ig, jg, kg, v, n, primitive, vert, norm);
  (*end)();
}

void single_rectangle(double x, double y, double z, int corner1, int corner2,
			    double grid[], int ig, int jg, int kg, int primitive,
			    void (*begin)(int), void (*end)(), void (*vert)(double, double, double),
			    void (*norm)(double, double, double))
{
  double v[3], v1[3], n[3];
  int save[2];
  register int i, j, k=0;

  (*begin)(primitive);
  j=0;
  for (i=0; i<3; i++)
  {
    if (edges[corner1][i] != corner2)
      save[j++]=edges[corner1][i];
  }
  set_vertex(x, y, z, corner1, save[1], grid, ig, jg, kg, v, n, primitive, vert, norm);
  set_vertex(x, y, z, corner1, save[0], grid, ig, jg, kg, v1, n, primitive, vert, norm);
  j=save[1];
  for (i=0; i<3; i++)
  {
    if (edges[corner2][i] != corner1)
      save[k++]=edges[corner2][i];
  }
  if (dist[save[0]][j] == 1)
  {
    set_vertex(x, y, z, corner2, save[0], grid, ig, jg, kg, v, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP)
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v[0], v[1], v[2]);
    }
    set_vertex(x, y, z, corner2, save[1], grid, ig, jg, kg, v, n, primitive, vert, norm);
  }
  else
  {
    set_vertex(x, y, z, corner2, save[1], grid, ig, jg, kg, v, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP)
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v[0], v[1], v[2]);
    }
    set_vertex(x, y, z, corner2, save[0], grid, ig, jg, kg, v, n, primitive, vert, norm);
  } 
  (*end)();
}

void three_triangles(double x, double y, double z, int corner1, int corner2,
                     int corner3, double grid[], int ig, int jg, int kg,
			   int primitive, void (*begin)(int), void (*end)(),
			   void (*vert)(double, double, double),
			   void (*norm)(double, double, double))
{
  double v[3], v1[3], v2[3], n[3];
  int d1, d2;
  register int i, j;

  d1=dist[corner1][corner2]+dist[corner1][corner3];
  d2=dist[corner3][corner1]+dist[corner3][corner2];
  if (d1 == 2)
  {
    i=corner1;
    corner1=corner2;
    corner2=i;
  }
  if (d2 == 2)
  {
    i=corner3;
    corner3=corner2;
    corner2=i;
  }

  (*begin)(primitive);
  for (i=0; i<3; i++)
    if (edges[corner2][i] != corner1 && edges[corner2][i] != corner3) break;
  i=edges[corner2][i];
  set_vertex(x, y, z, corner2, i, grid, ig, jg, kg, v, n, primitive, vert, norm);

  for (j=0; j<3; j++)
    if (edges[corner1][j] != corner2 && dist[edges[corner1][j]][i] == 1) break;
  set_vertex(x, y, z, corner1, edges[corner1][j], grid, ig, jg, kg, v1, n, primitive, vert, norm);

  for (j=0; j<3; j++)
    if (edges[corner3][j] != corner2 && dist[edges[corner3][j]][i] == 1) break;
  set_vertex(x, y, z, corner3, edges[corner3][j], grid, ig, jg, kg, v2, n, primitive, vert, norm);

  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
  for (j=0; j<3; j++)
    if (edges[corner1][j] != corner2 && dist[edges[corner1][j]][i] == 3) break;
  set_vertex(x, y, z, corner1, edges[corner1][j], grid, ig, jg, kg, v1, n, primitive, vert, norm);
 
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
  for (j=0; j<3; j++)
    if (edges[corner3][j] != corner2 && dist[edges[corner3][j]][i] == 3) break;
  set_vertex(x, y, z, corner3, edges[corner3][j], grid, ig, jg, kg, v, n, primitive, vert, norm);
  (*end)();
}

void pattern8(double x, double y, double z, int corner1, int corner2, int corner3,
         int corner4, double grid[], int ig, int jg, int kg, int primitive,
	   void (*begin)(int), void (*end)(), void (*vert)(double, double, double),
	   void (*norm)(double, double, double))
{
  double v[3], v1[3], v2[3], n[3];
  register int i, j;
  
  (*begin)(primitive);
  for (i=0; i<3; i++)
    if (edges[corner1][i] != corner2 && edges[corner1][i] != corner3 && edges[corner1][i] != corner4) break;
  j=edges[corner1][i];

  for (i=0; i<3; i++)
    if (edges[corner2][i] != corner1 && edges[corner2][i] != corner3 && edges[corner2][i] != corner4) break;
  set_vertex(x, y, z, corner2, edges[corner2][i], grid, ig, jg, kg, v, n, primitive, vert, norm);
  set_vertex(x, y, z, corner1, j, grid, ig, jg, kg, v1, n, primitive, vert, norm);

  for (i=0; i<3; i++)
    if (edges[corner3][i] != corner1 && edges[corner3][i] != corner2 && edges[corner3][i] != corner4) break;
  set_vertex(x, y, z, corner3, edges[corner3][i], grid, ig, jg, kg, v2, n, primitive, vert, norm);

  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();  
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }

  for (i=0; i<3; i++)
    if (edges[corner4][i] != corner1 && edges[corner4][i] != corner2 && edges[corner4][i] != corner3) break;
  set_vertex(x, y, z, corner4, edges[corner4][i], grid, ig, jg, kg, v, n, primitive, vert, norm);

  (*end)();
}

void pattern9(double x, double y, double z, int corner1, int corner2, int corner3,
         int corner4, double grid[], int ig, int jg, int kg, int primitive,
	   void (*begin)(int), void (*end)(), void (*vert)(double, double, double),
	   void (*norm)(double, double, double))
{
  int p[6];
  double v[3], v1[3], v2[3], n[3];
  register int i, j, k=0;

  for (i=0; i<3; i++)
    if (edges[corner2][i] != corner1) p[k++]=edges[corner2][i];
  if (p[0] > p[1])
  {
    i=p[0];
    p[0]=p[1];
    p[1]=i;
  }
  for (j=0; j<3; j++)
    if (edges[corner3][j] != corner1) p[k++]=edges[corner3][j];
  if (p[2] > p[3])
  {
    i=p[2];
    p[2]=p[3];
    p[3]=i;
  }
  for (j=0; j<3; j++)
    if (edges[corner4][j] != corner1) p[k++]=edges[corner4][j];
  if (p[4] > p[5])
  {
    i=p[4];
    p[4]=p[5];
    p[5]=i;
  }

  (*begin)(primitive);
  set_vertex(x, y, z, corner2, p[1], grid, ig, jg, kg, v, n, primitive, vert, norm);

  for (j=2; j<6; j++)
    if (p[j] == p[1]) break;

  if (j<4)
  {
    set_vertex(x, y, z, corner3, p[1], grid, ig, jg, kg, v2, n, primitive, vert, norm);
    set_vertex(x, y, z, corner2, p[0], grid, ig, jg, kg, v1, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP);
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v2[0], v2[1], v2[2]);
    }
    if (j == 2) j=3;
    set_vertex(x, y, z, corner3, p[j], grid, ig, jg, kg, v2, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP);
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v2[0], v2[1], v2[2]);
    }
    set_vertex(x, y, z, corner4, p[4], grid, ig, jg, kg, v1, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP)
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v2[0], v2[1], v2[2]);
    }
    set_vertex(x, y, z, corner4, p[5], grid, ig, jg, kg, v, n, primitive, vert, norm);
  }
  else
  {
    set_vertex(x, y, z, corner4, p[1], grid, ig, jg, kg, v2, n, primitive, vert, norm);
    set_vertex(x, y, z, corner2, p[0], grid, ig, jg, kg, v1, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP)
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v2[0], v2[1], v2[2]);
    }
    if (j == 4) j=5;
    set_vertex(x, y, z, corner4, p[j], grid, ig, jg, kg, v2, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP)
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v2[0], v2[1], v2[2]);
    }
    set_vertex(x, y, z, corner3, p[2], grid, ig, jg, kg, v1, n, primitive, vert, norm);
    if (primitive != GL_TRIANGLE_STRIP)
    {
      (*end)();
      (*begin)(primitive);
      (*vert)(v1[0], v1[1], v1[2]);
      (*vert)(v2[0], v2[1], v2[2]);
    }
    set_vertex(x, y, z, corner3, p[3], grid, ig, jg, kg, v, n, primitive, vert, norm);
  }
  (*end)();
}

void pattern11(double x, double y, double z, int corner1, int corner2, int corner3,
          int corner4, double grid[], int ig, int jg, int kg, int primitive,
	    void (*begin)(int), void (*end)(), void (*vert)(double, double, double),
	    void (*norm)(double, double, double))
{
  double v[3], v1[3], v2[3], v3[3], n[3], n1[3], n2[3];
  register int i, j;

  (*begin)(primitive);
  for (i=0; i<4; i++)
    if (edges[corner2][i] != corner1 && edges[corner2][i] != corner3) break;

  i=edges[corner2][i];
  
  for (j=0; j<4; j++)
    if (edges[corner1][j] != corner2 && dist[i][edges[corner1][j]] == 1) break;

  j=edges[corner1][j];
  set_vertex(x, y, z, corner1, j, grid, ig, jg, kg, v, n, primitive, vert, norm);
  set_vertex(x, y, z, corner2, i, grid, ig, jg, kg, v1, n1, primitive, vert, norm);

  for (i=0; i<4; i++)
    if (edges[corner1][i] != corner2 && edges[corner1][i] != j) break;

  i=edges[corner1][i];
  set_vertex(x, y, z, corner1, i, grid, ig, jg, kg, v3, n, primitive, vert, norm);
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v3[0], v3[1], v3[2]);
  }
  for (j=0; j<4; j++)
    if (edges[corner4][j] != corner3 && dist[edges[corner4][j]][i] == 1) break;
  
  j=edges[corner4][j];
  set_vertex(x, y, z, corner4, j, grid, ig, jg, kg, v2, n2, primitive, vert, norm);
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v2[0], v2[1], v2[2]);
    (*vert)(v3[0], v3[1], v3[2]);
  }
  set_vertex(x, y, z, corner3, i, grid, ig, jg, kg, v, n, primitive, vert, norm);
  (*end)();

  (*begin)(primitive);
  (*norm)(n1[0], n1[1], n1[2]);
  (*vert)(v1[0], v1[1], v1[2]);
  (*norm)(n2[0], n2[1], n2[2]);
  (*vert)(v2[0], v2[1], v2[2]);
  for (i=0; i<4; i++)
    if (edges[corner4][i] != corner3 && edges[corner4][i] != j) break;

  set_vertex(x, y, z, corner4, edges[corner4][i], grid, ig, jg, kg, v, n, primitive, vert, norm);
  (*end)();
}

void pattern3a(double x, double y, double z, int corner1, int corner2,
		   double grid[], int ig, int jg, int kg, int primitive,
		   void (*begin)(int), void (*end)(), void (*vert)(double, double, double),
		   void (*norm)(double, double, double))
{
  double v[3], v1[3], v2[3], n[3];
  register int i, j, k;

  (*begin)(primitive);
  for (i=0; i<3; i++)
    if (dist[edges[corner1][i]][corner2] == 3) break;
  i=edges[corner1][i];
  for (j=0; j<3; j++)
    if (edges[corner1][j] != i) break;
  j=edges[corner1][j];
  set_vertex(x, y, z, corner1, j, grid, ig, jg, kg, v, n, primitive, vert, norm);
  set_vertex(x, y, z, corner2, j, grid, ig, jg, kg, v2, n, primitive, vert, norm);
  set_vertex(x, y, z, corner1, i, grid, ig, jg, kg, v1, n, primitive, vert, norm);
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
  for (k=0; k<3; k++)
    if (dist[edges[corner2][k]][corner1] == 3) break;
  k=edges[corner2][k];
  set_vertex(x, y, z, corner2, k, grid, ig, jg, kg, v2, n, primitive, vert, norm);
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
  for (k=0; k<3; k++)
    if (edges[corner1][k] != i && edges[corner1][k] != j) break;
  k=edges[corner1][k];
  set_vertex(x, y, z, corner1, k, grid, ig, jg, kg, v1, n, primitive, vert, norm);
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
  set_vertex(x, y, z, corner2, k, grid, ig, jg, kg, v, n, primitive, vert, norm);
  (*end)();
}

void pattern6a(double x, double y, double z, int corner1, int corner2,
		   int corner3, double grid[], int ig, int jg, int kg,
		   int primitive, void (*begin)(int), void (*end)(),
		   void (*vert)(double, double, double),
		   void (*norm)(double, double, double))
{
  double v[3], v1[3], v2[3], n[3];
  register int i, j, k, l;

  if (dist[corner1][corner3] == 2)
  {
    i=corner2;
    corner2=corner1;
    corner1=i;
  }

  (*begin)(primitive);

  for (i=0; i<3; i++)
    if (edges[corner2][i] != corner1) break;
  i=edges[corner2][i];
  set_vertex(x, y, z, corner2, i, grid, ig, jg, kg, v, n, primitive, vert, norm);

  for (j=0; j<3; j++)
    if (edges[corner1][j] != corner2 && dist[edges[corner1][j]][i] == 1) break;
  j=edges[corner1][j];
  set_vertex(x, y, z, corner1, j, grid, ig, jg, kg, v1, n, primitive, vert, norm);
  set_vertex(x, y, z, corner3, i, grid, ig, jg, kg, v2, n, primitive, vert, norm);

  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }

  for (k=0; k<3; k++)
    if (edges[corner3][k] != i && dist[edges[corner3][k]][corner2] == 3) break;
  k=edges[corner3][k];
  set_vertex(x, y, z, corner3, k, grid, ig, jg, kg, v2, n, primitive, vert, norm);
  
  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
  for (l=0; l<3; l++)
    if (edges[corner1][l] != corner2 && edges[corner1][l] != j) break;
  l=edges[corner1][l];
  set_vertex(x, y, z, corner1, l, grid, ig, jg, kg, v1, n, primitive, vert, norm);

  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }

  for (l=0; l<3; l++)
    if (edges[corner3][l] != i && edges[corner3][l] != k) break;
  l=edges[corner3][l];
  set_vertex(x, y, z, corner3, l, grid, ig, jg, kg, v2, n, primitive, vert, norm);

  if (primitive != GL_TRIANGLE_STRIP)
  {
    (*end)();
    (*begin)(primitive);
    (*vert)(v1[0], v1[1], v1[2]);
    (*vert)(v2[0], v2[1], v2[2]);
  }
 
  for (l=0; l<3; l++)
    if (edges[corner2][l] != corner1 && edges[corner2][l] != i) break;
  l=edges[corner2][l];
  set_vertex(x, y, z, corner2, l, grid, ig, jg, kg, v, n, primitive, vert, norm);

 (*end)();
}

void set_vertex(double x, double y, double z, int c1, int c2, double grid[],
		    int ig, int jg, int kg, double *v, double *n, int primitive,
		    void (*vert)(double, double, double), void (*norm)(double, double, double))
{
  static double vertex[8][8][3]={{{0.0, 0.0, 0.0}, {0.5, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 0.5, 0.0},
                                  {0.0, 0.0, 0.5}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0}},
                                 {{-0.5, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {1.0, 0.5, 0.0}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {1.0, 0.0, 0.5},
                                  {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0}},
                                 {{0.0, 0.0, 0.0}, {1.0, -0.5, 0.0},
                                  {0.0, 0.0, 0.0}, {-0.5, 1.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {1.0, 1.0, 0.5}, {0.0, 0.0, 0.0}},
                                 {{0.0, -0.5, 0.0}, {0.0, 0.0, 0.0},
                                  {0.5, 1.0, 0.0}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 1.0, 0.5}},
                                 {{0.0, 0.0, -0.5}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.5, 0.0, 1.0},
                                  {0.0, 0.0, 0.0}, {0.0, 0.5, 1.0}},
                                 {{0.0, 0.0, 0.0}, {1.0, 0.0, -0.5},
                                  {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {-0.5, 0.0, 1.0}, {0.0, 0.0, 0.0},
                                  {1.0, 0.5, 1.0}, {0.0, 0.0, 0.0}},
                                 {{0.0, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {1.0, 1.0, -0.5}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {1.0, -0.5, 1.0},
                                  {0.0, 0.0, 0.0}, {-0.5, 1.0, 1.0}},
                                 {{0.0, 0.0, 0.0}, {0.0, 0.0, 0.0},
                                  {0.0, 0.0, 0.0}, {0.0, 1.0, -0.5},
                                  {0.0, -0.5, 1.0}, {0.0, 0.0, 0.0},
                                  {0.5, 1.0, 1.0}, {0.0, 0.0, 0.0}},
                                  };
  double x1=cube[c1], x2=cube[c2], l=level;
  register int i ;
 
  switch (c1)
  {
    case 1: ig++;
            break;
    case 2: ig++;
            jg++;
            break;
    case 3: jg++;
            break;
    case 4: kg++;
            break;
    case 5: ig++;
            kg++;
            break;
    case 6: ig++;
            jg++;
            kg++;
            break;
    case 7: jg++;
            kg++;
            break;
  }
  switch (interpol)
  {
    case IP_NONE:   v[0]=x+fabs(vertex[c1][c2][0])*step;
                    v[1]=y+fabs(vertex[c1][c2][1])*step;
                    v[2]=z+fabs(vertex[c1][c2][2])*step;
                    break;
    case IP_LOG:    if (x1 > 0.0) x1=log(x1);
                    if (x2 > 0.0) x2=log(x2);
                    if (l  > 0.0) l=log(l);
    case IP_LINEAR: x1-=l;
                    x2-=l;
                    for (i=0; i<3; i++)
                    {
                      v[i]=vertex[c1][c2][i];
                      if (v[i] == 0.5e0) v[i]=fabs(x1)/(fabs(x1)+fabs(x2));
                      if (v[i] == -0.5e0) v[i]=1.0-fabs(x1)/(fabs(x1)+fabs(x2));
                    }
                    v[0]=x+v[0]*step;
                    v[1]=y+v[1]*step;
                    v[2]=z+v[2]*step;
                    break;
  }
/*if (primitive == GL_TRIANGLE_STRIP)
  { */
    n[0]=v[0]+(grid[index(ig-1, jg, kg)]-grid[index(ig+1, jg, kg)])/(2.e0*step);
    n[1]=v[1]+(grid[index(ig, jg-1, kg)]-grid[index(ig, jg+1, kg)])/(2.e0*step);
    n[2]=v[2]+(grid[index(ig, jg, kg-1)]-grid[index(ig, jg, kg+1)])/(2.e0*step);
    l=sqrt(n[0]*n[0]+n[1]*n[1]+n[2]*n[2]);
    n[0]/=l;
    n[1]/=l;
    n[2]/=l;
    (*norm)(n[0], n[1], n[2]);
/*}*/
  (*vert)(v[0], v[1], v[2]);
}
