/*******************************************************************************
*                                                                              *
*                                   Viewmol                                    *
*                                                                              *
*                            B O N D O R D E R . C                             *
*                                                                              *
*                 Copyright (c) Joerg-R. Hill, December 1997                   *
*                                                                              *
********************************************************************************
*
* $Id: bondorder.c,v 1.1 1998/01/26 00:34:45 jrh Exp jrh $
* $Log: bondorder.c,v $
* Revision 1.1  1998/01/26 00:34:45  jrh
* Initial revision
*
*/
#include<math.h>
#include<stdlib.h>
#include<string.h>
#include "viewmol.h"

extern struct ATOM *atoms;

extern void *getmem(size_t, size_t);

void calculateShift(struct BOND *, int *, int, double, int, int);
void findConjugation(struct BOND *, int *, int, int, double);
void vectorProduct(double *, double *, double *);
double scalarProduct(double *, double *);

void bondOrder(struct BOND *bond, int *connectivity, int maxBondsPerAtom, double offset)
{
  int i, j, ni, nj, bo=1;

/* Assign bond orders by counting electrons. If the number of electrons
   for an atom is set to -1 assign only single bonds. */

  i=bond->first;
  if (atoms[i].nelectrons == (-1))
  {
    bond->order=1;
    return;
  }
  j=bond->second;
  if (atoms[j].nelectrons == (-1))
  {
    bond->order=1;
    return;
  }

/* Compare number of electrons with number of atoms atom is bound to
   and deduce bond order */

  do
  {
    if (atoms[i].nelectrons > 4) ni=8-atoms[i].nelectrons;
    else                         ni=atoms[i].nelectrons;
    if (atoms[j].nelectrons > 4) nj=8-atoms[j].nelectrons;
    else                         nj=atoms[j].nelectrons;
    ni-=atoms[i].nbonds;
    nj-=atoms[j].nbonds;
    if (ni < 0) ni=atoms[i].nelectrons-atoms[i].nbonds;
    if (nj < 0) nj=atoms[j].nelectrons-atoms[j].nbonds;

    if (ni > 0 && nj > 0)
    {
      atoms[i].nbonds++;
      atoms[j].nbonds++;
      bo++;
    }
    else
      break;
  } while (bo <= 4);
  bond->order=bo;

/* Bond order has been found, calculate shift vector to draw multiple bonds. */

  if (bo > 1) calculateShift(bond, connectivity, maxBondsPerAtom, offset, i, j);
}

void calculateShift(struct BOND *bond, int *connectivity, int maxBondsPerAtom,
                    double offset, int i, int j)
{
  double v1[3], v2[3], v3[3];
  int ni, nj, bo;
  register double r1, r2;
  register int k, l;

  /* Rules for multiple bonds: Shall be in the same plane as other bond framework,
                               but do not use hydrogens to determine this plane. */
  bo=abs(bond->order);
  ni=(atoms[i].nbonds >= atoms[j].nbonds ? i : j);
  nj=(ni == i ? j : i);
  l=0;
  do
  {
    k=connectivity[maxBondsPerAtom*ni+l];
    if (k != nj && strcmp(atoms[k].element->symbol, "H")) break;
    l++;
  } while (l < maxBondsPerAtom);
  v1[0]=atoms[k].x-atoms[ni].x;
  v1[1]=atoms[k].y-atoms[ni].y;
  v1[2]=atoms[k].z-atoms[ni].z;
  r1=scalarProduct(v1, v1);
/*r1=1.0/sqrt(scalarProduct(v1, v1));
  v1[0]*=r1;
  v1[1]*=r1;
  v1[2]*=r1; */
  v2[0]=atoms[nj].x-atoms[ni].x;
  v2[1]=atoms[nj].y-atoms[ni].y;
  v2[2]=atoms[nj].z-atoms[ni].z;
  r2=scalarProduct(v2, v2);
/*r2=1.0/sqrt(scalarProduct(v2, v2));
  v2[0]*=r2;
  v2[1]*=r2;
  v2[2]*=r2;*/
  if (r1 != 0.0 && r2 != 0.0)
    r1=fabs(scalarProduct(v1, v2)/(sqrt(r1*r2))+1.0);
  else
    r1=0.0;
  if (r1 > 1.0e-6)
  {
    vectorProduct(v1, v2, v3);
    vectorProduct(v2, v3, v1);
    r1=offset/sqrt(scalarProduct(v1, v1));
    if (bo == 2) r1*=0.5;
    bond->x=v1[0]*r1;
    bond->y=v1[1]*r1;
    bond->z=v1[2]*r1;
  }
  else
  {
    v1[0]=1.0;
    v1[1]=0.0;
    v1[2]=0.0;
    r1=scalarProduct(v1, v2)/sqrt(r2);
    if (bo == 2) r2=0.5;
    else         r2=1.0;
    bond->x=r2*offset*sqrt(1.0-r1*r1);
    bond->y=r2*offset*r1;
    bond->z=0.0;
  }
/*v3[0]=0.5*(v1[0]-v2[0]);
  v3[1]=0.5*(v1[1]-v2[1]);
  v3[2]=0.5*(v1[2]-v2[2]);
  r1=offset/sqrt(scalarProduct(v3, v3));
  if (bo == 2) r1*=0.5;
  bond->x=v3[0]*r1;
  bond->y=v3[1]*r1;
  bond->z=v3[2]*r1; */
}

void findConjugation(struct BOND *bonds, int *connectivity, int maxBondsPerAtom,
                     int nb, double offset)
{
  int atom1, atom2, conjugated;
  register int i, j, k, l;

  for (i=0; i<nb; i++)
  {
    if (abs(bonds[i].order) == 2)
    {
      atom1=bonds[i].first;
      for (j=0; j<nb; j++)
      {
        if ((bonds[j].first == atom1 || bonds[j].second == atom1) && bonds[j].order == 1)
        {
          atom2=bonds[j].first == atom1 ? bonds[j].second : bonds[j].first;
          for (k=0; k<nb; k++)
          {
            if ((bonds[k].first == atom2 || bonds[k].second == atom2) && abs(bonds[k].order) == 2)
            {
              conjugated=TRUE;
              for (l=0; l<nb; l++)
              {
                if (l == i || l == j || l == k) continue;
                if (bonds[l].first != atom1 && bonds[l].second != atom1) continue;
                if (bonds[l].order != 1 && bonds[l].order != (-2))
                {
                  conjugated=FALSE;
                  break;
                }
              }
              if (conjugated)
              {
                bonds[i].order=(-2);
                bonds[j].order=(-2);
                bonds[k].order=(-2);
                /* calculate shift vector for bond j */
                calculateShift(&bonds[j], connectivity, maxBondsPerAtom, offset, atom1, atom2);
                break;
              }
            }
          }
        }
      }
    }
  }
}

void vectorProduct(double *v1, double *v2, double *v3)
{
  v3[0]=v1[1]*v2[2]-v1[2]*v2[1];
  v3[1]=v1[2]*v2[0]-v1[0]*v2[2];
  v3[2]=v1[0]*v2[1]-v1[1]*v2[0];
}

double scalarProduct(double *v1, double *v2)
{
  return(v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2]);
}
