// MM1ENG1.H : experimental force field and computation engine.

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef MM1ENG1_H
#define MM1ENG1_H

struct mm1_exp1_bt1;		// bond stretching
struct mm1_exp1_bt2;		// angle bending
struct mm1_exp1_bt3;		// torsion

struct mm1_exp1_nbt1;		// nonbonded

class mm1_eng_exp1a;		// bt
class mm1_eng_exp1b1;		// nbt, non-periodic
class mm1_eng_exp1b2;		// nbt, exact periodic
class mm1_eng_exp1b3;		// nbt, minimum image model

class mm1_eng_exp1;
class mm1_eng_exp1_tst;
class mm1_eng_exp1_mim;
//class mm1_eng_exp1_mim_dc?!?!?!	double cutoffs???

// complex methods like the ewald method or multipole methods would be nice, but it is more
// important to make these ordinary methods run (and also perform well) parallel in clusters...

// mm1_eng_exp1_ti ??? THERMODYNAMICS !?!?!? c/d???
// mm1_eng_exp1_ti ??? THERMODYNAMICS !?!?!? c/d???
// mm1_eng_exp1_ti ??? THERMODYNAMICS !?!?!? c/d???

/*################################################################################################*/

class exp1_tables;		// mm1tab1.h

#include "mm1eng.h"

#include <vector>
using namespace std;

/*################################################################################################*/

struct mm1_exp1_bt1
{
	i32s atmi[2];
	
	f64 opt;
	f64 fc;
	
	i32s get_atmi(i32s index, bool dir)
	{
		return atmi[dir ? index : !index];
	}
};

struct mm1_exp1_bt2
{
	i32s atmi[3];
	
	i32s index1[2];
	bool dir1[2];
	
	f64 opt;
	f64 fc;
	
	i32s get_index(i32s index, bool dir)
	{
		return index1[dir ? index : !index];
	}
	
	bool get_dir(i32s index, bool dir)
	{
		return dir1[dir ? index : !index];
	}
};

struct mm1_exp1_bt3
{
	i32s atmi[4];
	
	i32s index2[2];
	i32s index1[4];
	bool dir1[4];
	
	f64 k; f64 s;
};

// here we must use floats to reduce memory consumption...
// also seems to be slightly faster (at least in intel-machines)

struct mm1_exp1_nbt1
{
	i32s atmi[2];
	
	float k1; float k2;
	float qq;
};

/*################################################################################################*/

/**	"##UNSTABLE", bonded part.

	Parameters for this experimental force field come mainly from the Tripos 5.2 
	parameter set. Some extra patameters (like the ones for 4- and 5-membered rings) 
	come from MMFF94 parameters and are only approximate. The system responsible for 
	atomic charges also come from a separate source...
	
	Clark M, Cramer III RD, Van Obdenbosch N : "##Validation of the General-Purpose Tripos 
	5.2 Force Field", J. Comp. Chem. 10, 982-1012, (1989)
	
	Halgren TA : "##Merck Molecular Force Field. I. Basis, Form, Scope, Parameterization, and 
	Performance of MMFF94", J. Comp. Chem. 17, 490-, (1996) [there are many parts in this paper]
	
	Tuzun RE, Noid DW, Sumpter BG : "##Efficient computation of potential energy first and second 
	derivatives for molecular dynamics, normal coordinate analysis, and molecular mechanics 
	calculations", Macromol. Theory Simul. 5, 771-778, (1996)
	
	Bekker H, Berendsen HJC, van Gunsteren WF : "##Force and virial of Torsional-Angle-Dependent 
	Potentials", J. Comp. Chem. 16, 527-533, (1995)
	
	THE FOLLOWING ARE JUST FUTURE PLANS:
	
	In the long run it would be best to build up a database of QM (and experimental, where 
	available) results, which could then be used to derive (and check!!!) the forcefield 
	parameters. This force field parameter stuff will probably always be inadequate/obsolete 
	somewhat, so when parameters are missing one could gather some extra information, add it 
	to the database, check how it matches with the existing old data, and derive the new parameters...
	
	QM results can be done using MPQC. large amount of molecules/conformations are needed. 
	How to generate the conformations??? Energies must be calculated. What about 1st derivatives??? 
	Or 2nd derivatives??? What is the strategy to derive the parameters?????
	
	Out-of-plane bending is not yet implemented at all, but the following might be a good way:
	
	Lee SH, Palmo K, Krimm S : "New Out-of-Plane Angle and Bond Angle Internal Coordinates and 
	Related Potential Energy Functions for Molecular Mechanics and Dynamics Simulations", 
	J. Comp. Chem. 20, 1067-1084, (1999)
*/

class mm1_eng_exp1a : virtual public mm1_eng
{
	protected:
	
	mm1_bt1_data * bt1data; mm1_bt2_data * bt2data;
	
	vector<mm1_exp1_bt1> bt1_vector; i32s bt1_range[2];
	vector<mm1_exp1_bt2> bt2_vector; i32s bt2_range[2];
	vector<mm1_exp1_bt3> bt3_vector; i32s bt3_range[2];
	
	public:
	
	mm1_eng_exp1a(mm1_mdl &, i32s, i32s);
	virtual ~mm1_eng_exp1a(void);
	
	protected:
	
	void ComputeBT1(i32s);		// virtual
	void ComputeBT2(i32s);		// virtual
	void ComputeBT3(i32s);		// virtual
};

/*################################################################################################*/

/// "##UNSTABLE", nonbonded part, non-periodic.

class mm1_eng_exp1b1 : virtual public mm1_eng
{
	protected:
	
	vector<mm1_exp1_nbt1> nbt1_vector; i32s nbt1_range[2];
	
	public:
	
	mm1_eng_exp1b1(mm1_mdl &, i32s, i32s);
	virtual ~mm1_eng_exp1b1(void);
	
	protected:
	
	void ComputeNBT1(i32s);		// virtual
};

/// "##UNSTABLE", nonbonded part, exact periodic.

class mm1_eng_exp1b2 : virtual public mm1_eng
{
	public:		// why public!?!?!?!?
	
	i32s layers;
	
	public:
	
	mm1_eng_exp1b2(mm1_mdl &, i32s, i32s);
	virtual ~mm1_eng_exp1b2(void);
	
	protected:
	
	void CompNBTerm(i32s, i32s, i32s, bool, f64 *, bool);
	
	void ComputeNBT1(i32s);		// virtual
};

// "##UNSTABLE", nonbonded part, minimum image model.

class mm1_eng_exp1b3 : public mm1_eng_pbc
{
	protected:
	
	vector<mm1_exp1_nbt1> nbt1_vector; i32s nbt1_range[2];
	
	f64 sw1; f64 sw2;
	f64 swA; f64 swB;
	
	f64 shft1;
	f64 shft2;
	
	public:		// why public!?!?!?!?
	
	f64 limit;
	bool update;
	
	public:
	
	mm1_eng_exp1b3(mm1_mdl &, i32s, i32s);
	virtual ~mm1_eng_exp1b3(void);
	
	protected:
	
	void UpdateTerms(void);
	
	void ComputeNBT1(i32s);		// virtual
};

/*################################################################################################*/

class mm1_eng_exp1 : public mm1_eng_exp1a, public mm1_eng_exp1b1
{
	public:
	
	mm1_eng_exp1(mm1_mdl & p1, i32s p2, i32s p3) :
		mm1_eng_exp1a(p1, p2, p3), mm1_eng_exp1b1(p1, p2, p3), mm1_eng(p1, p2, p3)
	{
	}
};

class mm1_eng_exp1_tst : public mm1_eng_exp1a, public mm1_eng_exp1b2
{
	public:
	
	mm1_eng_exp1_tst(mm1_mdl & p1, i32s p2, i32s p3) :
		mm1_eng_exp1a(p1, p2, p3), mm1_eng_exp1b2(p1, p2, p3), mm1_eng(p1, p2, p3)
	{
	}
};

class mm1_eng_exp1_mim : public mm1_eng_exp1a, public mm1_eng_exp1b3
{
	public:
	
	mm1_eng_exp1_mim(mm1_mdl & p1, i32s p2, i32s p3) :
		mm1_eng_exp1a(p1, p2, p3), mm1_eng_exp1b3(p1, p2, p3), mm1_eng(p1, p2, p3)
	{
	}
};

/*################################################################################################*/

#endif	// MM1ENG1_H

// eof
