// MM1MDL.H : data storage for "all atoms"-models (molecular mechanics).

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef MM1MDL_H
#define MM1MDL_H

class mm1_mdl;

class mm1_readpdb_mdata;
struct mm1_readpdb_mdata_chain;
struct mm1_readpdb_data_atom;
struct mm1_readpdb_data_ssbond;

/*################################################################################################*/

#include "model.h"
#include "mm1util.h"
#include "mm1alg.h"

#include "interface.h"

#include <list>
#include <vector>
using namespace std;

#define NOT_FOUND 0x7fffffff	// numeric_limits<i32s>::max()?!?!?!

// OELib includes...
// OELib includes...
// OELib includes...

#include "mol.h"
//#include "oeutil.h"
//#include "data.h"
//#include "typer.h"
using namespace OpenEye;

/*################################################################################################*/

#define ENG_MM1_NORMAL		0x100		// mm1_eng_exp9
#define ENG_MM1_PERIODIC	0x200		// mm1_eng_exp1_mim

/**	A "##model" class for "##all-atoms" molecular mechanics.

	A "model" class is for data storage -> try to keep this small and simple, avoiding 
	all complicated stuff... Instead, try to do that complicated stuff using OELIB?!?!?!
	
	Make automatic conversion of part of the system/the whole system into the OELIB?!?!?
	
	It's probably best to keep the current data storage (due to multiple coordinate sets etc) 
	and current typerules (quick, efficient, easy to use) as they are now...
	
	So, keep this slim and start using OELIB if possible!!!
*/

class mm1_mdl : public model_extended, public trajectory_interface
{
	protected:
	
	i32s nmol;
	vector<mm1_chn_info> * ref_civ;
	
	list<mm1_atom> atom_list;
	list<mm1_bond> bond_list;
	
// default engine settings begin!!!
// default engine settings begin!!!
// default engine settings begin!!!
	
	public:
	
	static const char * engtab1[];
	static const i32s engtab2[];
	
	i32s default_eng;
	
	//protected:	// the dialog boxes need these...
	
	bool periodic;
	f64 box_hdim[3];
	f64 box_fdim[3];
	
// default engine settings end!!!
// default engine settings end!!!
// default engine settings end!!!
	
	friend class mm1_eng;
	friend class mm1_eng_pbc;
	friend class mm1_sequencebuilder;
	friend class mm1_geomopt;
	friend class mm1_moldyn;
	friend class mm1_ribbon;
	
	friend class mm2_mdl;
	
	friend class qm1_mdl;
	friend class gnome_qm1_docv;	// temporary, for MM->QM conversion, will move into all_atoms_interface???
	
	friend class file_trans;
	
	friend void CopyCRD(mm1_mdl *, mm1_eng *, i32u);
	friend void CopyCRD(mm1_eng *, mm1_mdl *, i32u);
	
	friend fGL mm1_GetESPValue(fGL *, mm1_mdl *, fGL *);
	friend fGL mm1_GetVDWSValue(fGL *, mm1_mdl *, fGL *);
	
	friend void DefineSecondaryStructure(mm1_mdl *);
	friend f64 HBondEnergy(mm1_mdl *, i32s *, i32s *);
	
	public:
	
	static mm1_sequencebuilder amino_builder;
	static mm1_sequencebuilder nucleic_builder;
	
	public:
	
	mm1_mdl(ostream *, class_factory &);
	~mm1_mdl(void);
	
	const char * GetProjectFileNameExtension(void);		// virtual
	
	mm1_eng * CreateDefaultEngine(void);
	
	void PushCRDSets(i32u);			// virtual
	void PopCRDSets(i32u);			// virtual
	
	void CopyCRDSet(i32u, i32u);		// virtual
	void SwapCRDSets(i32u, i32u);		// virtual
	
	void CenterCRDSet(i32u);		// virtual
	void ReserveCRDSets(i32u);		// virtual
	
	// methods for adding new atoms and bonds:
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	/**	This will just push new atom to the atom list and update colors if necessary.
		Mark for updated connectivity (for GUI's and "save")???
	*/
	void AddAtom(mm1_atom &);
	
	/// This will delete all bonds associated with this atom, and erase atom from the list...
	void RemoveAtom(iter_mm1al);
	
	/// This will add neighbor infos for both atoms and add the new bond into the bond list.
	void AddBond(mm1_bond &);
	
	/// This will remove infos from the atoms and erase bond from the bond list.
	void RemoveBond(iter_mm1bl);
	
	/// This will remove all atoms and bonds.
	void Clear(void);
	
	// methods for accessing atom/bond lists:
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	iter_mm1al GetAtomsBegin(void) { return atom_list.begin(); }
	iter_mm1al GetAtomsEnd(void) { return atom_list.end(); }

	iter_mm1bl GetBondsBegin(void) { return bond_list.begin(); }
	iter_mm1bl GetBondsEnd(void) { return bond_list.end(); }
	
	// methods for file I/O...
	
	bool ReadStream_OLD(istream &);		// this is the very old version...
//	bool ReadStream_v100(istream &);	// this will be the version 1.00, etc...

	bool ReadStream(istream &);	// virtual
	
	/// This is v1.00 output function for the mm1gp file format. The "!Section" idea is used here.
	void WriteStream(ostream &);	// virtual
	
	/**	Here we add hydrogens to elements C, N and O.
	
		Currently we use tetrahedral SP3 default geometries here in all cases, 
		and this works properly only for part of the cases...
		
		Random numbers are used, but not properly initialized ?!?!?!
	*/
	void AddHydrogens(void);
	void RemoveHydrogens(void);
	
	/// This will return atom_list.size().
	i32s GetAtomCount(void) { return atom_list.size(); }
	
	/// This will return bond_list.size().
	i32s GetBondCount(void) { return bond_list.size(); }
	
	/// This will return nmol (see GatherGroups() for more info).
	i32s GetMoleculeCount(void) { return nmol; }
	
	/**	This will group the atom list so that molecules (and chains/residues if defined) 
		will form continuous groups. Will validate mm1_model::nmol and permit the use 
		of mm1_model::GetRange()-functions.
	*/
	void GatherGroups(void);
	
	/// This is just a default version of GetRange() using the full range of atom list iterators...
	void GetRange(i32s, i32s, iter_mm1al *);
	
	/**	This will reduce the initial range of two atom list iterators to some subrange
		with certain values in certain mm1_atom::id[]-fields. Before using this you MUST
		call mm1_model::GatherGroups() to arrange the atom list !!!
		
		What the above explanation really tries to say, is that using this function you 
		can pick up a certain part of the model; for example a molecule, or a chain in a 
		macromolecule, or a range of residue in a chain.
	*/
	void GetRange(i32s, iter_mm1al *, i32s, iter_mm1al *);
	
	/**	Adding or removing atoms or bonds will generally scramble the grouping, 
		and when this happens one should call this function to discard all grouping information...
	*/
	void InvalidateGroups(void);
	
	i32s FindPath(mm1_atom *, mm1_atom *, i32s, i32s, i32s = 0);
	bool FindRing(mm1_atom *, mm1_atom *, signed char *, i32s, i32s, i32s = 0);
	
	void UpdateIndex(void);
	
	private:
	
	/**	This will set molecule numbers quickly using a recursive search algorithm. 
		This is private because only mm1_model::GatherGroups() should use this...
	*/
	void GatherAtoms(mm1_atom *, i32s);
	
	public:
	
	// the "readpdb" functions here are used to import PDB files as correctly as possible.
	// the PDB files represent experimental results, and in many cases the structures in files
	// have gapped and/or incomplete sequences, incomplete residues, and so on...
	
	// the "readpdb" functions do the import in two stages: in first stage read in the "metadata"
	// (all headers and remarks about the data including the original sequence), and in second stage
	// read in the data as correctly as possible. later, results from these two can be compared, for
	// example to evaluate quality of the data or to match the data with records in other databases.
	
	mm1_readpdb_mdata * readpdb_ReadMData(const char *);
	
	void readpdb_ReadData(const char *, mm1_readpdb_mdata *, i32s);
	i32s readpdb_ReadData_sub1(vector<mm1_readpdb_data_atom> &, i32s *, const char *, bool);
	void readpdb_ReadData_sub2(vector<mm1_readpdb_data_atom> &, i32s *, const char *, const char *, char);
	
	// here we have a set of Do???()-functions. the idea is that there is a set
	// of features that we wish to behave EXACTLY same way in each target/platform.
	// we then create a Do???()-function for the feature, and hide the details of
	// the user interface in a set of virtual functions.
	
	/// This will perform an energy calculation, and report the result.
	void DoEnergy(void);
	
	/// This will perform geometry optimization.
	void DoGeomOpt(mm1_geomopt_param &);
	
	/// This is used to ask user the GO options; this default function will just silently accept the defaults.
	virtual void GeomOptGetParam(mm1_geomopt_param &);
	
	/// This will perform molecular dynamics.
	void DoMolDyn(mm1_moldyn_param &);
	
	/// This is used to ask user the MD options; this default function will just silently accept the defaults.
	virtual void MolDynGetParam(mm1_moldyn_param &);
	
	/// This will perform a random search using torsions as variables. AN INITIAL VERSION...
	void DoRandomSearch(void);
	
	/// This will print the molecular formula and weight.
	void DoFormula(void);
	
	// these come from trajectory_interface...
	// these come from trajectory_interface...
	// these come from trajectory_interface...
	
	void OpenTrajectory(const char *);	// virtual
	void CloseTrajectory(void);		// virtual
	void ReadFrame(void);			// virtual
};

/*################################################################################################*/

fGL mm1_GetESPValue(fGL *, mm1_mdl *, fGL *);
fGL mm1_GetVDWSValue(fGL *, mm1_mdl *, fGL *);

/*################################################################################################*/

// define struct mm1_readpdb_mdata_chain before class mm1_readpdb_mdata, since the latter uses
// former in some inline functions...

// how to best relate mm1_readpdb_mdata_chain and mm1_chn_info !??!?!?!?!?
// maybe just by storing the alpha-carbon pointers here...

struct mm1_readpdb_mdata_chain
{
	char chn_id;
	char * seqres;
	
	vector<i32s> missing_residues;
	vector<mm1_atom *> alpha_carbons;
};

// class mm1_readpdb_mdata is a class just to make the memory management easier. the data members in
// the class are filled in mm1_mdl::readpdb_ReadMData(), and at end the object is just to be deleted.

class mm1_readpdb_mdata
{
	public:
	
	vector<mm1_readpdb_mdata_chain *> chn_vector;
	
	public:
	
	mm1_readpdb_mdata(void)
	{
	}
	
	~mm1_readpdb_mdata(void)
	{
		for (i32u n1 = 0;n1 < chn_vector.size();n1++)
		{
			delete[] chn_vector[n1]->seqres;	// delete the sequence...
			delete chn_vector[n1];			// delete the whole record...
		}
	}
};

// READPDB_MAX_CRDSETS is relevant only if READPDB_ENABLE_MULTIPLE_CRDSETS is defined...
// READPDB_MAX_CRDSETS is relevant only if READPDB_ENABLE_MULTIPLE_CRDSETS is defined...
// READPDB_MAX_CRDSETS is relevant only if READPDB_ENABLE_MULTIPLE_CRDSETS is defined...

#define READPDB_MAX_CRDSETS 10

struct mm1_readpdb_data_atom
{
	char chn_id;
	
	i32s res_num;
	char res_name[5];
	char atm_name[5];
	fGL crd[READPDB_MAX_CRDSETS][3];
	
	mm1_atom * ref;
};

struct mm1_readpdb_data_ssbond
{
	char chn_id;
	i32s res_num;
	
	mm1_atom * ref;
};

/*################################################################################################*/

#endif	// MM1MDL_H

// eof
