// MM2MDL.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm2mdl.h"

#include <algorithm>
#include <strstream>
using namespace std;

/*################################################################################################*/

mm2_mdl::mm2_mdl(ostream * p1, class_factory & p2, mm2_eng_param * p3) :
	model_extended(p1, p2), model_simple(p1, p2), trajectory_interface()
{
	prm.wtmp1 = 0.875;			// 0.875	// bt2 fc weight (in strands and loops)
	prm.wtmp2 = 1.000;			// 1.000	// bt4 fc weight
	prm.wtmp3 = 0.00025;			// 0.00025	// angle rep fc
	
	prm.wtor1 = 2.75;			// 2.75
	prm.wtor2 = 3.25;			// 3.25
	prm.lenjon = 1.2550;			// 1.2550	// DOMINATING LJ-TERM -> LOST SENSITIVITY ?!?!?!?
	prm.vdwrad = 1.35;			// 1.35		// will give an average radius of 0.275 nm...
	prm.screen1 = -20.929;			// -20.929
	prm.screen2 = 78.4 - prm.screen1;	// 78.4 minus above...
	prm.screen3 = 3.4781;			// 3.4781
	prm.screen4 = -0.01787;			// -0.01787
	prm.screen5 = 0.001;			// 0.001	// about 4 ...
	prm.dipole1 = 3.0;			// 3.0		// will give 27 kJ/mol for HB's in SSE's. (27+7)/2 = 17
	prm.dipole2 = 4.0;			// 4.0		// EASY TO CHECK USING A SINGLE ALPHA-HELIX, LIKE ALA12!!!
	prm.pbility1 = 0.0025;			// 0.0025	// should be about 0.0015 : CRC handbook of Chem & Phys
	prm.pbility2 = 0.0005;			// 0.0005	// 1st Student Edition, 1988, CRC Press Inc. (page E-62)
	prm.sasa1 = -0.5100;			// -0.5100	// near -0.5 -> in the middle, both effects?!?!?!
	prm.sasa2 = -52.00;			// -52.00	// somewhere near -50.0 !??!?!?!?!
	prm.solvrad = 0.15;			// 0.15		// H2O -> 0.15
	
//////////////////////////////////////////////////
//////////////////////////////////////////////////
//////////////////////////////////////////////////

	if (p3 != NULL)
	{
		prm.wtmp1 = p3->wtmp1;
		prm.wtmp2 = p3->wtmp2;
		prm.wtmp3 = p3->wtmp3;
		
		prm.wtor1 = p3->wtor1;
		prm.wtor2 = p3->wtor2;
		prm.lenjon = p3->lenjon;
		prm.vdwrad = p3->vdwrad;
		prm.screen1 = p3->screen1;
		prm.screen2 = p3->screen2;
		prm.screen3 = p3->screen3;
		prm.screen4 = p3->screen4;
		prm.screen5 = p3->screen5;
		prm.dipole1 = p3->dipole1;
		prm.dipole2 = p3->dipole2;
		prm.pbility1 = p3->pbility1;
		prm.pbility2 = p3->pbility2;
		prm.sasa1 = p3->sasa1;
		prm.sasa2 = p3->sasa2;
		prm.solvrad = p3->solvrad;
	}
}

mm2_mdl::~mm2_mdl(void)
{
}

const char * mm2_mdl::GetProjectFileNameExtension(void)
{
	static const char ext[] = "mm2gp";
	return ext;
}

i32s mm2_mdl::GetVirtAtomCount(void)
{
	i32s virt_atom_count = 0;
	
	for (i32u t1 = 0;t1 < chn_vector.size();t1++)
	{
		for (i32u t2 = 0;t2 < chn_vector[t1].res_vector.size();t2++)
		{
			virt_atom_count += chn_vector[t1].res_vector[t2].natm;
		}
	}
	
	return virt_atom_count;
}

void mm2_mdl::Reduce(mm1_mdl * mdl, mm1_readpdb_mdata *)
{
	mdl->GatherGroups();
	mm1_mdl::amino_builder.Identify(mdl);
	DefineSecondaryStructure(mdl);
	
	// check that the number of crd-sets matches...
	
	i32s csets = cs_vector.size(); i32s new_csets = (mdl->cs_vector.size() - csets);
	cout << "there were " << csets << " old crd-sets, creating " << new_csets << " new..." << endl;
	
	PushCRDSets(new_csets);
	
	for (i32u n1 = 0;n1 < cs_vector.size();n1++) cs_vector[n1].visible = true;
	
	// ok, start working...
	
	vector<mm1_chn_info> & ci_vector = (* mdl->ref_civ);
	
////////////////////////////////////////////////// debug!!!
////////////////////////////////////////////////// debug!!!
//	for (iter_mm1al it = mdl->atom_list.begin();it != mdl->atom_list.end();it++)
//	{
//		const fGL * color = (* it).el.GetColor();
//		mm2_debug newdb;
//		for (i32s n1 = 0;n1 < 3;n1++)
//		{
//			newdb.crd[n1] = (* it).crd_vector[0][n1];
//			newdb.color[n1] = color[n1] * 0.75;
//		}
//		debug_vector.push_back(newdb);
//	}
////////////////////////////////////////////////// debug!!!
////////////////////////////////////////////////// debug!!!

	// here we should check that we don't have any nucleic acid chains...
	// here we should check that we don't have any nucleic acid chains...
	// here we should check that we don't have any nucleic acid chains...
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		iter_mm1al range1[2]; mdl->GetRange(1, n1, range1);
		mm2_chn newchn; chn_vector.push_back(newchn);
		
		iter_mm1al range2[2]; i32s res = 0;
		
		while (true)
		{
			mdl->GetRange(2, range1, res, range2);
			if (range2[0] == range2[1]) break;
			
			i32s tmp1[2]; tmp1[0] = (* range2[0]).res_id >> 8;
			for (tmp1[1] = 0;tmp1[1] < (i32s) mm1_mdl::amino_builder.residue_vector.size();tmp1[1]++)
			{
				if (mm1_mdl::amino_builder.residue_vector[tmp1[1]].id == tmp1[0]) break;
			} if (tmp1[1] == (i32s) mm1_mdl::amino_builder.residue_vector.size()) continue;
			
			mm2_res newres = mm2_res(mm1_mdl::amino_builder.residue_vector[tmp1[1]].symbol, cs_vector.size(), & prm);
			
			vector<i32s> idv; idv.push_back(0x01);
			for (i32u n2 = 0;n2 < cs_vector.size();n2++)
			{
				Reduce_sub1(range2[0], idv, newres.crd_vector[0][n2].data, n2);
			}
			
			if (newres.natm > 1)
			{
				idv.resize(0);
				switch (newres.symbol)
				{
					case 'R':
					idv.push_back(0x21); idv.push_back(0x22);
					break;
					
					case 'N':
					idv.push_back(0x21);
					break;
					
					case 'D':
					idv.push_back(0x21);
					break;
					
					case 'C':
					idv.push_back(0x21);
					break;
					
					case 'Q':
					idv.push_back(0x21); idv.push_back(0x22);
					break;
					
					case 'E':
					idv.push_back(0x21); idv.push_back(0x22);
					break;
					
					case 'H':
					idv.push_back(0x21); idv.push_back(0x22);
					idv.push_back(0x23); idv.push_back(0x24); idv.push_back(0x25);
					break;
					
					case 'I':
					idv.push_back(0x22);
					break;
					
					case 'L':
					idv.push_back(0x21);
					break;
					
					case 'K':
					idv.push_back(0x21);
					break;
					
					case 'M':
					idv.push_back(0x21); idv.push_back(0x22); idv.push_back(0x23);
					break;
					
					case 'F':
					idv.push_back(0x21); idv.push_back(0x24);
					break;
					
					case 'W':
					idv.push_back(0x21); idv.push_back(0x23); idv.push_back(0x24);
					break;
					
					case 'Y':
					idv.push_back(0x21); idv.push_back(0x24);
					break;
				}
				
				for (i32u n2 = 0;n2 < cs_vector.size();n2++)
				{
					Reduce_sub1(range2[0], idv, newres.crd_vector[1][n2].data, n2);
				}
			}
			
			if (newres.natm > 2)
			{
				idv.resize(0);
				switch (newres.symbol)
				{
					case 'R':
					idv.push_back(0x24);
					break;
					
					case 'K':
					idv.push_back(0x23);
					break;
					
					case 'W':
					idv.push_back(0x25); idv.push_back(0x28);
					break;
				}
				
				for (i32u n2 = 0;n2 < cs_vector.size();n2++)
				{
					Reduce_sub1(range2[0], idv, newres.crd_vector[2][n2].data, n2);
				}
			}
			
			chn_vector.back().res_vector.push_back(newres);
			res++;
		}
		
		// peptide units...
		
		for (i32s n2 = 1;n2 < ((i32s) chn_vector.back().res_vector.size()) - 2;n2++)
		{
			mdl->GetRange(2, range1, n2, range2);
			
			iter_mm1al itc = range2[0];
			while (((* itc).res_id & 0xFF) != 0x01) itc++;
			
			iter_mm1al ito = range2[0];
			while (((* ito).res_id & 0xFF) != 0x10) ito++;
			
			for (i32u n3 = 0;n3 < cs_vector.size();n3++)
			{
				fGL * curr = chn_vector.back().res_vector[n2].crd_vector[0][n3].data;
				fGL * prev = chn_vector.back().res_vector[n2 - 1].crd_vector[0][n3].data;
				fGL * next = chn_vector.back().res_vector[n2 + 1].crd_vector[0][n3].data;
				
				v3d<fGL> v1(curr, prev); v3d<fGL> v2(curr, next);
				v3d<fGL> v3((* ito).crd_vector[n3].data, (* itc).crd_vector[n3].data);
				
				chn_vector.back().res_vector[n2].peptide_vector[n3] = v1.tor(v2, v3);
			}
		}
	}
	
	// disulphide bridges...
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		iter_mm1al range1a[2];
		mdl->GetRange(1, n1, range1a);
		
		// intra-chain ones...
		
		vector<i32s> cys_data1;
		vector<mm1_atom *> cys_ref1;
		
		for (i32s n2 = 0;n2 < ci_vector[n1].length;n2++)
		{
			if (ci_vector[n1].sequence[n2] == 'C')
			{
				bool flag = true;
				
				iter_mm1al range1b[2];
				mdl->GetRange(2, range1a, n2, range1b);
				if (range1b[0] == range1b[1]) flag = false;
				
				iter_mm1al it1 = range1b[0];
				while (it1 != range1b[1] && ((* it1).res_id & 0xFF) != 0x21) it1++;
				if (it1 == range1b[1]) flag = false;
				
				if (flag)
				{
					cys_data1.push_back(n2);
					cys_ref1.push_back(& (* it1));
				}
			}
		}
		
		for (i32s n2 = 0;n2 < ((i32s) cys_ref1.size()) - 1;n2++)
		{
			for (i32s n3 = n2 + 1;n3 < (i32s) cys_ref1.size();n3++)
			{
				mm1_bond tb = mm1_bond(cys_ref1[n2], cys_ref1[n3], bondtype('S'));
				iter_mm1bl it1 = find(mdl->bond_list.begin(), mdl->bond_list.end(), tb);
				if (it1 != mdl->bond_list.end())
				{
					mm2_dsb newdsb;
					newdsb.chn[0] = n1; newdsb.res[0] = cys_data1[n2];
					newdsb.chn[1] = n1; newdsb.res[1] = cys_data1[n3];
					dsb_vector.push_back(newdsb);
				}
			}
		}
		
		// interchain ones...
		
		for (i32u n2 = n1 + 1;n2 < ci_vector.size();n2++)
		{
			iter_mm1al range2a[2];
			mdl->GetRange(1, n2, range2a);
			
			vector<i32s> cys_data2;
			vector<mm1_atom *> cys_ref2;
			
			for (i32s n3 = 0;n3 < ci_vector[n2].length;n3++)
			{
				if (ci_vector[n2].sequence[n3] == 'C')
				{
					bool flag = true;
					
					iter_mm1al range2b[2];
					mdl->GetRange(2, range2a, n3, range2b);
					if (range2b[0] == range2b[1]) flag = false;
					
					iter_mm1al it1 = range2b[0];
					while (it1 != range2b[1] && ((* it1).res_id & 0xFF) != 0x21) it1++;
					if (it1 == range2b[1]) flag = false;
					
					if (flag)
					{
						cys_data2.push_back(n3);
						cys_ref2.push_back(& (* it1));
					}
				}
			}
			
			for (i32u n3 = 0;n3 < cys_ref1.size();n3++)
			{
				for (i32u n4 = 0;n4 < cys_ref2.size();n4++)
				{
					mm1_bond tb = mm1_bond(cys_ref1[n3], cys_ref2[n4], bondtype('S'));
					iter_mm1bl it1 = find(mdl->bond_list.begin(), mdl->bond_list.end(), tb);
					if (it1 != mdl->bond_list.end())
					{
						mm2_dsb newdsb;
						newdsb.chn[0] = n1; newdsb.res[0] = cys_data1[n3];
						newdsb.chn[1] = n2; newdsb.res[1] = cys_data2[n4];
						dsb_vector.push_back(newdsb);
					}
				}
			}
		}
	}
	
	// secondary structure...
	
	// the default state is always loop. strands have the same places as in K&S.
	// helices are shifted: the smallest helical peptide is "4...." -> "LHHHL"!!!!!
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		for (i32s n2 = 0;n2 < ci_vector[n1].length;n2++)
		{
			switch (ci_vector[n1].state[n2])
			{
				case 'S':
				chn_vector[n1].res_vector[n2].state = STATE_STRAND;
				break;
				
				default:
				chn_vector[n1].res_vector[n2].state = STATE_LOOP;
				break;
			}
		}
				
		for (i32s n2 = 2;n2 < ci_vector[n1].length - 2;n2++)
		{
			switch (ci_vector[n1].state[n2 - 2])
			{
				case '4':
				chn_vector[n1].res_vector[n2 - 1].state = STATE_HELIX;
				chn_vector[n1].res_vector[n2 + 0].state = STATE_HELIX;
				chn_vector[n1].res_vector[n2 + 1].state = STATE_HELIX;
				break;
			}
		}
	}
	
	// still it would be best to check that results of the above are ok:
	// * secondary structural elements should be separated by at least one loop-residue!!!
	// * what is the smallest meaningful element ??? FOR HELIX 3, FOR STRAND 1 !!!!!!!!!!!!
	
	UpdateConstraints();
}

void mm2_mdl::Reduce_sub1(iter_mm1al it1, vector<i32s> & idv, fGL * crd, i32u cset)
{
	vector<mm1_atom *> refv;
	for (i32u n1 = 0;n1 < idv.size();n1++)
	{
		iter_mm1al it2 = it1;
		while (((* it2).res_id & 0xFF) != idv[n1]) it2++;
		refv.push_back(& (* it2));
	}
	
	for (i32u n1 = 0;n1 < 3;n1++)
	{
		crd[n1] = 0.0;
		for (i32u n2 = 0;n2 < refv.size();n2++)
		{
			crd[n1] += refv[n2]->crd_vector[cset][n1];
		}
		
		crd[n1] /= (f64) refv.size();
	}
}

// the name of this function is quite misleading...
// the name of this function is quite misleading...
// the name of this function is quite misleading...

// here we only set charges for terminal residues!!!!!!!

void mm2_mdl::UpdateConstraints(void)
{
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		chn_vector[n1].res_vector.front().type[0] = TYPE_CPOS;
		chn_vector[n1].res_vector.back().type[0] = TYPE_CNEG;
	}
}

bool mm2_mdl::ReadStream(istream & istr)
{
	i32s ncs; istr >> buffer >> ncs;
	PushCRDSets(ncs - cs_vector.size());
	
	i32s n1; i32s n2; i32s n3; i32s n4;
	char tmp1; char tmp2; fGL tmp3;
	
	while (istr.peek() != '#')
	{
		if (istr.peek() == 'c')		// chn
		{
			mm2_chn newchn;
			chn_vector.push_back(newchn);
		}
		
		if (istr.peek() == 'r')		// res
		{
			istr >> buffer >> n1 >> tmp1 >> tmp2 >> tmp3;
			istr.getline(buffer, sizeof(buffer));
			
			mm2_res newres = mm2_res(tmp1, cs_vector.size(), & prm);
			
			switch (tmp2)
			{
				case 'H':	newres.state = STATE_HELIX; break;
				case 'S':	newres.state = STATE_STRAND; break;
				default:	newres.state = STATE_LOOP;
			}
			
			for (n1 = 0;n1 < ncs;n1++) newres.peptide_vector[n1] = tmp3;
			
			for (n1 = 0;n1 < (i32s) newres.natm;n1++)
			{
				for (n2 = 0;n2 < ncs;n2++)
				{
					istr >> tmp3; newres.crd_vector[n1][n2][0] = tmp3;
					istr >> tmp3; newres.crd_vector[n1][n2][1] = tmp3;
					istr >> tmp3; newres.crd_vector[n1][n2][2] = tmp3;
				}
				
				if (n1 + 1 == (i32s) newres.natm) continue;
				else istr.getline(buffer, sizeof(buffer));
			}
			
			chn_vector.back().res_vector.push_back(newres);
		}
		
		if (istr.peek() == 'B')		// bridge
		{
			mm2_dsb newdsb; istr.get();
			istr >> n1 >> n2 >> n3 >> n4;
			
			newdsb.chn[0] = n1; newdsb.res[0] = n2;
			newdsb.chn[1] = n3; newdsb.res[1] = n4;
			
			dsb_vector.push_back(newdsb);
		}
		
		istr.getline(buffer, sizeof(buffer));
	}
	
	UpdateConstraints();
	
	return true;
}

void mm2_mdl::WriteStream(ostream & ostr)
{
	ostr << "ncs " << cs_vector.size() << endl;
	
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		ostr << "chn " << n1 << endl;
		
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			ostr << "res " << n2 << " ";
			ostr << chn_vector[n1].res_vector[n2].symbol << " ";
			
			switch (chn_vector[n1].res_vector[n2].state)
			{
				case STATE_HELIX:	ostr << "H"; break;
				case STATE_STRAND:	ostr << "S"; break;
				default:		ostr << "L";
			} ostr << " ";
			
			for (i32u n3 = 0;n3 < cs_vector.size();n3++)
			{
				ostr << chn_vector[n1].res_vector[n2].peptide_vector[n3] << endl;
			}
			
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				for (i32u n4 = 0;n4 < cs_vector.size();n4++)
				{
					ostr << chn_vector[n1].res_vector[n2].crd_vector[n3][n4][0] << " ";
					ostr << chn_vector[n1].res_vector[n2].crd_vector[n3][n4][1] << " ";
					ostr << chn_vector[n1].res_vector[n2].crd_vector[n3][n4][2] << " ";
				} ostr << endl;
			}
		}
		
		ostr << "endchn" << endl;
	}
	
	for (i32u n1 = 0;n1 < dsb_vector.size();n1++)
	{
		ostr << "B ";
		ostr << dsb_vector[n1].chn[0] << " ";
		ostr << dsb_vector[n1].res[0] << " ";
		ostr << dsb_vector[n1].chn[1] << " ";
		ostr << dsb_vector[n1].res[1] << endl;
	}
	
	ostr << "#end" << endl;
}

void mm2_mdl::PushCRDSets(i32u p1)
{
	for (i32u n1 = 0;n1 < p1;n1++) cs_vector.push_back(crd_set());
	
	fGL_a3 newcrd = { 0.0, 0.0, 0.0 };
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				for (i32u n4 = 0;n4 < p1;n4++)
				{
					chn_vector[n1].res_vector[n2].crd_vector[n3].push_back(newcrd);
				}
			}
			
			for (i32u n3 = 0;n3 < p1;n3++)
			{
				chn_vector[n1].res_vector[n2].peptide_vector.push_back(0.0);
			}
		}
	}
}

void mm2_mdl::PopCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::CopyCRDSet(i32u p1, i32u p2)
{
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				chn_vector[n1].res_vector[n2].crd_vector[n3][p2][0] = chn_vector[n1].res_vector[n2].crd_vector[n3][p1][0];
				chn_vector[n1].res_vector[n2].crd_vector[n3][p2][1] = chn_vector[n1].res_vector[n2].crd_vector[n3][p1][1];
				chn_vector[n1].res_vector[n2].crd_vector[n3][p2][2] = chn_vector[n1].res_vector[n2].crd_vector[n3][p1][2];
			}
			
			chn_vector[n1].res_vector[n2].peptide_vector[p2] = chn_vector[n1].res_vector[n2].peptide_vector[p1];
		}
	}
}

void mm2_mdl::SwapCRDSets(i32u, i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::CenterCRDSet(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::ReserveCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::DoEnergy(void)
{
	mm2_eng * eng = new mm2_eng(* this);
	CopyCRD(this, eng, 0); eng->Compute(0);
	
	char buffer[1024];
	ostrstream str(buffer, sizeof(buffer)); str.setf(ios::fixed); str.precision(8);
	
// this will print also the components...
// this will print also the components...
// this will print also the components...
//str << eng->energy_bt1 << " ";
//str << eng->energy_bt2 << " ";
//str << eng->energy_bt3 << " ";
//str << eng->energy_bt4 << " ";
//str << eng->energy_nbt1a << " ";
//str << eng->energy_nbt1b << " ";
//str << eng->energy_nbt2a << " ";
//str << eng->energy_nbt2b << " ";
//str << eng->energy_nbt3 << endl;

	str << "Energy = " << eng->energy << " kJ/mol + " << eng->constraints << " kJ/mol = ";
	str << (eng->energy + eng->constraints) << " kJ/mol" << ends;
	
	err->Message(buffer);
	delete eng;
}

void mm2_mdl::DoGeomOpt(mm2_geomopt_param & param)
{
  char buffer[1024];
	GeomOptGetParam(param);
	if (!param.confirm) return;
	
	mm2_eng * eng = new mm2_eng(* this); CopyCRD(this, eng, 0);
	mm2_geomopt * opt = new mm2_geomopt(eng, 50, 0.025);	// optimal settings?!?!?
	
	for (i32s n1 = 0;n1 < param.nsteps;n1++)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		sprintf(buffer, "%d %10.3f  %4.6e", n1, opt->optval, opt->optstp);
		cout << buffer << endl;
		
		if (!(n1 % 10) || (n1 + 1) == param.nsteps)
		{
			CopyCRD(eng, this, 0); CenterCRDSet(0);
			UpdateAllGraphicsViews(true);
		}
	}
	
	delete opt;
	delete eng;
}

void mm2_mdl::GeomOptGetParam(mm2_geomopt_param & param)
{
	param.confirm = true;
}

#define CFRQ 1
#define CSIZE 100

#define START 10000

void mm2_mdl::DoMolDyn(mm2_moldyn_param & param)
{
	MolDynGetParam(param);
	if (!param.confirm) return;
	
	mm2_eng * eng = new mm2_eng(* this); CopyCRD(this, eng, 0);
	mm2_moldyn * dyn = new mm2_moldyn(eng, param.temperature, param.timestep);
	
// WHAT IS THE LARGEST STABLE INTEGRATION TIME STEP !?!?!??!
// WHAT IS THE LARGEST STABLE INTEGRATION TIME STEP !?!?!??!
// WHAT IS THE LARGEST STABLE INTEGRATION TIME STEP !?!?!??!

	ofstream ofile;
	ofile.open(param.filename, ios::out | ios::binary);
	
	const char file_id[10] = "traj_v10";
	const int number_of_atoms = GetVirtAtomCount();
	const int total_frames = param.nsteps / 100;
	
	ofile.write((char *) file_id, 8);					// file id, 8 chars.
	ofile.write((char *) & number_of_atoms, sizeof(number_of_atoms));	// number of atoms, int.
	ofile.write((char *) & total_frames, sizeof(total_frames));		// total number of frames, int.
	
	i32s ccnt = 0; f64 control[CSIZE];
	
	for (i32s n1 = 0;n1 < START + param.nsteps;n1++)
	{
		dyn->TakeMDStep();
		if (!(n1 % 50))		// 250
		{
			cout << n1 << " T = " << dyn->ConvEKinTemp(dyn->kin);
			cout << " Ep = " << dyn->pot << " Et = " << (dyn->kin + dyn->pot) << endl;
		}
		
		if (n1 < START)
		{
			if (!(n1 % CFRQ))
			{
				control[ccnt++] = dyn->kin;
			}
			
			if (ccnt == CSIZE)
			{
				f64 sum = 0.0;
				for (i32s n9 = 0;n9 < ccnt;n9++)
				{
					sum += control[n9];
				}
				
				sum /= (f64) ccnt; ccnt = 0; f64 rtemp = (f64) (n1 + CFRQ) / (f64) START;
				cout << "average = " << sum << " kJ/mol = " << dyn->ConvEKinTemp(sum) << " K." << endl;
				dyn->SetEKin(dyn->kin + (dyn->ConvTempEKin(dyn->temp * rtemp) - sum));
			}
		}
		else if (!(n1 % 100))
		{
			CopyCRD(eng, this, 0);
			
			const float ekin = dyn->kin;
			const float epot = dyn->pot;
			
			ofile.write((char *) & ekin, sizeof(ekin));	// kinetic energy, float.
			ofile.write((char *) & epot, sizeof(epot));	// potential energy, float.
			
			for (i32u t1 = 0;t1 < chn_vector.size();t1++)
			{
				for (i32u t2 = 0;t2 < chn_vector[t1].res_vector.size();t2++)
				{
					for (i32u t3 = 0;t3 < chn_vector[t1].res_vector[t2].natm;t3++)
					{
						for (i32u t4 = 0;t4 < 3;t4++)	// all coordinates, float.
						{
							float t1a = chn_vector[t1].res_vector[t2].crd_vector[t3][0][t4];
							ofile.write((char *) & t1a, sizeof(t1a));
						}
					}
				}
			}
		}
		
		if (!(n1 % 50))
		{
			CopyCRD(eng, this, 0);
			UpdateAllGraphicsViews(true);
		}
	}
	
	ofile.close();
	
	delete dyn; delete eng;
	UpdateAccumValues();
}

void mm2_mdl::MolDynGetParam(mm2_moldyn_param & param)
{
	param.confirm = true;
}

void mm2_mdl::OpenTrajectory(const char * fn)
{
	if (!trajfile)
	{
		trajfile = new ifstream(fn, ios::in | ios::binary);
		
		trajfile->seekg(8, ios::beg);	// skip the file id...
		
		int natoms;
		trajfile->read((char *) & natoms, sizeof(natoms));
		
		if (natoms != GetVirtAtomCount())
		{
			cout << "incompatible file : different number of atoms!" << endl;
			CloseTrajectory(); return;
		}
		
		trajfile->read((char *) & total_traj_frames, sizeof(total_traj_frames));
		cout << "the trajectory file contains " << total_traj_frames << " frames." << endl;
		
		current_traj_frame = 0;
	}
	else cout << "trajectory already exists!" << endl;
}

void mm2_mdl::CloseTrajectory(void)
{
	if (trajfile != NULL)
	{
		trajfile->close();
		delete trajfile;
		
		trajfile = NULL;
	}
}

void mm2_mdl::ReadFrame(void)
{
	i32s atoms = GetVirtAtomCount();
	
	i32s place = 8 + 2 * sizeof(int);					// skip the header...
	place += (2 + 3 * atoms) * sizeof(float) * current_traj_frame;		// get the correct frame...
	place += 2 * sizeof(float);						// skip epot and ekin...
	
	trajfile->seekg(place, ios::beg);
	
	for (i32u t1 = 0;t1 < chn_vector.size();t1++)
	{
		for (i32u t2 = 0;t2 < chn_vector[t1].res_vector.size();t2++)
		{
			for (i32u t3 = 0;t3 < chn_vector[t1].res_vector[t2].natm;t3++)
			{
				for (i32u t4 = 0;t4 < 3;t4++)
				{
					float t1a; trajfile->read((char *) & t1a, sizeof(t1a));
					chn_vector[t1].res_vector[t2].crd_vector[t3][0][t4] = t1a;
				}
			}
		}
	}
}

/*################################################################################################*/

// f = a/r
// df/dr = -a/r^2

// f = a/r^2
// df/dr = -2a/r^-3

fGL mm2_GetESPValue(fGL * crd, mm2_mdl * ref, fGL * grad)
{
	fGL vdwsv = 0.0;
	if (grad != NULL) grad[0] = grad[1] = grad[2] = 0.0;
	
	// contribution from charges. PRIMARY!!! this will give the large-scale shape...
	// contribution from charges. PRIMARY!!! this will give the large-scale shape...
	// contribution from charges. PRIMARY!!! this will give the large-scale shape...
	
	for (i32u chn = 0;chn < ref->chn_vector.size();chn++)
	{
		for (i32u res = 0;res < ref->chn_vector[chn].res_vector.size();res++)
		{
			for (i32u atm = 0;atm < ref->chn_vector[chn].res_vector[res].natm;atm++)
			{
				fGL charge;
				if (ref->chn_vector[chn].res_vector[res].type[atm] == TYPE_CPOS) charge = +1.0;
				else if (ref->chn_vector[chn].res_vector[res].type[atm] == TYPE_CNEG) charge = -1.0;
				else continue;
				
				fGL tmp1[3]; fGL r2 = 0.0;
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					tmp1[n1] = crd[n1] - ref->chn_vector[chn].res_vector[res].crd_vector[atm][0][n1];
					r2 += tmp1[n1] * tmp1[n1];
				}
				
				if (r2 == 0.0) return +1.0e+35;		// numeric_limits<fGL>::max()?!?!?!
				fGL r1 = sqrt(r2);
				
				// do we have a correct constant here??? I think so, if we define
				// electrostatic potential as potential energy of a unit positive charge.
				
	// e = A + B/(1+k*exp(L*B*r))
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2
	
	f64 t7a = exp(ref->prm.screen2 * ref->prm.screen4 * r1);
	f64 t7b = 1.0 + ref->prm.screen3 * t7a; f64 t7c = t7b * t7b;
	f64 t7d = ref->prm.screen1 + ref->prm.screen2 / t7b;
	
	f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * ref->prm.screen2 * ref->prm.screen2 * ref->prm.screen3 * ref->prm.screen4;
	
				// f = Q/(e*r)
				// df/dr = -Q * ((1/e*r^2) + (de/dr)/(e^2*r))
				
				fGL qq = 138.9354518 * charge;
				vdwsv += qq / (t7d * r1);
				
				if (grad != NULL)	// sign??? constants ???
				{
					fGL df = -qq * (1.0 / (t7d * r2) - t7f / (t7c * t7e * r1));
					
					for (i32s n1 = 0;n1 < 3;n1++)
					{
						fGL dr = tmp1[n1] / r1;
						grad[n1] += df * dr;
					}
				}
			}
		}
	}
	
	// contribution from dipoles. SECONDARY!!! mostly just adds complexity (and runtime)...
	// contribution from dipoles. SECONDARY!!! mostly just adds complexity (and runtime)...
	// contribution from dipoles. SECONDARY!!! mostly just adds complexity (and runtime)...
	
/*	// DISABLED, TO MAKE THIS RUN FASTER...

	for (i32u chn = 0;chn < ref->chn_vector.size();chn++)
	{
		for (i32s res = 1;res < ((i32s) ref->chn_vector[chn].res_vector.size()) - 2;res++)
		{
			if (ref->chn_vector[chn].res_vector[res + 1].symbol == 'P') continue;
			
			fGL * pp = ref->chn_vector[chn].res_vector[res - 1].crd_vector[0][0].data;
			fGL * p1 = ref->chn_vector[chn].res_vector[res + 0].crd_vector[0][0].data;
			fGL * p2 = ref->chn_vector[chn].res_vector[res + 1].crd_vector[0][0].data;
			
			fGL comp[3];
			comp[0] = crd[0] - (p1[0] + p2[0]) / 2.0;
			comp[1] = crd[1] - (p1[1] + p2[1]) / 2.0;
			comp[2] = crd[2] - (p1[2] + p2[2]) / 2.0;
			
			fGL r2 = comp[0] * comp[0] + comp[1] * comp[1] + comp[2] * comp[2];
			fGL r1 = sqrt(r2);
			
			comp[0] /= r1; comp[1] /= r1; comp[2] /= r1;
			
			v3d<fGL> va = v3d<fGL>(p1, pp); va = va / va.len();
			v3d<fGL> vb = v3d<fGL>(p1, p2); vb = vb / vb.len();
			v3d<fGL> vc = va.vpr(vb); vc = vc / vc.len();
			v3d<fGL> vn = vb.vpr(vc); vn = vn / vn.len();
			
			fGL angle = ref->chn_vector[chn].res_vector[res].peptide_vector[0];
			v3d<fGL> vp = vn * cos(angle) - vc * sin(angle);
			
			fGL cosine = vp[0] * comp[0] + vp[1] * comp[1] + vp[2] * comp[2];
			fGL qd = 138.9354518 * 0.077711 * ref->prm.dipole1;
			
	f64 t9x = r2 * r2;	// this is r^4 !!!!!!!
	f64 t9y = t9x * t9x;	// this is r^8 !!!!!!!
	
	// e = A + B/(1+k*exp(L*B*r)) + Z/r^9
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2 - 9Z/r^10
	
	f64 t7a = exp(ref->prm.screen2 * ref->prm.screen4 * r1);
	f64 t7b = 1.0 + ref->prm.screen3 * t7a; f64 t7c = t7b * t7b;
	f64 t7d = ref->prm.screen1 + ref->prm.screen2 / t7b + ref->prm.screen5 / (t9y * r1);
	
	f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * ref->prm.screen2 * ref->prm.screen2 * ref->prm.screen3 * ref->prm.screen4;
	f64 t7g = 9.0 * ref->prm.screen5 / (t9y * r2);
	
			// f = Q/(e*r^2)
			// df/dr = -Q * ((2/e*r^3) + (de/dr)/(e^2*r^2))
			
			vdwsv += qd * cosine / (t7d * r2);
			
			if (grad != NULL)	// sign??? constants???
			{
				fGL t3a = -qd * cosine * (2.0 / (t7d * r2 * r1) - (t7f / t7c + t7g) / (t7e * r2));
				fGL t3b = qd / (t7d * r2);
				
				fGL deri[3];
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					deri[n1] = 0.0;
					for (i32s n2 = 0;n2 < 3;n2++)
					{
						fGL dunitr;
						if (n1 != n2) dunitr = -comp[n1] * comp[n2] / r1;
						else dunitr = (1.0 - comp[n1] * comp[n2]) / r1;
						
						deri[n1] += dunitr * vp[n2];
					}
				}
				
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					grad[n1] += t3a * comp[n1] + t3b * deri[n1];
				}
			}
		}
	}
	
*/	// DISABLED, TO MAKE THIS RUN FASTER...

	return vdwsv;
}

/*################################################################################################*/

// f = sum[(r/a)^-3] = sum[a^3 * r^-3]		// is it now r^-12 ?!?!?
// df/dr = -3 * sum[a^3 * r^-4]

fGL mm2_GetSASValue(fGL * crd, mm2_mdl * ref, fGL * grad)
{
	fGL sasv = 0.0;
	if (grad != NULL) grad[0] = grad[1] = grad[2] = 0.0;
	
	for (i32u chn = 0;chn < ref->chn_vector.size();chn++)
	{
		for (i32u res = 0;res < ref->chn_vector[chn].res_vector.size();res++)
		{
			for (i32u atm = 0;atm < ref->chn_vector[chn].res_vector[res].natm;atm++)
			{
				fGL tmp1[3]; fGL r2 = 0.0;
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					tmp1[n1] = crd[n1] - ref->chn_vector[chn].res_vector[res].crd_vector[atm][0][n1];
					r2 += tmp1[n1] * tmp1[n1];
				}
				
				if (r2 == 0.0) return +1.0e+35;		// numeric_limits<fGL>::max()?!?!?!
				fGL r1 = sqrt(r2);
				
				fGL tmp2 = r1 / (ref->chn_vector[chn].res_vector[res].vdwr[atm] + 0.15);
				fGL qqq = tmp2 * tmp2 * tmp2 * tmp2;
				
				fGL tmp3 = 1.0 / (qqq * qqq * qqq);
				sasv += tmp3;
				
				if (grad != NULL)	// sign ??? constant ???
				{
					for (i32s n1 = 0;n1 < 3;n1++)
					{
						fGL tmp4 = tmp1[n1] / r1;
						fGL tmp5 = tmp4 * tmp3 / tmp2;
						grad[n1] += tmp5;
					}
				}
			}
		}
	}
	
	return sasv;
}

/*################################################################################################*/

mm2_superimpose::mm2_superimpose(mm2_mdl * p1, i32s p2, i32s p3) : superimpose(p2, p3)
{
	mdl = p1;
}

mm2_superimpose::~mm2_superimpose(void)
{
}

f64 mm2_superimpose::GetValue(void)
{
	value = 0.0; counter = 0;
	
	for (i32u n1 = 0;n1 < mdl->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < mdl->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < mdl->chn_vector[n1].res_vector[n2].natm;n3++)
/////////////////////////////////////////////
//i32u n3 = 0;	// MAIN-CHAIN ONLY !!!!!
/////////////////////////////////////////////
			{
				f64 d1[3];
				d1[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[0];
				d1[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[1];
				d1[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[2];
				
				f64 d2[3];
				d2[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0];
				d2[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1];
				d2[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2];
				
				Compare(d1, d2, false);
			}
		}
	}
	
	return value;
}

f64 mm2_superimpose::GetGradient(void)
{
	value = 0.0; counter = 0;
	dloc[0] = dloc[1] = dloc[2] = 0.0;
	drot[0] = drot[1] = drot[2] = 0.0;
	
	for (i32u n1 = 0;n1 < mdl->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < mdl->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < mdl->chn_vector[n1].res_vector[n2].natm;n3++)
/////////////////////////////////////////////
//i32u n3 = 0;	// MAIN-CHAIN ONLY !!!!!
/////////////////////////////////////////////
			{
				f64 d1[3];
				d1[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[0];
				d1[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[1];
				d1[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[2];
				
				f64 d2[3];
				d2[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0];
				d2[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1];
				d2[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2];
				
				Compare(d1, d2, true);
			}
		}
	}
	
//////////////////////////////
//////////////////////////////
//
//cout << "analytical:" << endl;
//cout << dloc[0] << endl;
//cout << dloc[1] << endl;
//cout << dloc[2] << endl;
//cout << drot[0] << endl;
//cout << drot[1] << endl;
//cout << drot[2] << endl;
//
//cout << "numerical:" << endl;
//f64 vara1 = value;
//
//for (i32s qq = 0;qq < 3;qq++)
//{
//	f64 vara2 = loc[qq];
//	
//	loc[qq] = vara2 + 0.00001;
//	cout << ((GetValue() - vara1) / 0.00001) << endl;
//	
//	loc[qq] = vara2;
//}
//
//for (i32s qq = 0;qq < 3;qq++)
//{
//	f64 vara2 = rot[qq];
//	
//	rot[qq] = vara2 + 0.00001;
//	cout << ((GetValue() - vara1) / 0.00001) << endl;
//	
//	rot[qq] = vara2;
//}
//
//value = vara1;
//int zzz; cin >> zzz;
//
//////////////////////////////
//////////////////////////////

	return value;
}

void mm2_superimpose::Transform(void)
{
	value = 0.0; counter = 0;
	
	for (i32u n1 = 0;n1 < mdl->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < mdl->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < mdl->chn_vector[n1].res_vector[n2].natm;n3++)
/////////////////////////////////////////////
//i32u n3 = 0;	// MAIN-CHAIN ONLY !!!!!
/////////////////////////////////////////////
			{
				f64 d1[3];
				d1[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[0];
				d1[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[1];
				d1[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[2];
				
				f64 d2[3];
				d2[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0];
				d2[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1];
				d2[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2];
				
				f64 d3[3];
				Compare(d1, d2, false, d3);
				
				mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0] = d3[0];
				mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1] = d3[1];
				mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2] = d3[2];
			}
		}
	}
}

/*################################################################################################*/

// eof
