// MODEL.H : a set of "model" base classes common to all models.

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef MODEL_H
#define MODEL_H

class model_simple;
class model_extended;

class crd_set;
class superimpose;

/*################################################################################################*/

#include "conjgrad.h"
#include "factory.h"

#include <math.h>

#include <vector>
#include <fstream>
using namespace std;

/*################################################################################################*/

/**	A base class for data storage; "##model" classes are used to implement the 
	necessary data storage of various models (like different quantum-mechanical models 
	and molecular mechanics models).
	
	If we study the design of this software and wish to name some central feature that 
	would give the best brief description of this design, then the feature should be 
	"##the separation of data storage and computation". By "##data storage" we mean the 
	description of the system under study: in MM models that means the data about atoms 
	and bonds, and in QM models that might mean data about atoms and the state of the 
	system (total charge, spin state???). This data is common to all models that belong 
	to the same category (MM, QM), and can be easily separated from the individual 
	implementations of different computational techniques (the forcefield X, the 
	quantum-chemical theory Y). This is because the graphical user interfaces are 
	dependent on those descriptions of the systems, and not on those 
	computational details.
	
	Practically this division is made in the following way: we call those data storage 
	classes as "##model"-classes, and those classes that implement the computational 
	details as "##engine"-classes. The "##model" class always decribes the system under 
	study. When we want to compute something, we create an instance of some suitable 
	"##engine"-class using our "##model"-class. The "##engine"-class will copy that 
	information it needs, and calculates those results it is supposed to calculate. 
	If we calculate some useful results that change our original system, we can copy 
	those results back to our "##model"-class.
	
	The "##simple" models can handle only a single "##conformation", or a single set 
	of coordinates.
*/

class model_simple
{
	public:
	
	/**	A general-purpose output stream; 
		if this is NULL then no output is allowed/needed.
	*/
	
	ostream * ostr;
	
	/// A general-purpose text buffer.
	
	char buffer[1024];
	
	/**	A newer system, which should eventually replace "##ostr" in user interaction. 
		But what about logfiles? Do we need them???
		(GRH: The err_util class could optionally direct output to a logfile as well)
	*/
	
	err_util * err;
	prefs	 * model_prefs;
	
	protected:
	
	class_factory * factory;
	
	private:
	
	static char local_fn_buffer[256];
	
	public:
	
	model_simple(ostream *, class_factory &);
	virtual ~model_simple(void);
	
	static const char * GetFullPrimaryFN(const char *);
	static const char * GetFullSecondaryFN(const char *);
	
	static void OpenParameterFile(ifstream &, bool, const char *);
	
	virtual const char * GetProjectFileNameExtension(void) = 0;
	
	/// This will read the model from a file.
	
	virtual bool ReadStream(istream &) = 0;
	
	/// This will write the model into a file.
	
	virtual void WriteStream(ostream &) = 0;
	
	/// The "##docview"-class will override this...
	
	virtual void UpdateAllGraphicsViews(bool = false) { }
};

/*################################################################################################*/

/**	The "##extended" models can handle multiple "##conformations", or multiple 
	sets of coordinates.
*/

class model_extended : virtual public model_simple
{
	protected:
	
	vector<crd_set> cs_vector;
	
	// this is needed for access to cs_vector...
	// is there any easier way?!?!?! a read-access function perhaps???
	// usually only "default" set is needed, and rest is ignored...
	friend class mm1_sequencebuilder;
	
	public:
	
	model_extended(ostream *, class_factory &);
	virtual ~model_extended(void);
	
	void UpdateAccumValues(void);
	
	/**	This will return the number of coordinate sets. 
		It is supposed that at least one coordinate set exists all the time!!!
	*/
	
	i32u GetCRDSetCount(void);
	
	virtual void PushCRDSets(i32u) = 0;
	virtual void PopCRDSets(i32u) = 0;
	
	virtual void CopyCRDSet(i32u, i32u) = 0;
	virtual void SwapCRDSets(i32u, i32u) = 0;
	
	virtual void CenterCRDSet(i32u) = 0;
	virtual void ReserveCRDSets(i32u) = 0;
};

/*################################################################################################*/

/**	A coordinate-set class. 
	We will not implement data storage here, because different models 
	might have different requitements...
*/

class crd_set
{
	protected:
	
	char * description;
	
// bring also the coloring information here??? -> would allow different colors for different crdsets!!!
// bring also the coloring information here??? -> would allow different colors for different crdsets!!!
// bring also the coloring information here??? -> would allow different colors for different crdsets!!!

	public:
	
// these are tricky to protect, since these are still developing and are not yet used much.
// so, these are public as long as some reasonable strategy is found...

	fGL accum_weight;
	fGL accum_value;
	bool visible;
	
	public:
	
	crd_set(void);
	crd_set(const crd_set &);
	~crd_set(void);
	
	void SetDescription(const char *);
};

/*################################################################################################*/

/// This class can be used to superimpose coordinate sets.

class superimpose : public conjugate_gradient
{
	protected:
	
	i32s index[2];
	i32s counter;
	
	f64 value;
	f64 rot[3]; f64 drot[3];
	f64 loc[3]; f64 dloc[3];
	
	public:
	
	superimpose(i32s, i32s);
	virtual ~superimpose(void);
	
	f64 GetRMS(void) { return sqrt(value / (f64) counter); }
	void Compare(const f64 *, const f64 *, bool, f64 * = NULL);
	
	virtual void Transform(void) = 0;
};

/*################################################################################################*/

#endif	// MODEL_H

// eof
