// QM1MDL.CPP

// Copyright (C) 2000 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "qm1mdl.h"	// config.h is here -> we get ENABLE-macros here...

#include "mm1mdl.h"	// for import...

#include "qm1e_mopac.h"
#include "qm1e_mpqc.h"

#include <fstream>
#include <strstream>
using namespace std;

/*################################################################################################*/

qm1_atom::qm1_atom(void)
{
	selected = false;
}

qm1_atom::qm1_atom(element p1, fGL * p2, i32u p3)
{
	el = p1;
	
	fGL_a3 data = { p2[0], p2[1], p2[2] };
	for (i32u n1 = 0;n1 < p3;n1++) crd_vector.push_back(data);
	
	selected = false;
}

qm1_atom::qm1_atom(const qm1_atom & p1)
{
	el = p1.el;
	crd_vector = p1.crd_vector;
	
	selected = p1.selected;
}

qm1_atom::~qm1_atom(void)
{
}

// for qm1_atoms, equality is tested using pointers -> you can find the
// iterator (using STL's find-function) if the pointer is known.

bool qm1_atom::operator==(const qm1_atom & p1) const
{
	return (this == (& p1));
}

/*################################################################################################*/

qm1_bond::qm1_bond(void)
{
	atmr[0] = atmr[1] = NULL;
}

qm1_bond::qm1_bond(qm1_atom * p1, qm1_atom * p2, bondtype p3)
{
	atmr[0] = p1; atmr[1] = p2; bt = p3;
}

qm1_bond::qm1_bond(const qm1_bond & p1)
{
	atmr[0] = p1.atmr[0]; atmr[1] = p1.atmr[1]; bt = p1.bt;
}

qm1_bond::~qm1_bond(void)
{
}

// for bonds, equality is tested using atom pointers. if you need to find a certain bond,
// just make a temporary mm1_bond and use that to find the original one (using STL's find-function).

bool qm1_bond::operator==(const qm1_bond & p1) const
{
	if (atmr[0] == p1.atmr[0] && atmr[1] == p1.atmr[1]) return true;
	if (atmr[0] == p1.atmr[1] && atmr[1] == p1.atmr[0]) return true;
	
	return false;
}

/*################################################################################################*/

qm1_mdl::qm1_mdl(ostream * p1, class_factory & p2) : model_extended(p1, p2), model_simple(p1, p2)
{
	current_eng = NULL;
	
	total_charge = 0;
	current_orbital = 0;
	
	default_eng = 0;
}

qm1_mdl::~qm1_mdl(void)
{
	if (current_eng != NULL) delete current_eng;
}

const char * qm1_mdl::GetProjectFileNameExtension(void)
{
	static const char ext[] = "qm1gp";
	return ext;
}

bool qm1_mdl::CheckEngSettings(void)
{
	i32s electron_count = 0;
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		electron_count += (* it1).el.GetAtomicNumber();
	}
	
	// calculate the total number of electrons in the system.
	// calculate the total number of electrons in the system.
	// calculate the total number of electrons in the system.
	
	i32s total_electron_count = electron_count - total_charge;
	
	// checks start now...
	// checks start now...
	// checks start now...
	
	if (total_electron_count < 1)
	{
		err->ErrorMessage("Less than one electron in the system!\nPlease check the \"total charge\" setting.");
		return false;
	}
	
	if (total_electron_count % 2)
	{
		err->ErrorMessage("Odd number of electrons in the system!\nOnly singlet states with an even number\nof electrons are supported at the moment.\nPlease check the \"total charge\" setting.");
		return false;
	}
	
	// passed...
	// passed...
	// passed...
	
	return true;
}

/*##############################################*/
/*##############################################*/

const char * qm1_mdl::engtab1[] =
{
	"qm1_eng_mopac : MOPAC7 / MNDO",
	"qm1_eng_mopac : MOPAC7 / MINDO/3",
	"qm1_eng_mopac : MOPAC7 / AM1",
	"qm1_eng_mopac : MOPAC7 / PM3",
	
#ifdef ENABLE_MPQC

	"qm1_eng_mpqc : MPQC / STO-3G",
	"qm1_eng_mpqc : MPQC / STO-6G",
	"qm1_eng_mpqc : MPQC / 3-21G",
	"qm1_eng_mpqc : MPQC / 3-21G*",
	"qm1_eng_mpqc : MPQC / 4-31G",
	"qm1_eng_mpqc : MPQC / 4-31G*",
	"qm1_eng_mpqc : MPQC / 4-31G**",
	"qm1_eng_mpqc : MPQC / 6-31G",
	"qm1_eng_mpqc : MPQC / 6-31G*",
	"qm1_eng_mpqc : MPQC / 6-31G**",
	
#endif	// ENABLE_MPQC

	NULL
};

const i32s qm1_mdl::engtab2[] =
{
	(ENG_QM1_MOPAC | MOPAC_MNDO),
	(ENG_QM1_MOPAC | MOPAC_MINDO3),
	(ENG_QM1_MOPAC | MOPAC_AM1),
	(ENG_QM1_MOPAC | MOPAC_PM3),
	
#ifdef ENABLE_MPQC

	(ENG_QM1_MPQC | MPQC_STO3G),
	(ENG_QM1_MPQC | MPQC_STO6G),
	(ENG_QM1_MPQC | MPQC_3_21G),
	(ENG_QM1_MPQC | MPQC_3_21GS),
	(ENG_QM1_MPQC | MPQC_4_31G),
	(ENG_QM1_MPQC | MPQC_4_31GS),
	(ENG_QM1_MPQC | MPQC_4_31GSS),
	(ENG_QM1_MPQC | MPQC_6_31G),
	(ENG_QM1_MPQC | MPQC_6_31GS),
	(ENG_QM1_MPQC | MPQC_6_31GSS),
	
#endif	// ENABLE_MPQC

	NOT_DEFINED
};

qm1_eng * qm1_mdl::CreateDefaultEngine(void)
{
	// perform a sanity check of settings, and return NULL if failed. the user will be notified...
	// perform a sanity check of settings, and return NULL if failed. the user will be notified...
	// perform a sanity check of settings, and return NULL if failed. the user will be notified...
	
	if (CheckEngSettings() == false) return NULL;
	
	i32s engtype1 = (engtab2[default_eng] & 0xff00);
	i32s engtype2 = (engtab2[default_eng] & 0x00ff);
	
	switch (engtype1)
	{

#ifdef ENABLE_MPQC

		case ENG_QM1_MPQC:
		return new qm1_eng_mpqc(* this, engtype2);
		
#endif	// ENABLE_MPQC

		default:	// ENG_QM1_MOPAC
		if (qm1_eng_mopac::GetLock() != NULL)
		{
			err->ErrorMessage("MOPAC lock failed!!!\nCan't run multiple MOPAC calculations.");
			return NULL;
		}
		else return new qm1_eng_mopac(* this, engtype2);
	}
}

void qm1_mdl::DiscardCurrentEng(void)
{
	if (current_eng != NULL)
	{
		delete current_eng;
		current_eng = NULL;
	}
}

void qm1_mdl::SetupPlotting(void)
{
	if (current_eng != NULL) current_eng->SetupPlotting();
}

/*##############################################*/
/*##############################################*/

// this stuff is pretty much stolen from the mm1_mdl...
// this stuff is pretty much stolen from the mm1_mdl...
// this stuff is pretty much stolen from the mm1_mdl...

void qm1_mdl::PushCRDSets(i32u p1)
{
//cs_vector.resize(cs_vector.size() + p1);				// more convenient way?!?!?!
//for (i32u n1 = 0;n1 < p1;n1++) cs_vector.push_back();			// more crash-proof way?!?!?!?
	for (i32u n1 = 0;n1 < p1;n1++) cs_vector.push_back(crd_set());	// or even this way?!?!?!?
	
	fGL_a3 newcrd = { 0.0, 0.0, 0.0 };
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32u n1 = 0;n1 < p1;n1++)
		{
			(* it1).crd_vector.push_back(newcrd);
		}
	}
}

void qm1_mdl::PopCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void qm1_mdl::CopyCRDSet(i32u, i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void qm1_mdl::SwapCRDSets(i32u, i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void qm1_mdl::CenterCRDSet(i32u p1)
{
	fGL sum[3] = { 0.0, 0.0, 0.0 };
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++) sum[n1] += (* it1).crd_vector[p1][n1];
	}
	
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			fGL tmp1 = sum[n1] / (f64) atom_list.size();
			(* it1).crd_vector[p1][n1] -= tmp1;
		}
	}
}

void qm1_mdl::ReserveCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void qm1_mdl::AddAtom(qm1_atom & p1)
{
	DiscardCurrentEng();
	atom_list.push_back(p1);
}

void qm1_mdl::RemoveAtom(iter_qm1al it1)
{
	DiscardCurrentEng();
	
	// before removing an atom, also remove all bonds that relate to this atom.
	// if a bond is removed, the search is restarted; not very efficient but should not be a bottleneck...
	
	iter_qm1bl it2 = bond_list.begin();
	while (it2 != bond_list.end())
	{
		bool must_be_removed = false;
		if ((* it2).atmr[0] == & (* it1)) must_be_removed = true;
		if ((* it2).atmr[1] == & (* it1)) must_be_removed = true;
		
		if (must_be_removed)
		{
			bond_list.erase(it2);
			it2 = bond_list.begin();	// restart!!!
		}
		else it2++;
	}
	
	atom_list.erase(it1);
}

// also this seems to relate all_atoms_interface...
// also this seems to relate all_atoms_interface...
// also this seems to relate all_atoms_interface...

void qm1_mdl::ImportMM1(const char * fn)
{
	mm1_mdl * mdl = new mm1_mdl(NULL, (* factory));
	
	ifstream ifile(fn, ios::in);
	mdl->ReadStream(ifile);
	ifile.close();

	vector<mm1_atom *> mmtab;	// store the pointers here...
	vector<qm1_atom *> qmtab;	// ...to replicate the bonds!
		
	iter_mm1al it1 = mdl->atom_list.begin();
	while (it1 != mdl->atom_list.end())
	  {
	    element tmp1 = (* it1).el;
	    fGL * tmp2 = (* it1).crd_vector[0].data;
			
	    qm1_atom newatom(tmp1, tmp2, 1);
	    AddAtom(newatom);
			
	    mmtab.push_back(& (* it1));
	    qmtab.push_back(& atom_list.back());
			
	    it1++;
	  }
		
	iter_mm1bl it2 = mdl->bond_list.begin();
	while (it2 != mdl->bond_list.end())
	  {
	    i32u index1 = 0; while (index1 < mmtab.size()) if ((* it2).atmr[0] == mmtab[index1]) break; else index1++;
	    if (index1 == mmtab.size()) { cout << "FATAL ERROR : could not find atom #1." << endl; exit(EXIT_FAILURE); }
			
	    i32u index2 = 0; while (index2 < mmtab.size()) if ((* it2).atmr[1] == mmtab[index2]) break; else index2++;
	    if (index2 == mmtab.size()) { cout << "FATAL ERROR : could not find atom #2." << endl; exit(EXIT_FAILURE); }
			
	    bondtype tmp1 = (* it2).bt;
			
	    qm1_bond newbond(qmtab[index1], qmtab[index2], tmp1);
	    bond_list.push_back(newbond);
			
	    it2++;
	  }
	
	delete mdl;
}

void qm1_mdl::DoEnergy(void)
{
	if (!current_eng) current_eng = CreateDefaultEngine();
	if (!current_eng) return;	// sanity check failed...
	
	CopyCRD(this, current_eng, 0); current_eng->Compute(0);
	
	char buffer[128];
	ostrstream str(buffer, sizeof(buffer)); str.setf(ios::fixed); str.precision(8);
	
	str << "Energy = " << current_eng->energy << " kJ/mol" << ends;
	
	err->Message(buffer);
	
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	
	SetupPlotting();
}

void qm1_mdl::DoGeomOpt(qm1_geomopt_param & param)
{
	GeomOptGetParam(param);
	if (!param.confirm) return;
	
	if (!current_eng) current_eng = CreateDefaultEngine();
	if (!current_eng) return;	// sanity check failed...
	
	CopyCRD(this, current_eng, 0);
	
	qm1_geomopt * opt = new qm1_geomopt(current_eng, 20, 0.0125);	// optimal settings?!?!?
	
	char buffer[1024];
	for (i32s n1 = 0;n1 < param.nsteps;n1++)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		sprintf(buffer, "%d %10.4f  %4.6e", n1, opt->optval, opt->optstp);
		cout << buffer << endl;
		
		if (!(n1 % 10))
		{
			CopyCRD(current_eng, this, 0); CenterCRDSet(0);
			UpdateAllGraphicsViews(true);
		}
	}
	
	delete opt;
	
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	
	SetupPlotting();
}

void qm1_mdl::GeomOptGetParam(qm1_geomopt_param & param)
{
	param.confirm = true;
}

/*################################################################################################*/

// eof
