// QM1MDL.H : model class for molecular QM models...

// Copyright (C) 2000 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef QM1MDL_H
#define QM1MDL_H

class qm1_mdl;

/*################################################################################################*/

#include "model.h"

#include "qm1eng.h"
#include "qm1alg.h"

#include "utility.h"

#include <list>
#include <vector>
using namespace std;

/*################################################################################################*/

class qm1_atom
{
//	protected:
	public:		// heavily used...
	
	element el;
	vector<a3_fGL> crd_vector;
	
	bool selected;
	
	public:
	
	qm1_atom(void);
	qm1_atom(element, fGL *, i32u);
	qm1_atom(const qm1_atom &);
	~qm1_atom(void);
	
	bool operator==(const qm1_atom &) const;	// using pointers...
};

// the bonds are not really needed in QM models, but we need to store the bond information also
// in QM models if we wish to do conversions MM<->QM models. we might also render the bonds,
// since many users might expect that?!?!?!?

class qm1_bond
{
//	protected:
	public:		// heavily used...
	
	qm1_atom * atmr[2];
	bondtype bt;
	
	public:
	
	qm1_bond(void);
	qm1_bond(qm1_atom *, qm1_atom *, bondtype);
	qm1_bond(const qm1_bond &);
	~qm1_bond(void);
	
	bool operator==(const qm1_bond &) const;	// using pointers...
};

typedef list<qm1_atom>::iterator iter_qm1al;		// al = atom list
typedef list<qm1_bond>::iterator iter_qm1bl;		// bl = bond list

/*################################################################################################*/

// 0xff00 determines the engine class...

#define ENG_QM1_MOPAC		0x100		// qm1_eng_mopac
#define ENG_QM1_MPQC		0x200		// qm1_eng_mpqc

// 0xff00 determines the hamiltonian/basis set...

#define MOPAC_MNDO		0x01
#define MOPAC_MINDO3		0x02
#define MOPAC_AM1		0x03
#define MOPAC_PM3		0x04

#define MPQC_STO3G		0x01
#define MPQC_STO6G		0x02
#define MPQC_3_21G		0x03
#define MPQC_3_21GS		0x04
#define MPQC_4_31G		0x05
#define MPQC_4_31GS		0x06
#define MPQC_4_31GSS		0x07
#define MPQC_6_31G		0x08
#define MPQC_6_31GS		0x09
#define MPQC_6_31GSS		0x0a

/**	A "##model" class for molecular quantum mechanics models.
	
	My hope is that a simple set of QM methods would be implemented in this package, 
	for educational purposes if nothing else... But it's a very VERY demanding job, 
	which will take at least a decade, or most likely forever, to complete.
	
	So, now we have to borrow that code outside. this design is mostly an adaption 
	to the MPQC program, where we currently borrow our QM energy/gradient code...
*/

class qm1_mdl : public model_extended
{
	protected:
	
	list<qm1_atom> atom_list;
	list<qm1_bond> bond_list;
	
	qm1_eng * current_eng;
	
	i32s total_charge;
	i32s current_orbital;
	
// default engine settings begin!!!
// default engine settings begin!!!
// default engine settings begin!!!
	
	public:
	
	static const char * engtab1[];
	static const i32s engtab2[];
	
	i32s default_eng;
	
// default engine settings end!!!
// default engine settings end!!!
// default engine settings end!!!
	
	friend class qm1_eng;
	friend class qm1_eng_mopac;
	friend class qm1_eng_mpqc;
	
	friend class qm1_geomopt;
	
	friend class enlevdiag_view;
	
	friend class gnome_mm1_docv;	// temporary, for MM->QM conversion, will move into all_atoms_interface???
	
	friend void CopyCRD(qm1_mdl *, qm1_eng *, i32u);
	friend void CopyCRD(qm1_eng *, qm1_mdl *, i32u);
	
	public:
	
	qm1_mdl(ostream *, class_factory &);
	~qm1_mdl(void);
	
	const char * GetProjectFileNameExtension(void);		// virtual
	
	/// This is used for settings sanity-check; for example number of electrons vs. spin state etc...
	bool CheckEngSettings(void);
	
	qm1_eng * CreateDefaultEngine(void);
	virtual void DiscardCurrentEng(void);
	virtual void SetupPlotting(void);
	
	bool ReadStream(istream &) { return false; }	// virtual
	void WriteStream(ostream &) { }			// virtual
	
	void PushCRDSets(i32u);			// virtual
	void PopCRDSets(i32u);			// virtual
	
	void CopyCRDSet(i32u, i32u);		// virtual
	void SwapCRDSets(i32u, i32u);		// virtual
	
	void CenterCRDSet(i32u);		// virtual
	void ReserveCRDSets(i32u);		// virtual
	
	void AddAtom(qm1_atom &);
	void RemoveAtom(iter_qm1al);
	
	void ImportMM1(const char *);
	
	// these are for glade callbacks etc...
	
	i32s GetTotalCharge(void) { return total_charge; }
	void SetTotalCharge(i32s p1) { total_charge = p1; }
	
	public:
	
	// here we have a set of Do???()-functions. the idea is that there is a set
	// of features that we wish to behave EXACTLY same way in each target/platform.
	// we then create a Do???()-function for the feature, and hide the details of
	// the user interface in a set of virtual functions.

	/// This will perform an energy calculation.
	void DoEnergy(void);
	
	/// This will perform geometry optimization.
	void DoGeomOpt(qm1_geomopt_param &);
	
	/// This is used to ask user the GO options; this default function will just silently accept the defaults.
	virtual void GeomOptGetParam(qm1_geomopt_param &);
};

/*################################################################################################*/

#endif	// QM1MDL_H

// eof
