// VIEWS.H : base classes for "views", to be used with the "docview" class.

// Copyright (C) 1998 Tommi Hassinen, Jarno Huuskonen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef VIEWS_H
#define VIEWS_H

#ifdef ENABLE_GRAPHICS

class view;
class ogl_view;

class graphics_view;

class enlevdiag_view;

class plotting_view;
class plot2d_view;
class plot3d_view;

class treelist_view;

class graphics_class_factory;

/*################################################################################################*/

class camera;		// camera.h

class color_mode;	// model.h
class docview;		// docview.h

#include "mtools.h"
#include "factory.h"
#include "qm1mdl.h"

/*################################################################################################*/

/**	The "##view" is a base class for all views. All things dependent on operating 
	system and/or windowing environment are to be wrapped in ctor/dtor or in a set of 
	pure virtual functions.
	
	The "##view"-objects should not be created directly, but a "##view_factory" object 
	should create them instead. Different platforms/operating systems might have different 
	and tricky requirements on how windows should be created and initialized there. It's 
	up to "##view_factory" and/or "##view" itself to meet these requirements, using 
	specific callbacks or whatever is necessary...
*/

class view
{
	protected:
	
	// no relevant common data!??!!?
	// no relevant common data!??!!?
	// no relevant common data!??!!?
	
	public:
	
	view(void);
	virtual ~view(void);
	
	virtual docview * GetDV(void) = 0;
	
	/**	This should update the visible contents of this view. 
		The parameter is a "mode" of update, either "##immediate" (true) or 
		"##as soon as possible" (false, is default).
	*/
	
	virtual void Update(bool = false) = 0;
	
	/// This should set the window title text for this view.
	
	virtual void SetTitle(const char *) = 0;
};

/*################################################################################################*/

/**	The "##ogl_view" is a view that will use OpenGL graphics to present it's contents, 
	and is (optionally) connected to the standard set of mouse tool classes.
*/

class ogl_view : virtual public view
{
	protected:
	
	i32s size[2];	// size of the viewing window, in pixels.
	
	// static data about mouse/keyboard state...
	
	static mouse_tool::state state;
	static mouse_tool::button button;
	
	static bool shift_down;
	static bool ctrl_down;
	
	// all available mouse tools are here...
	
	static draw_tool tool_draw;
	static erase_tool tool_erase;
	
	static select_tool tool_select;
	static zoom_tool tool_zoom;
	
	static translate_xy_tool tool_translate_xy;
	static translate_z_tool tool_translate_z;
	
	static orbit_xy_tool tool_orbit_xy;
	static orbit_z_tool tool_orbit_z;
	
	static rotate_xy_tool tool_rotate_xy;
	static rotate_z_tool tool_rotate_z;

	static measure_tool tool_measure;
	
	friend class camera;
	friend class gnome_mdi_app;
	
	public:
	
	// pointer to the currently selected mouse tool.
	// OBSOLETE?!?!? SOMETHING ELSE IS NEEDED FOR THIS SOON!!!
	// OBSOLETE?!?!? SOMETHING ELSE IS NEEDED FOR THIS SOON!!!
	// OBSOLETE?!?!? SOMETHING ELSE IS NEEDED FOR THIS SOON!!!
	
	static mouse_tool * current_tool;
	
	public:
	
	ogl_view(void);
	virtual ~ogl_view(void);
	
	void SetSize(i32s, i32s);
	
	void WriteBitmapString(void *, const char *);		// for glut only !??!!?
	i32s GetBitmapStringWidth(void *, const char *);	// for glut only !??!!?
	
	/**	This should do all what is necessary to make the OpenGL rendering 
		context of this view "##current", so that the following OpenGL 
		commands would affect this view.
	*/
	
	virtual void SetCurrent(void) = 0;
	
	/// This should contain the necessary OpenGL initialization commands.
	
	virtual void InitGL(void) = 0;
	
	/// This should contain the OpenGL commands that create the visible contents of this view.
	
	virtual void Render(void) = 0;
};

/*################################################################################################*/

/// "##graphics_view" presents the molecular models in OpenGL 3D graphics.

class graphics_view : virtual public ogl_view
{
	protected:
	
	camera * cam;
	
	fGL range[2];	// dimensions of the view, in nanometers.
	
	i32s render; i32s label;
	color_mode * colormode;
	
	bool enable_fog;
	bool accumulate;
	
	static bool quick_update;	// some static flags...
	static bool draw_info;		// some static flags...
	
	friend class draw_tool;
	friend class erase_tool;
	friend class select_tool;
	friend class zoom_tool;
	friend class translate_xy_tool;
	friend class translate_z_tool;
	friend class orbit_xy_tool;
	friend class orbit_z_tool;
	friend class rotate_xy_tool;
	friend class rotate_z_tool;
	friend class measure_tool;
	
	friend class camera;
	
	friend class docview;
	
	friend class qm1_docv;
	friend class glut_qm1_docv;
	
	friend class qm2_docv;
	friend class glut_qm2_docv;
	
	friend class mm1_docv;
	friend class glut_mm1_docv;
	
	friend class mm2_docv;
	friend class glut_mm2_docv;
	
	public:
	
	graphics_view(camera *);
	virtual ~graphics_view(void);
	
	void GetCRD(i32s *, fGL *);
	
	void InitGL(void);	// virtual
	void Render(void);	// virtual
};

/*################################################################################################*/

/// enlevdiag_view is an energy level diagram for QM models.

class enlevdiag_view : virtual public ogl_view
{
	protected:
	
	qm1_mdl * mdl;
	
	fGL center;
	fGL scale;
	
	iGLu * select_buffer;
	
	friend class zoom_tool;
	friend class translate_xy_tool;
	
	public:
	
	enlevdiag_view(qm1_mdl *);
	~enlevdiag_view(void);
	
	void SetCenterAndScale(void);
	
	void InitGL(void);	// virtual
	void Render(void);	// virtual
};

/*################################################################################################*/

/**	"##plotting_view" is reserved for 2d and 3d plotting...
	
	The future of these views is uncertain. Would it be better to develop some 
	kind of spreadsheet program interaction that these views !?!?!?!?
*/

class plotting_view : virtual public ogl_view
{
	public:
	
	plotting_view(void);
	virtual ~plotting_view(void);
};

/*################################################################################################*/

class plot2d_view : public plotting_view
{
	public:
	
	plot2d_view(void);
	virtual ~plot2d_view(void);
};

/*################################################################################################*/

class plot3d_view : public plotting_view
{
	public:
	
	plot3d_view(void);
	virtual ~plot3d_view(void);
};

/*################################################################################################*/

#ifdef ENABLE_TREELIST_VIEW

/// "##treelist_view" is made optional, since some complicated system services are needed...

class treelist_view : virtual public view
{
	public:
	
	treelist_view(void);
	virtual ~treelist_view(void);
	
	virtual void LightAdded(light *) = 0;
	virtual void LightRemoved(light *) = 0;
	
	virtual void ObjectAdded(smart_object *) = 0;
	virtual void ObjectRemoved(smart_object *) = 0;
};

#endif	// ENABLE_TREELIST_VIEW

/*################################################################################################*/

/**	"##graphics_class_factory" is a graphics_class capable to create also "view" objects.

	For each different platform (just like a "##view") a "##graphics_class_factory" should
	be made which produces new "view" objects for this particular platform...
*/

class graphics_class_factory : public class_factory
{
	protected:
	
	public:
	
	graphics_class_factory(void) : class_factory() { };
	virtual ~graphics_class_factory(void) { };
	
	virtual graphics_view * ProduceGraphicsView(docview *, camera *) = 0;
	virtual enlevdiag_view * ProduceEnLevDiagView(qm1_mdl *) = 0;
	
#ifdef ENABLE_TREELIST_VIEW

	virtual treelist_view * ProduceTreeListView(docview *) = 0;
	
#endif	// ENABLE_TREELIST_VIEW

};

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS
#endif	// VIEWS_H

// eof
