// Copyright 1993, U.C.S.F. Computer Graphics Laboratory
// $Id: genclass.cc,v 2.17 1994/09/09 20:57:44 gregc Exp $

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "mcd.h"
#include "clex.h"
#include "String.LibClassesPtr.VHMap.h"
#include "String.VHSet.h"

SLList<String>	compileFiles;

const unsigned int	FUNC_EQ = 0x1, FUNC_LE = 0x2, FUNC_CMP = 0x4,
			FUNC_HASH = 0x8, SLOW_LENGTH = 0x10,
			VECTOR_CLASS = 0x20, ABSTRACT_CLASS = 0x40,
			TEMPLATE = 0x80, MAP_CLASS = 0x100;

// TODO: StringStrSetMap	all_iter_defs; key is itertype, value is set
StringVHSet		all_iter_defs;

struct LibClasses {
	String		class_name;
	const char	*gen_classes;
	unsigned int	flags;
	const char	*add_element;
	const char	*del_element;
	const char	*iter_type;
	StringVHSet	iter_defs;
		LibClasses(): class_name(), gen_classes(0), flags(0),
			add_element(0), del_element(0), iter_type(0),
			iter_defs() {}
};

static LibClasses	*libInfo[MAX_MOLCLASS + 1][MAX_MOLCLASS + 1];

static void	genClass(int for_class, int in_class);
static void	genPtrClass(int for_class, int in_class);
static void	parse_error(Clex *);

static StringLibClassesPtrVHMap	info((LibClasses *) 0);

static Regex	mapSuffix("Map$");	// for genClass and genPtrClass

static void
parse_classes(const char *filename)
{
	Clex lex(filename);
	lex.errorFunction(parse_error);
	for (lex.nextToken(); lex.token() != Clex::EOI; lex.nextToken()) {
		LibClasses	*l;

		if (lex.token() != Clex::CLASS) {
			parse_error(&lex);
			return;
		}
		lex.nextToken(Clex::IDENT);
		l = new LibClasses;
		l->class_name = lex.value();
		info[l->class_name] = l;
		lex.nextToken(Clex::LEFT_CURLY);
		for (lex.nextToken(); lex.token() != Clex::EOI; lex.nextToken()) {
			switch (lex.token()) {
			default:
				parse_error(&lex);
				return;
			case Clex::RIGHT_CURLY:
				goto next_class;
			case Clex::GENCLASS:
				lex.nextToken();
				if (lex.token() != Clex::IDENT
				&& lex.token() != Clex::STRING) {
					parse_error(&lex);
					return;
				}
				l->gen_classes = strdup(lex.value().chars());
				// assert(!template)
				// assert(class_name in gen_classes)
				break;
			case Clex::ADD_ELEMENT:
				lex.nextToken();
				if (lex.token() != Clex::IDENT
				&& lex.token() != Clex::STRING) {
					parse_error(&lex);
					return;
				}
				l->add_element = strdup(lex.value().chars());
				break;
			case Clex::DEL_ELEMENT:
				lex.nextToken();
				if (lex.token() != Clex::IDENT
				&& lex.token() != Clex::STRING) {
					parse_error(&lex);
					return;
				}
				l->del_element = strdup(lex.value().chars());
				break;
			case Clex::ITERTYPE:
				lex.nextToken();
				if (lex.token() != Clex::IDENT
				&& lex.token() != Clex::STRING) {
					parse_error(&lex);
					return;
				}
				l->iter_type = strdup(lex.value().chars());
assert(lex.value() == "Pix");
				break;
			case Clex::ITERDEF:
				lex.nextToken();
				if (lex.token() != Clex::IDENT
				&& lex.token() != Clex::STRING) {
					parse_error(&lex);
					return;
				}
				l->iter_defs.add(lex.value());
				if (l->iter_type == NULL) {
					// error message
					return;
				}
				// TODO: all_iter_defs[l->iter_type].add ...
				all_iter_defs.add(lex.value());
				break;
			case Clex::TEMPLATE: l->flags |= TEMPLATE; break;
			case Clex::ABSTRACT: l->flags |= ABSTRACT_CLASS; break;
			case Clex::MAP: l->flags |= MAP_CLASS; break;
			case Clex::VECTOR: l->flags |= VECTOR_CLASS; break;
			case Clex::SLOW_LENGTH: l->flags |= SLOW_LENGTH; break;
			case Clex::EQ: l->flags |= FUNC_EQ; break;
			case Clex::LE: l->flags |= FUNC_LE; break;
			case Clex::CMP: l->flags |= FUNC_CMP; break;
			case Clex::HASH: l->flags |= FUNC_HASH; break;
			}
		}
	next_class:;
	}
}

static void
parse_error(Clex *context)
{
	fprintf(stderr,
		"Unexpected %s on line %d of %s, skipping rest of file\n",
		context->token() == Clex::IDENT ? context->value()
					: Clex::tokenNames[context->token()],
		context->lineNum(), context->fileName().chars());
}

void
initialize_class_information()
{
	String filename(OTFHOME);
	filename += "/lib/genmol.classes";
	parse_classes(filename.chars());
	if (access("genmol.classes", R_OK) == 0)
		parse_classes("genmol.classes");

	LibClasses *l = new LibClasses;
	l->class_name = "_array";
	l->iter_type = "_array";
	info[l->class_name] = l;
}

static LibClasses *
findLibClasses(const String &class_name)
{
	Pix i = info.seek(class_name);
	if (i == 0)
		return NULL;
	return info.contents(i);
}

String
class_operators(int for_class, const char *base)
{
	String	result = "";
	int flags = 0;
	for (int in_class = 0; in_class <= MAX_MOLCLASS; in_class += 1)
		if (libInfo[for_class][in_class])
			flags |= libInfo[for_class][in_class]->flags;
	if (flags & (FUNC_EQ|FUNC_CMP))
		result =
			String("\tint operator==(const ")
				+ molClassNames[for_class] + " &r) const\n"
			"\t{\n"
			"\t\treturn " + base + "::operator==(r);\n"
			"\t}\n";
	if (flags & (FUNC_LE|FUNC_CMP))
		result += 
			String("\tint operator<=(const ")
				+ molClassNames[for_class] + " &r) const\n"
			"\t{\n"
			"\t\treturn " + base + "::operator<=(r);\n"
			"\t}\n";
	return result;
}

static String
expand(const String &arg, int indent, int for_class, int in_class, const char *container)
{
	static Regex	token("CONTAINER\\|TYPE\\|KEY\\|ORIGFOR\\|FOR\\|SIZE\\|\n");
	String		start(arg);
	String		result;

	if (start == "")
		return start;

	if (container == NULL)
		container = use[for_class][in_class].instance;

	for (int i = 0; i < indent; i += 1)
		result += '\t';
	for (;;) {
		if (!start.contains(token)) {
			result += start;
			break;
		}
		String prefix(start.before(token));
		start.del(prefix);
		result += prefix;
		switch (start[0]) {
		case 'C':
			start.del("CONTAINER");
			result += container;
			break;
		case 'T':
			start.del("TYPE");
			if (isTemplate(use[for_class][in_class].name)) {
				result += use[for_class][in_class].name + '<'
					+ molClassNames[for_class];
				if (isPtr(for_class))
					result += "Ptr";
				result += '>';
				break;
			}
			if (isMap(use[for_class][in_class].name))
				result += use[for_class][in_class].keyType;
			result += molClassNames[for_class];
			if (isPtr(for_class))
				result += "Ptr";
			if (use[for_class][in_class].name.contains("_array", 0))
				result += "Array";
			else
				result += use[for_class][in_class].name;
			break;
		case 'K':
			start.del("KEY");
			result += use[for_class][in_class].keyMember;
			break;
		case 'O':
			start.del("ORIGFOR");
			result += origClassNames[for_class];
			break;
		case 'F':
			start.del("FOR");
			result += molClassNames[for_class];
			break;
		case 'S':
			start.del("SIZE");
			// extract size from container name
			if (use[for_class][in_class].name.contains("_array", 0))
				result += use[for_class][in_class].name.from(6);
			else
				result += "UnKnOwN";
			break;
		case '\n':
			start.del("\n");
			result += '\n';
			for (int i = 0; i < indent; i += 1)
				result += '\t';
			break;
		}
	}
	return result;
}

String
expand_add_element(int indent, int for_class, int in_class, const char *container)
{
	return expand(libInfo[for_class][in_class]->add_element, indent,
						for_class, in_class, container);
}

String
expand_del_element(int indent, int for_class, int in_class, const char *container)
{
	return expand(libInfo[for_class][in_class]->del_element, indent,
						for_class, in_class, container);
}

String
expand_iterator(int for_class, int in_class)
{
	char	buf[BUFSIZ];

	int isArray = use[for_class][in_class].name.contains("_array", 0);
	if (!isArray && libInfo[for_class][in_class] == NULL)
		return "";

	String	tmp("unifdef");
	if (isPtr(for_class))
		tmp += " -D";
	else
		tmp += " -U";
	tmp += "PTRCLASS";
	if (isMap(use[for_class][in_class].name))
		tmp += " -D";
	else
		tmp += " -U";
	tmp += "MAPCLASS";
	// TODO: all_iter_defs[libinfo[].iter_type].first, etc.
	for (Pix i = all_iter_defs.first(); i != 0; all_iter_defs.next(i)) {
		String	*s = &all_iter_defs(i);
		if (libInfo[for_class][in_class]->iter_defs.contains(*s))
			tmp += " -D";
		else
			tmp += " -U";
		tmp += *s;
	}
	tmp += ' ';
	// TODO: find directory that the iterator "template" is in
	tmp += OTFHOME;
	tmp += "/lib/genmol.";
	tmp += libInfo[for_class][in_class]->iter_type;
	if (verbose)
		fprintf(stderr, "%s\n", tmp.chars());
	FILE *f = popen(tmp.chars(), "r");
	if (f == NULL)
		return "";
	tmp = "";
	while (fgets(buf, BUFSIZ, f) != NULL)
		tmp += buf;
	(void) pclose(f);
	return expand(tmp, 0, for_class, in_class, NULL);
}

static void
genIter(int for_class)
{
	char	buf[BUFSIZ];

	String filename(OTFHOME);
	filename += "/lib/genmol.moliter";
	FILE *f = fopen(filename.chars(), "r");
	String tmp = "";
	while (fgets(buf, BUFSIZ, f) != NULL)
		tmp += buf;
	(void) fclose(f);
	tmp = expand(tmp, 0, for_class, for_class, NULL);

	filename = molClassNames[for_class];
	filename += "Iter.cc";
	if ((f = fopen(filename.chars(), "w")) == NULL) {
		fprintf(stderr, "unable to open %s for writing\n",
							filename.chars());
		exit(1);
	}
	fwrite(tmp.chars(), 1, tmp.length(), f);
	if (fclose(f) == EOF) {
		fprintf(stderr, "fatal error writing %s.\n", filename.chars());
		exit(1);
	}
	compileFiles.append(filename);
}

void
genClasses(const char * /*output_name*/ )
{
	String	array("_array");
	for (int for_class = 0; for_class <= MAX_MOLCLASS; for_class += 1) {
		for (int in_class = 0; in_class <= MAX_MOLCLASS; in_class += 1){
			if (use[for_class][in_class].name == 0
			|| use[for_class][in_class].name == "MolGraph"
			|| use[for_class][in_class].name == "reference")
				continue;
			if (use[for_class][in_class].name.contains("_array", 0)) {
				libInfo[for_class][in_class]
							= findLibClasses(array);
				continue;
			}
			if (in_class != 0)
				for (int i = 0; i < in_class; i += 1)
					if (use[for_class][in_class].name
					== use[for_class][i].name) {
						libInfo[for_class][in_class]
							= libInfo[for_class][i];
						goto next_class;
					}
			if (isPtr(for_class))
				genPtrClass(for_class, in_class);
			else
				genClass(Coord, in_class);
		next_class:;
		}
		genIter(for_class);
	}
}

inline static void
printstr(FILE *f, const char *s)
{
	if (verbose)
		fprintf(stderr, "%s", s);
	fprintf(f, "%s", s);
}

static int
ed_file(const String &header_file, const String &what, const String &container)
{
	String	cmd;

	cmd = "ed - " + header_file;
	if (verbose)
		fprintf(stderr, "%s\n", cmd.chars());
	FILE *f = popen(cmd.chars(), "w");
	if (f == NULL) {
		fprintf(stderr, "unable to edit %s container's %s.\n",
					container.chars(), header_file.chars());
		return 1;
	}
	printstr(f, what.chars());
	printstr(f, "w\nq\n");
	if (pclose(f) == EOF) {
		fprintf(stderr, "unable to edit %s container's %s.\n",
					container.chars(), header_file.chars());
		return 1;
	}
	return 0;
}

static void
genClass(int for_class, int in_class)
{
	String	cmd, defsh;

	LibClasses *l = findLibClasses(use[for_class][in_class].name);
	if (l == NULL) {
		fprintf(stderr, "warning:  unable to find libg++ class '%s'.\n",
				use[for_class][in_class].name.chars());
		return;
	}
	if ((l->flags & ABSTRACT_CLASS) != 0)
		fprintf(stderr, "warning: %s is an abstract class\n",
					use[for_class][in_class].name.chars());
	libInfo[for_class][in_class] = l;

	defsh = String(molClassNames[for_class]) + ".defs.h";
	if (!isMap(use[for_class][in_class].name)
	&& access(defsh.chars(), R_OK) == -1) {
		// make defs.h
		cmd = String("genclass ") + molClassNames[for_class]
								+ " ref defs";
		if (verbose)
			fprintf(stderr, "%s\n", cmd.chars());
		if (system(cmd.chars()) != 0) {
			fprintf(stderr, "'%s' failed.\n", cmd.chars());
			return;
		}
		// edit defs.h to do comparisons on pointed to classes
		String ed_cmds = String("/^#define/+a\n#include \"")
				+ molClassNames[for_class] + ".h\"\n.\n";
		if (ed_file(defsh, ed_cmds, use[for_class][in_class].name))
			return;
	}

	// make other class files
	// tokenize l->gen_classes, and genclass (and compile) each in turn
	const int max_bases = 32;
	String bases[max_bases];
	int count = split(l->gen_classes, bases, max_bases, RXwhite);
	for (int i = 0; i < count; i += 1) {
		cmd = "genclass ";
		int is_map = bases[i].contains(mapSuffix);
		if (is_map)
			cmd += "-2 " + use[for_class][in_class].keyType
								+ " ref ";
		cmd = cmd + molClassNames[for_class] + " ref " + bases[i];
		if (verbose)
			fprintf(stderr, "%s\n", cmd.chars());
		if (system(cmd.chars()) != 0) {
			fprintf(stderr, "'%s' failed.\n", cmd.chars());
			return;
		}
		if (!is_map)
			cmd = "";
		else {
			cmd = use[for_class][in_class].keyType + '.';
			if (bases[i] == "Map") {
				String inc_file = cmd + "defs.h";
				String ed_cmds
					= String("/^#define/+a\n#include \"")
					+ molClassNames[for_class]
					+ ".h\"\n.\n";
				if (access(inc_file.chars(), R_OK) != 0)
					ed_cmds += '/' + inc_file
						+ "/s,,mol/" + inc_file + ",\n";
				ed_file(cmd + molClassNames[for_class] + '.'
							+ bases[i] + ".h",
					ed_cmds, use[for_class][in_class].name);
			}
		}
		cmd = cmd + molClassNames[for_class] + '.' + bases[i] + ".cc";
		compileFiles.append(cmd);
	}
}

static void
genPtrClass(int for_class, int in_class)
{
	String	cmd, defsh;

	LibClasses *l = findLibClasses(use[for_class][in_class].name);
	if (l == NULL) {
		fprintf(stderr, "warning:  unable to find libg++ class '%s'.\n",
				use[for_class][in_class].name.chars());
		return;
	}
	if ((l->flags & ABSTRACT_CLASS) != 0)
		fprintf(stderr, "warning: %s is an abstract class\n",
					use[for_class][in_class].name.chars());
	libInfo[for_class][in_class] = l;

	defsh = String(molClassNames[for_class]) + "Ptr.defs.h";
	if (!isMap(use[for_class][in_class].name)
	&& access(defsh.chars(), R_OK) == -1) {
		// make defs.h
		cmd = String("genclass ") + molClassNames[for_class]
							+ "Ptr val defs";
		if (verbose)
			fprintf(stderr, "%s\n", cmd.chars());
		if (system(cmd.chars()) != 0) {
			fprintf(stderr, "'%s' failed.\n", cmd.chars());
			return;
		}
		// edit defs.h to do comparisons on pointed to classes
		String ed_cmds = String("/^#define/+a\n#include \"")
			+ molClassNames[for_class] + ".h\"\ntypedef "
			+ molClassNames[for_class] + " *"
			+ molClassNames[for_class] + "Ptr;\n.\n";
		ed_cmds += "1,$s/(a)/*(a)/g\n";
		ed_cmds += "1,$s/(b)/*(b)/g\n";
		if (l->flags & FUNC_HASH) {
			ed_cmds += "/int hash(/s/(/(const /\ns/Ptr )/ \\&)/\n";
			ed_cmds += "/hash(x)/s//hash(*x)\n";
		}
		if (ed_file(defsh, ed_cmds, use[for_class][in_class].name))
			return;
	}

	// make other class files
	// tokenize l->gen_classes, and genclass (and compile) each in turn
	const int max_bases = 32;
	String bases[max_bases];
	int count = split(l->gen_classes, bases, max_bases, RXwhite);
	for (int i = 0; i < count; i += 1) {
		int is_map = bases[i].contains(mapSuffix);
		cmd = "genclass ";
		if (is_map)
			cmd += "-2 " + use[for_class][in_class].keyType
								+ " ref ";
		cmd = cmd + molClassNames[for_class] + "Ptr val " + bases[i];
		if (verbose)
			fprintf(stderr, "%s\n", cmd.chars());
		if (system(cmd.chars()) != 0) {
			fprintf(stderr, "'%s' failed.\n", cmd.chars());
			return;
		}
		if (!is_map)
			cmd = "";
		else {
			cmd = use[for_class][in_class].keyType + '.';
			if (bases[i] == "Map") {
				String inc_file = cmd + "defs.h";
				String ed_cmds
					= String("/^#define/+a\n#include \"")
						+ molClassNames[for_class]
						+ ".h\"\ntypedef "
						+ molClassNames[for_class]
						+ " *"
						+ molClassNames[for_class]
						+ "Ptr;\n.\n";
				if (access(inc_file.chars(), R_OK) != 0)
					ed_cmds += '/' + inc_file
						+ "/s,,mol/" + inc_file + ",\n";
				ed_file(cmd + molClassNames[for_class] + "Ptr."
							+ bases[i] + ".h",
					ed_cmds, use[for_class][in_class].name);
			}
		}
		cmd = cmd + molClassNames[for_class] + "Ptr." + bases[i] + ".cc";
		compileFiles.append(cmd);
	}
}

int
isTemplate(const String &name)
{
	if (name.empty())
		return 0;
	LibClasses *l = findLibClasses(name);
	return (l != NULL) && (l->flags & TEMPLATE);
}

int
isMap(const String &name)
{
	if (name.empty())
		return 0;
	LibClasses *l = findLibClasses(name);
	return (l != NULL) && (l->flags & MAP_CLASS);
}
