// Copyright 1993, U.C.S.F. Computer Graphics Laboratory
// $Id: lex.cc,v 2.6 1994/09/07 18:14:13 gregc Exp $

#include "lex.h"
#include <ctype.h>

char const	*const Lex::tokenNames[] = {
			"start of input",
			"end of input",
			"members keyword",
			"constructor keyword",
			"destructor keyword",
			"use keyword",
			"for keyword",
			"in keyword",
			"feature keyword",
			"keymember keyword",
			"keytype keyword",
			"rename keyword",
			"to keyword",
			"named keyword",
			"includefile keyword",
			"auxfile keyword",
			":", "{", "}", "(", ")", ";",
			"identifier",
			"member data or function call",
			"any token expected"
		};

Lex::Lex(const char *fname): line_num(1), t(SOI), errFunc(0)
{
	if (fname == NULL) {
		f = stdin;
		filename = "stdin";
	} else if ((f = fopen(fname, "r")) == NULL) {
		fprintf(stderr, "unable to open %s for reading\n", fname);
		filename = "error";
		t = EOI;
	} else
		filename = fname;
}

int
Lex::nextToken(Token tok)
{
	int	c;

	if (f == NULL)
		return 0;

	for (;;) {
		for (c = getc(f); c != EOF && isspace(c); c = getc(f))
			if (c == '\n')
				line_num += 1;
		/* look for comment */
		if (c != '/')
			break;
		c = getc(f);
		if (c != '/') {
			(void) ungetc(c, f);
			c = '/';
			break;
		}
		/* skip rest of line */
		for (c = getc(f); c != EOF && c != '\n'; c = getc(f))
			continue;
		if (c == EOF)
			break;
		line_num += 1;
	}

	switch (c) {
	case EOF: t = EOI; goto finish_up;
	case ':': t = COLON; goto finish_up;
	case '{': t = LEFT_CURLY; goto finish_up;
	case '}': t = RIGHT_CURLY; goto finish_up;
	case '(': t = LEFT_PAREN; goto finish_up;
	case ')': t = RIGHT_PAREN; goto finish_up;
	case ';': t = SEMICOLON; goto finish_up;
	}
	if (!isalpha(c)) {
		fprintf(stderr, "illegal character '\\%03o' on line %d of %s,"
			" skipping rest of input\n", c & 0xff, line_num,
			filename.chars());
		t = EOI; return 0;
	}

	ident = c;
	if (tok != MEMBERDATAFUNC)
		for (c = getc(f); isalnum(c) || c == '_'; c = getc(f))
			ident += c;
	else
		for (c = getc(f); isalnum(c) || c == '_' || c == '-' || c == '>'
				|| c == '.' || c == '(' || c == ')';
				c = getc(f))
			ident += c;
	(void) ungetc(c, f);

	if (tok == IDENT || tok == MEMBERDATAFUNC)
		t = tok;
	else if (ident == "members")
		t = MEMBERS;
	else if (ident == "constructor")
		t = CONSTRUCTOR;
	else if (ident == "destructor")
		t = DESTRUCTOR;
	else if (ident == "use")
		t = USE;
	else if (ident == "for")
		t = FOR;
	else if (ident == "in")
		t = IN;
	else if (ident == "feature")
		t = FEATURE;
	else if (ident == "keymember")
		t = KEYMEMBER;
	else if (ident == "keytype")
		t = KEYTYPE;
	else if (ident == "rename")
		t = RENAME;
	else if (ident == "to")
		t = TO;
	else if (ident == "named")
		t = NAMED;
	else if (ident == "includefile")
		t = INCLUDE_FILE;
	else if (ident == "auxfile")
		t = AUX_FILE;
	else
		t = IDENT;
finish_up:
	if (tok == ANY || tok == t)
		return 1;
	if (errorFunction())
		errorFunction()(this);
	return 0;
}

String
Lex::snarfText(char end_char)
{
	int	c;
	int	pair, num_pair;
	String	s;

	if (f == NULL)
		return "";

	if (!isspace(end_char)) {
		// skip leading blanks
		for (c = getc(f); c != EOF && isspace(c); c = getc(f))
			if (c == '\n')
				line_num += 1;
		if (c != EOF)
			ungetc(c, f);
	}
	num_pair = 0;
	switch (end_char) {
	default:	pair = -1; break;
	case '}':	pair = '{'; break;
	case ')':	pair = '('; break;
	}
	while ((c = getc(f)) != EOF && (num_pair > 0 || c != end_char)) {
		if (c == '\n')
			line_num += 1;
		else if (c == pair)
			num_pair += 1;
		else if (c == end_char)
			num_pair -= 1;
		s += c;
	}
	return s;
}

String
Lex::getFilename()
{
	int	c;
	String	s;

	if (f == NULL)
		return "";

	for (c = getc(f); c != EOF && isspace(c); c = getc(f))
		if (c == '\n')
			line_num += 1;
	if (c != '"')
		for (; c != EOF && !isspace(c) && c != ';'; c = getc(f))
			s += c;
	else {
		c = getc(f);
		for (; c != EOF && c != '"'; c = getc(f)) {
			if (c == '\n')
				line_num += 1;
			else if (c == '\\')
				c = getc(f);
			s += c;
		}
	}
	ungetc(c, f);
	return s;
}
