// Copyright 1993, U.C.S.F. Computer Graphics Laboratory
// $Id: parse_mcd.cc,v 2.13 1994/09/07 18:16:10 gregc Exp $

#include "mcd.h"
#include "lex.h"
#include <string.h>

SLList<Constructor>	constructors[MAX_MOLCLASS + 1];
Members		members[MAX_MOLCLASS + 1];
String		cleanupCode[MAX_MOLCLASS + 1];
SLList<String>	includeFiles[MAX_MOLCLASS + 1];
SLList<String>	auxFiles;

char const	*molClassNames[] = {
	"Bond", "Atom", "Ring", "Residue", "Coord", "CoordSet", "Molecule"
};

static int	parse_members(), parse_constructor(), parse_destructor(),
		parse_use(), parse_feature(), parse_includefile(),
		parse_auxfile(), parse_rename();
static void	parse_error(Lex *);
static int	lookup_class(const String &name);

static Lex	*lex;

void
parse_mcd_file(const char *filename)
{
	default_use();

	lex = new Lex(filename);
	lex->errorFunction(parse_error);
	for (lex->nextToken(); lex->token() != Lex::EOI; lex->nextToken()) {
		switch (lex->token()) {
		default:
			parse_error(lex);
			return;
		case Lex::MEMBERS:
			if (parse_members())
				return;
			break;
		case Lex::CONSTRUCTOR:
			if (parse_constructor())
				return;
			break;
		case Lex::DESTRUCTOR:
			if (parse_destructor())
				return;
			break;
		case Lex::USE:
			if (parse_use())
				return;
			break;
		case Lex::FEATURE:
			if (parse_feature())
				return;
			break;
		case Lex::INCLUDE_FILE:
			if (parse_includefile())
				return;
			break;
		case Lex::AUX_FILE:
			if (parse_auxfile())
				return;
			break;
		case Lex::RENAME:
			if (parse_rename())
				return;
			break;
		}
	}
	delete lex;
}

static int
parse_members()
{
	int	the_class;

	if (!lex->nextToken(Lex::IDENT))
		return 1;
	the_class = lookup_class(lex->id());
	if (the_class == -1) {
		parse_error(lex);
		return 1;
	}
	lex->nextToken();
	if (lex->token() == Lex::COLON) {
		members[the_class].baseClassList = lex->snarfText('{');
	} else if (lex->token() != Lex::LEFT_CURLY) {
		parse_error(lex);
		return 1;
	}
	members[the_class].memberText = lex->snarfText('}');
	return 0;
}

static int
parse_constructor()
{
	int		the_class;
	Constructor	c;

	if (!lex->nextToken(Lex::IDENT))
		return 1;
	the_class = lookup_class(lex->id());
	if (the_class == -1) {
		parse_error(lex);
		return 1;
	}

	lex->nextToken();
	if (lex->token() == Lex::LEFT_PAREN) {
		c.arguments = lex->snarfText(')');
		lex->nextToken();
	}
	if (lex->token() == Lex::COLON) {
		c.memberInit = lex->snarfText('{');
	} else if (lex->token() != Lex::LEFT_CURLY) {
		parse_error(lex);
		return 1;
	}
	c.setupCode = lex->snarfText('}');
	constructors[the_class].append(c);
	return 0;
}

static int
parse_destructor()
{
	int	the_class;
	String	cleanup_text;

	if (!lex->nextToken(Lex::IDENT))
		return 1;
	the_class = lookup_class(lex->id());
	if (the_class == -1) {
		parse_error(lex);
		return 1;
	}
	if (!lex->nextToken(Lex::LEFT_CURLY))
		return 1;
	cleanup_text = lex->snarfText('}');
	if (cleanupCode[the_class] != NULL)
		fprintf(stderr,
			"warning: line %d: overriding destructor code\n",
			lex->lineNum());
	cleanupCode[the_class] = cleanup_text;
	return 0;
}

static int
parse_use()
{
	int		for_class, in_class;
	ContainerInfo	ci;

	if (!lex->nextToken(Lex::IDENT))
		return 1;
	ci.name = lex->id();
	for_class = -1;
	in_class = -1;
	if (!lex->nextToken(Lex::FOR))
		return 1;
	if (!lex->nextToken(Lex::IDENT))
		return 1;
	for_class = lookup_class(lex->id());
	if (for_class == -1) {
		parse_error(lex);
		return 1;
	}
	if (!lex->nextToken(Lex::IN))
		return 1;
	if (!lex->nextToken(Lex::IDENT))
		return 1;
	in_class = lookup_class(lex->id());
	if (in_class == -1) {
		parse_error(lex);
		return 1;
	}
	lex->nextToken();
	if (lex->token() == Lex::NAMED) {
		if (!lex->nextToken(Lex::IDENT))
			return 1;
		ci.instance = lex->id();
		lex->nextToken();
	}
	if (lex->token() != Lex::KEYMEMBER) {
		if (isMap(ci.name))
			fprintf(stderr, "warning: line %d: %s is a map\n",
					lex->lineNum(), ci.name.chars());
		if (ci.name == "nothing")
			ci.name = "";
	} else {
		if (ci.name == "nothing" || ci.name.contains("_array", 0)) {
			fprintf(stderr,
			"error: line %d: \"use %s\" not allowed for maps\n",
				lex->lineNum(), ci.name.chars());
			return 1;
		}
		if (!isMap(ci.name))
			fprintf(stderr,
				"warning: line %d: %s is not a map\n",
				lex->lineNum(), ci.name.chars());
		if (!lex->nextToken(Lex::MEMBERDATAFUNC))
			return 1;
		ci.keyMember = lex->id();
		if (!lex->nextToken(Lex::KEYTYPE))
			return 1;
		if (!lex->nextToken(Lex::IDENT))
			return 1;
		ci.keyType = lex->id();
		lex->nextToken();
	}
	if (lex->token() != Lex::SEMICOLON) {
		parse_error(lex);
		return 1;
	}

	if (ci.instance.empty())
		ci.instance = use[for_class][in_class].instance;
	use[for_class][in_class] = ci;
	return 0;
}

static int
parse_feature()
{
	if (!lex->nextToken(Lex::IDENT))
		return 1;
	if (lex->id() == "MolGraph") {
		ContainerInfo	ci;

		ci.name = "MolGraph";
		use[Ring][Bond].name = "reference";
		use[Ring][Atom].name = "reference";
		use[Bond][Molecule] = ci;
		use[Bond][Molecule].instance = "allBonds";
		use[Atom][Molecule] = ci;
		use[Atom][Molecule].instance = "allAtoms";
		use[Ring][Molecule].name = "reference";
		use[Molecule][Molecule] = ci;
	} else if (lex->id() == "PDB") {
		use[Atom][Residue].name = "VHMap";
		use[Atom][Residue].keyMember = "name()";
		use[Atom][Residue].keyType = "Symbol";
		use[Residue][Molecule].name = "AVLMap";
		use[Residue][Molecule].keyMember = "id()";
		use[Residue][Molecule].keyType = "MolResId";
	} else
		fprintf(stderr, "warning: line %d: unknown feature %s\n",
			lex->lineNum(), lex->id().chars());
	if (!lex->nextToken(Lex::SEMICOLON))
		return 1;
	return 0;
}

static int
parse_includefile()
{
	int	in_class = -1;

	String	fn = lex->getFilename();
	lex->nextToken();
	if (lex->token() == Lex::IN) {
		if (!lex->nextToken(Lex::IDENT))
			return 1;
		in_class = lookup_class(lex->id());
		if (in_class == -1) {
			parse_error(lex);
			return 1;
		}
		lex->nextToken();
	}
	if (lex->token() != Lex::SEMICOLON) {
		parse_error(lex);
		return 1;
	}
	if (in_class != -1)
		includeFiles[in_class].append(fn);
	else for (int i = 0; i <= MAX_MOLCLASS; i += 1)
		includeFiles[i].append(fn);
	return 0;
}

static int
parse_auxfile()
{
	String	fn = lex->getFilename();
	if (!lex->nextToken(Lex::SEMICOLON))
		return 1;
	auxFiles.append(fn);
	return 0;
}

static int
parse_rename()
{
	int	the_class;

	if (!lex->nextToken(Lex::IDENT))
		return 1;
	the_class = lookup_class(lex->id());
	if (the_class == -1) {
		parse_error(lex);
		return 1;
	}
	if (!lex->nextToken(Lex::TO))
		return 1;
	if (!lex->nextToken(Lex::IDENT))
		return 1;
	molClassNames[the_class] = strdup(lex->id().chars());
	if (!lex->nextToken(Lex::SEMICOLON))
		return 1;
	return 0;
}

void
parse_error(Lex *context)
{
	fprintf(stderr,
		"warning: line %d: Unexpected %s, skipping rest of input\n",
		context->lineNum(),
		context->token() == Lex::IDENT ? context->id()
					: Lex::tokenNames[context->token()]);
}

static int
lookup_class(const String &name)
{
	for (int i = 0; i <= Molecule; i += 1)
		if (name == molClassNames[i])
			return i;
	return -1;
}
