// Copyright 1993, U.C.S.F. Computer Graphics Laboratory
// $Id: write_body.cc,v 2.32 94/09/29 13:34:26 gregc Exp $

#include <stdlib.h>
#include "mcd.h"

static void	writeBond();
static void	writeAtom();
static void	writeRing();
static void	writeResidue();
static void	writeCoord();
static void	writeCoordSet();
static void	writeMolecule();

#define	AtomName	molClassNames[Atom]
#define	BondName	molClassNames[Bond]
#define	RingName	molClassNames[Ring]
#define	ResidueName	molClassNames[Residue]
#define	CoordName	molClassNames[Coord]
#define	CoordSetName	molClassNames[CoordSet]
#define	MoleculeName	molClassNames[Molecule]

void
write_mcd_body(const char *output_name)
{
	writeBond();
	writeAtom();
	writeRing();
	writeResidue();
	writeCoord();
	writeCoordSet();
	writeMolecule();
}

static void
write_header(MolClasses in_class, FILE **fpp)
{
	String filename = String(molClassNames[in_class]) + ".cc";
	if ((*fpp = fopen(filename.chars(), "w")) == NULL) {
		fprintf(stderr, "unable to open %s for writing\n",
							filename.chars());
		exit(1);
	}
	compileFiles.append(filename);

	for (int for_class = 0; for_class <= MAX_MOLCLASS; for_class += 1) {
		if (for_class == in_class) {
			fprintf(*fpp, "#include \"%s.h\"\n",
						molClassNames[for_class]);
			continue;
		}
		if (use[for_class][in_class].name.empty())
			continue;
		if (use[for_class][in_class].name == "reference"
		|| use[for_class][in_class].name.contains("_array", 0)) {
			fprintf(*fpp, "#include \"%s.h\"\n",
						molClassNames[for_class]);
			continue;
		}
	}
	switch (in_class) {
	default:
		break;
	case Bond:
		fprintf(*fpp, "#include \"%s.h\"\n", MoleculeName);
		break;
	}
	fprintf(*fpp, "\n");
}

static void
write_trailer(MolClasses in_class, FILE **fpp)
{
	if (fclose(*fpp) == EOF) {
		fprintf(stderr, "fatal error writing %s.cc.\n",
						molClassNames[in_class]);
		exit(1);
	}
	*fpp = NULL;
}

static void
dump_constructors(FILE *f, MolClasses m, const String &head, const String &init,
							const String &code)
{
	if (constructors[m].empty()) {
		fprintf(f, "%s::%s)", molClassNames[m], head.chars());
		if (init != 0)
			fprintf(f, ": %s", init.chars());
		fprintf(f, "\n{\n%s}\n", code.chars());
		return;
	}
	for (Pix i = constructors[m].first(); i != 0; constructors[m].next(i)) {
		fprintf(f, "%s::%s", molClassNames[m], head.chars());
		if (constructors[m](i).arguments != 0)
			fprintf(f, ", %s)",
				constructors[m](i).arguments.chars());
		else
			fprintf(f, ")");
		if (init != 0 || constructors[m](i).memberInit != 0) {
			char c = ':';
			if (init != 0) {
				fprintf(f, "%c %s", c, init.chars());
				c = ',';
			}
			if (constructors[m](i).memberInit != 0)
				fprintf(f, "%c %s", c,
					constructors[m](i).memberInit.chars());
		}
		fprintf(f, "\n{\n%s%s}\n", code.chars(),
					constructors[m](i).setupCode.chars());
	}
}

static void
dump_destructor(FILE *f, MolClasses m)
{
	int	is_map;
	String	s, *sp;

	int isMolGraph = (use[Molecule][Molecule].name == "MolGraph");
	fprintf(f, "%s::~%s()\n{\n", molClassNames[m],
							molClassNames[m]);
	switch (m) {
	case Bond:
		if (isMolGraph || use[Bond][Molecule].name.empty())
			break;

		sp = &use[Atom][Bond].instance;
		fprintf(f,
			"\tif (!(%s[0]->molecule()->flags & MOL_DELETE)) {\n",
			sp->chars());
		fprintf(f, "\t\t// delete bond from mol0's list\n");
		s = *sp + "[0]->molecule()->" + use[Bond][Molecule].instance;
		fprintf(f, "%s\n", expand_del_element(2, Bond, Molecule,
							s.chars()).chars());
		fprintf(f, "\t\tif (!(%s[0]->flags & MOL_DELETE))\n",
								sp->chars());
		fprintf(f, "\t\t\t// delete bond from atom0's list\n");
		s = *sp + "[0]->" + use[Bond][Atom].instance;
		fprintf(f, "%s\n", expand_del_element(3, Bond, Atom,
							s.chars()).chars());
		fprintf(f, "\t}\n");

		fprintf(f,
			"\tif (!(%s[1]->molecule()->flags & MOL_DELETE)) {\n",
			sp->chars());
		fprintf(f, "\t\tif (%s[0]->molecule() != %s[1]->molecule())\n",
						sp->chars(), sp->chars());
		fprintf(f, "\t\t\t// delete bond from mol1's list\n");
		s = *sp + "[1]->molecule()->" + use[Bond][Molecule].instance;
		fprintf(f, "%s\n", expand_del_element(3, Bond, Molecule,
							s.chars()).chars());

		fprintf(f, "\t\tif (!(%s[1]->flags & MOL_DELETE))\n",
								sp->chars());
		fprintf(f, "\t\t\t// delete bond from atom1's list\n");
		s = *sp + "[1]->" + use[Bond][Atom].instance;
		fprintf(f, "%s\n", expand_del_element(3, Bond, Atom,
							s.chars()).chars());
		fprintf(f, "\t}\n");
		break;
	case Atom:
		fprintf(f,
			"\tif (molecule()->flags & MOL_DELETE)\n\t\treturn;\n");
		fprintf(f, "\tflags |= MOL_DELETE;\n");
		if (!isMolGraph) {
			fprintf(f, "\t// delete bonds using atom\n");
			// NOTE: this is O(n^2) due to Bond destructor
			is_map = isMap(use[Bond][Atom].name);
			if (!use[Bond][Atom].name.empty())
				fprintf(f,
					"\tif (!%s.empty())\n"
			"\t\tfor (Pix i = %s.first(); i != 0; %s.next(i))\n"
				"\t\t\tdelete %s%s(i);\n",
					use[Bond][Atom].instance.chars(),
					use[Bond][Atom].instance.chars(),
					use[Bond][Atom].instance.chars(),
					use[Bond][Atom].instance.chars(),
					is_map ? ".contents" : "");
			if (!use[Atom][Molecule].name.empty()) {
				fprintf(f, "\t// delete atom from molecule\n");
				s = use[Molecule][Atom].instance + "->"
						+ use[Atom][Molecule].instance;
				fprintf(f, "%s\n", expand_del_element(1, Atom,
						Molecule, s.chars()).chars());
			}
		}
		fprintf(f, "\tif (%s->flags & MOL_DELETE)\n\t\treturn;\n",
					use[Residue][Atom].instance.chars());
		fprintf(f, "\t// delete atom from residue's list\n");
		if (!use[Atom][Residue].name.empty()) {
			s = use[Residue][Atom].instance + "->"
						+ use[Atom][Residue].instance;
			fprintf(f, "%s\n", expand_del_element(1, Atom, Residue,
							s.chars()).chars());
		}
		break;
	case Ring:
		// TODO: ring stuff
		break;
	case Residue:
		if (use[Residue][Molecule].name.empty())
			break;
		fprintf(f, "\tif (%s->flags & MOL_DELETE)\n\t\treturn;\n",
				use[Molecule][Residue].instance.chars());
		fprintf(f, "\tflags |= MOL_DELETE;\n");
		is_map = isMap(use[Atom][Residue].name);
		if (!use[Atom][Residue].name.empty())
			fprintf(f,
				"\tif (!%s.empty())\n"
			"\t\tfor (Pix i = %s.first(); i != 0; %s.next(i))\n"
				"\t\t\tdelete %s%s(i);\n",
				use[Atom][Residue].instance.chars(),
				use[Atom][Residue].instance.chars(),
				use[Atom][Residue].instance.chars(),
				use[Atom][Residue].instance.chars(),
				is_map ? ".contents" : "");
		fprintf(f, "\t// delete residue from mol's list\n");
		s = use[Molecule][Residue].instance + "->"
					+ use[Residue][Molecule].instance;
		fprintf(f, "%s\n", expand_del_element(1, Residue, Molecule,
						s.chars()).chars());
		break;
	case Coord:
		// since we don't store pointers to coordinates, do nothing
		break;
	case CoordSet:
		if (use[CoordSet][Molecule].name.empty())
			break;
		fprintf(f, "\tif (%s->flags & MOL_DELETE)\n\t\treturn;\n",
				use[Molecule][CoordSet].instance.chars());
		fprintf(f, "\t// delete coordinate set from mol's list\n");
		s = use[Molecule][CoordSet].instance + "->"
					+ use[CoordSet][Molecule].instance;
		fprintf(f, "%s\n", expand_del_element(1, CoordSet, Molecule,
							s.chars()).chars());
		break;
	case Molecule:
		fprintf(f, "\tPix\ti;\n\tflags |= MOL_DELETE;\n");
		if (!isMolGraph) {
			is_map = isMap(use[Atom][Molecule].name);
			if (!use[Bond][Molecule].name.empty())
				fprintf(f,
					"\tif (!%s.empty())\n"
			"\t\tfor (Pix i = %s.first(); i != 0; %s.next(i))\n"
					"\t\t\tdelete %s%s(i);\n",
					use[Bond][Molecule].instance.chars(),
					use[Bond][Molecule].instance.chars(),
					use[Bond][Molecule].instance.chars(),
					use[Bond][Molecule].instance.chars(),
					is_map ? ".contents" : "");
			if (!use[Atom][Molecule].name.empty())
				fprintf(f,
					"\tif (!%s.empty())\n"
			"\t\tfor (Pix i = %s.first(); i != 0; %s.next(i))\n"
					"\t\t\tdelete %s%s(i);\n",
					use[Atom][Molecule].instance.chars(),
					use[Atom][Molecule].instance.chars(),
					use[Atom][Molecule].instance.chars(),
					use[Atom][Molecule].instance.chars(),
					is_map ? ".contents" : "");
			is_map = isMap(use[Bond][Molecule].name);
		}
		is_map = isMap(use[Residue][Molecule].name);
		if (!use[Residue][Molecule].name.empty())
			fprintf(f,
				"\tif (!%s.empty())\n"
			"\t\tfor (Pix i = %s.first(); i != 0; %s.next(i))\n"
				"\t\t\tdelete %s%s(i);\n",
				use[Residue][Molecule].instance.chars(),
				use[Residue][Molecule].instance.chars(),
				use[Residue][Molecule].instance.chars(),
				use[Residue][Molecule].instance.chars(),
				is_map ? ".contents" : "");
		is_map = isMap(use[CoordSet][Molecule].name);
		if (!use[CoordSet][Molecule].name.empty())
			fprintf(f,
				"\tif (!%s.empty())\n"
			"\t\tfor (Pix i = %s.first(); i != 0; %s.next(i))\n"
				"\t\t\tdelete %s%s(i);\n",
				use[CoordSet][Molecule].instance.chars(),
				use[CoordSet][Molecule].instance.chars(),
				use[CoordSet][Molecule].instance.chars(),
				use[CoordSet][Molecule].instance.chars(),
				is_map ? ".contents" : "");
		break;
	}
	fprintf(f, "%s}\n", cleanupCode[m].chars());
}

static void
use_init(String *init, int in_class)
{
	for (int for_class = 0; for_class <= MAX_MOLCLASS; for_class += 1) {
		if (!isMap(use[for_class][in_class].name))
			continue;
		*init += ", " + use[for_class][in_class].instance;
		if (isPtr(for_class))
			*init += String("((") + molClassNames[for_class]
								+ " *) 0)";
		else
			*init += String("(") + molClassNames[for_class] + "())";
	}
}

static void
writeBond()
{
	FILE	*f;
	String	head, init, code;

	write_header(Bond, &f);
	if (f == NULL)
		return;

	int isMolGraph = (use[Bond][Molecule].name == "MolGraph");
	init = "BaseBond()";
	code = "";
	if (isMolGraph) {
		init += String(", MolEdge<") + AtomName + ", " + BondName
				+ ", " + MoleculeName + ", " + RingName
				+ ">(a0, a1)";
	} else {
		if (!use[Bond][Molecule].name.empty()) {
			String s0, s1;
			s0 = use[Atom][Bond].instance + "[0]->molecule()->"
				+ use[Bond][Molecule].instance;
			s1 = use[Atom][Bond].instance + "[1]->molecule()->"
				+ use[Bond][Molecule].instance;
			code = expand_add_element(1, Bond, Molecule, s0.chars())
				+ "\n\tif (" + use[Atom][Bond].instance
				+ "[0]->molecule() != "
				+ use[Atom][Bond].instance
				+ "[1]->molecule())\n"
				+ expand_add_element(2, Bond, Molecule,
								s1.chars());
			code += '\n';
		}
		if (!use[Bond][Atom].name.empty()) {
			String s0, s1;
			s0 = use[Atom][Bond].instance + "[0]->"
						+ use[Bond][Atom].instance;
			s1 = use[Atom][Bond].instance + "[1]->"
						+ use[Bond][Atom].instance;
			code = expand_add_element(1, Bond, Atom, s0.chars())
				+ '\n'
				+ expand_add_element(1, Bond, Atom, s1.chars())
				+ '\n'
				+ code;
		}
		code = "\t" + use[Atom][Bond].instance + "[0] = a0;\n\t"
			+ use[Atom][Bond].instance + "[1] = a1;\n" + code;
	}
	use_init(&init, Bond);
	head = String(BondName) + "(" + AtomName + " *a0, " + AtomName + " *a1";
	dump_constructors(f, Bond, head, init, code);

	init = "BaseBond(order)";
	if (isMolGraph) {
		init += String(", MolEdge<") + AtomName + ", " + BondName
				+ ", " + MoleculeName + ", " + RingName
				+ ">(a0, a1)";
	}
	use_init(&init, Bond);
	head = String(BondName) + "(" + AtomName + " *a0, " + AtomName
							+ " *a1, float order";
	dump_constructors(f, Bond, head, init, code);

	dump_destructor(f, Bond);

	write_trailer(Bond, &f);
}

static void
writeAtom()
{
	FILE	*f;
	String	head, init, code;

	write_header(Atom, &f);
	if (f == NULL)
		return;

	int isMolGraph = (use[Atom][Molecule].name == "MolGraph");
	if (isMolGraph) {
		init = String("MolVertex<") + AtomName + ", " + BondName
			+ ", " + MoleculeName + ", " + RingName
			+ ">(m), BaseAtom()";
		if (!use[Atom][Residue].name.empty())
			init += String(", ") + use[Residue][Atom].instance
								+ "(r)";
		code = "";
	} else {
		init = use[Molecule][Atom].instance + "(m), BaseAtom()";
		if (!use[Atom][Residue].name.empty())
			init += String(", ") + use[Residue][Atom].instance
								+ "(r)";
		if (use[Atom][Molecule].name.empty())
			code = "";
		else {
			String s = use[Molecule][Atom].instance + "->"
						+ use[Atom][Molecule].instance;
			code = expand_add_element(1, Atom, Molecule, s.chars())
									+ '\n';
		}
	}
	if (!use[Atom][Residue].name.empty()) {
		String s = use[Residue][Atom].instance + "->"
						+ use[Atom][Residue].instance;
		code += expand_add_element(1, Atom, Residue, s.chars()) + '\n';
	}
	use_init(&init, Atom);
	head = String(AtomName) + "(" + MoleculeName + " *m";
	if (!use[Atom][Residue].name.empty())
		head += String(", ") + ResidueName + " *r";
	dump_constructors(f, Atom, head, init, code);
	if (isMolGraph)
		init = String("MolVertex<") + AtomName + ", " + BondName
			+ ", " + MoleculeName + ", " + RingName
			+ ">(m), BaseAtom(n, element)";
	else
		init = use[Molecule][Atom].instance + "(m), BaseAtom(n, element)";
	if (!use[Atom][Residue].name.empty())
		init += String(", ") + use[Residue][Atom].instance + "(r)";
	head = String(AtomName) + "(" + MoleculeName + " *m, ";
	if (!use[Atom][Residue].name.empty())
		head += String(ResidueName) + " *r, ";
	head += "Symbol n, AtomicSymbol element";
	dump_constructors(f, Atom, head, init, code);
	if (isMolGraph)
		init = String("MolVertex<") + AtomName + ", " + BondName
			+ ", " + MoleculeName + ", " + RingName
			+ ">(m), BaseAtom(n, element, atomType)";
	else
		init = use[Molecule][Atom].instance
				+ "(m), BaseAtom(n, element, atomType)";
	if (!use[Atom][Residue].name.empty())
		init += String(", ") + use[Residue][Atom].instance + "(r)";
	head = String(AtomName) + "(" + MoleculeName + " *m, ";
	if (!use[Atom][Residue].name.empty())
		head += String(ResidueName) + " *r, ";
	head += "Symbol n, AtomicSymbol element, AtomType atomType";
	dump_constructors(f, Atom, head, init, code);
	dump_destructor(f, Atom);

	if (use[Coord][CoordSet].name.empty()) {
		write_trailer(Atom, &f);
		return;
	}

	fprintf(f, "\nvoid\n%s::newCoord() const\n"
		"{\n"
		"\tint\tfirst = 1;\n\n"
		"\tfor (CIter<%s> csi = molecule()->citerCoordSet(); csi.ok(); csi.next()) {\n"
		"\t\tif (first)\n"
		"\t\t\tindex = csi->%s.add(%s());\n"
		"\t\telse if (!csi->%s.valid(index))\n"
		"\t\t\twhile (csi->%s.add(%s()) < index)\n"
		"\t\t\t\tcontinue;\n"
		"\t\tfirst = 0;\n"
		"\t}\n"
		"}\n",
		AtomName,
		CoordSetName,
		use[Coord][CoordSet].instance.chars(),
		CoordName,
		use[Coord][CoordSet].instance.chars(),
		use[Coord][CoordSet].instance.chars(),
		CoordName);

	fprintf(f, "\n%s *\n%s::getCoord() const\n"
		"{\n"
		"\t%s\t*cs;\n\n",
		CoordName, AtomName, CoordSetName);
	fprintf(f, "\tif ((cs = molecule()->activeCoordSet()) == NULL) {\n");
	if (isMap(use[CoordSet][Molecule].name)) {
		// TODO: assert that key type is MolCoordSetId
		fprintf(f,
			"\t\tMolCoordSetId csid(Symbol(\"default\"));\n"
		"\t\tif ((cs = molecule()->lookupCoordSet(csid)) == NULL)\n"
			"\t\t\tcs = new %s(molecule(), csid);\n",
			CoordSetName);
	} else {
		if (!use[CoordSet][Molecule].name.empty())
			fprintf(f,
				"\t\tif (!%s.empty())\n"
				"\t\t\tcs = %s(%s.first());\n"
				"\t\telse\n\t",
				use[CoordSet][Molecule].instance.chars(),
				use[CoordSet][Molecule].instance.chars(),
				use[CoordSet][Molecule].instance.chars());
		fprintf(f, "\t\tcs = new %s(molecule());\n", CoordSetName);
	}
	fprintf(f, "\t\tmolecule()->activeCoordSet(cs);\n");
	fprintf(f,
		"\t}\n"
		"\tif (index == -1)\n"
		"\t\tnewCoord();\n"
		"\treturn cs->lookupCoord(index);\n"
		"}\n");

	fprintf(f, "\n%s *\n%s::getCoord(%s *cs) const\n"
		"{\n"
		"\tif (index == -1)\n"
		"\t\tnewCoord();\n"
		"\treturn cs->lookupCoord(index);\n"
		"}\n",
		CoordName, AtomName, CoordSetName);
	write_trailer(Atom, &f);
}

static void
writeRing()
{
	FILE	*f;

	write_header(Ring, &f);
	if (f == NULL)
		return;

	// TODO

	write_trailer(Ring, &f);
}

static void
writeResidue()
{
	FILE	*f;
	String	head, init, code;

	write_header(Residue, &f);
	if (f == NULL)
		return;

	if (use[Residue][Molecule].name.empty())
		code = "";
	else {
		String	s = use[Molecule][Residue].instance + "->"
					+ use[Residue][Molecule].instance;
		code = expand_add_element(1, Residue, Molecule, s.chars())
									+ '\n';
	}
	init = use[Molecule][Residue].instance
					+ "(m), BaseResidue(t, rid), flags(0)";
	use_init(&init, Residue);
	head = String(ResidueName) + "(" + MoleculeName
						+ " *m, Symbol t, MolResId rid";
	dump_constructors(f, Residue, head, init, code);
	head = String(ResidueName) + "(" + MoleculeName
			+ " *m, Symbol t, Symbol chain, int pos, char insert";
	init = use[Molecule][Residue].instance
			+ "(m), BaseResidue(t, chain, pos, insert), flags(0)",
	use_init(&init, Residue);
	dump_constructors(f, Residue, head, init, code);
	dump_destructor(f, Residue);

	write_trailer(Residue, &f);
}

static void
writeCoord()
{
	FILE	*f;
	String	head, init;

	write_header(Coord, &f);
	if (f == NULL)
		return;

	init = "BaseCoord()";
	use_init(&init, Coord);
	head = String(CoordName) + '(';
	dump_constructors(f, Coord, head, init, _nilString);
	dump_destructor(f, Coord);

	write_trailer(Coord, &f);
}

static void
writeCoordSet()
{
	FILE	*f;
	String	head, init, code;

	write_header(CoordSet, &f);
	if (f == NULL)
		return;

	head = String(CoordSetName) + '(' + MoleculeName + " *m";
	code = String("\t") + CoordSetName + " *cs = m->activeCoordSet();\n"
		"\tif (cs == NULL)\n"
		"\t\tcs = m->citerCoordSet();\n"
		"\tif (cs != NULL)\n"
		"\t\tcoords = cs->coords;\n";
	if (isMap(use[CoordSet][Molecule].name))
		head += ", " + use[CoordSet][Molecule].keyType + " k";
	if (use[Coord][CoordSet].name.empty()) {
		init = use[Molecule][CoordSet].instance + "(m)";
		if (isMap(use[CoordSet][Molecule].name))
			init += ", BaseCoordSet(k)";
	} else {
		init = use[Molecule][CoordSet].instance + "(m)";
		if (isMap(use[CoordSet][Molecule].name))
			init += ", BaseCoordSet(k)";
		init += ", " + use[Coord][CoordSet].instance + "()";
		String	s = use[Molecule][CoordSet].instance + "->"
					+ use[CoordSet][Molecule].instance;
		code += expand_add_element(1, CoordSet, Molecule, s.chars())
									+ '\n';
	}
	use_init(&init, CoordSet);
	dump_constructors(f, CoordSet, head, init, code);
	dump_destructor(f, CoordSet);

	write_trailer(CoordSet, &f);
}

static void
writeMolecule()
{
	FILE	*f;
	String	head, init;

	write_header(Molecule, &f);
	if (f == NULL)
		return;

	int isMolGraph = (use[Molecule][Molecule].name == "MolGraph");
	if (isMolGraph)
		init = String("MolGraph<") + AtomName + ", " + BondName
			+ ", " + MoleculeName + ", " + RingName
			+ ">(), BaseMolecule(), flags(0), activeCS(NULL)";
	else
		init = String("BaseMolecule(), flags(0), activeCS(NULL)");
	use_init(&init, Molecule);
	head = String(MoleculeName) + '(';
	dump_constructors(f, Molecule, head, init, _nilString);
	dump_destructor(f, Molecule);
	if (isMolGraph)
		fprintf(f,
		"void %s::rotateBond(%s *b, const %s *a, float angle)\n"
		"{\n"
		"\tGeom3d::Xform\tt, tinv;\n"
		"\t%s\t\t*o;\n"
		"\tconst %s\t*bp;\n"
		"\n"
		"\tif ((o = b->otherAtom(a)) == NULL)\n"
		"\t\treturn;\n"
		"\tbp = b->getBreakPoint();\n"
		"\ttinv.setZAlign(a->getCoord()->xyz, o->getCoord()->xyz);\n"
		"\tt = tinv;\n"
		"\tt.invert();\n"
		"\tt.zRotate(angle);\n"
		"\tt.multiply(tinv);\n"
		"\n"
		"\tb->setBreakPoint(a);\n"
		"\tCIter<%s> ai = o->molecule()->traverseVertices(o->root());\n"
		"\tfor (; ai.ok(); ai.next()) {\n"
		"\t\tGeom3d::Point &p = ai->getCoord()->xyz;\n"
		"\t\tp = t.apply(p);\n"
		"\t}\n"
		"\tb->setBreakPoint(bp);\n"
		"}\n",
		MoleculeName, BondName, AtomName, AtomName, AtomName, AtomName);
	write_trailer(Molecule, &f);
}
