// Copyright 1993, U.C.S.F. Computer Graphics Laboratory
// $Id: write_header.cc,v 2.24 94/10/04 16:53:13 gregc Exp $

#include <stdlib.h>
#include "mcd.h"

#define	BondName	molClassNames[Bond]
#define	AtomName	molClassNames[Atom]
#define	RingName	molClassNames[Ring]
#define	ResidueName	molClassNames[Residue]
#define	CoordName	molClassNames[Coord]
#define	CoordSetName	molClassNames[CoordSet]
#define	MoleculeName	molClassNames[Molecule]

const char *origClassNames[MAX_MOLCLASS + 1] = {
	"Bond", "Atom", "Ring", "Residue", "Coord", "CoordSet", "Molecule"
};

static void	writeBond();
static void	writeAtom();
static void	writeRing();
static void	writeResidue();
static void	writeCoord();
static void	writeCoordSet();
static void	writeMolecule();
static void	dump_includes(FILE *f, int for_class);

static Regex	imapSuffix("\\(Plex\\|Vec\\)$");

void
write_mcd_header(const char * /* output_name */)
{
	writeBond();
	writeAtom();
	writeRing();
	writeResidue();
	writeCoord();
	writeCoordSet();
	writeMolecule();
}

//	dump include files needed by class

static void
dump_includes(FILE *f, int in_class)
{
	fprintf(f, "#include <mol/Iter.h>\n");
	if (use[in_class][Molecule].name == "MolGraph")
		switch (in_class) {
		case Bond:
			fprintf(f, "#include <mg/MolEdge.h>\n");
			break;
		case Atom:
			fprintf(f, "#include <mg/MolVertex.h>\n");
			break;
		case Molecule:
			fprintf(f, "#include <mg/MolGraph.h>\n");
			break;
		}

	for (int for_class = 0; for_class <= MAX_MOLCLASS; for_class += 1) {
		if (for_class == in_class
		|| use[for_class][in_class].name.empty()
		|| use[for_class][in_class].name == "MolGraph")
			continue;
		if (use[for_class][in_class].name == "reference"
		|| use[for_class][in_class].name.contains("_array", 0)) {
			fprintf(f, "class %s;\n", molClassNames[for_class]);
			continue;
		}
		if (isTemplate(use[for_class][in_class].name)) {
			fprintf(f, "#include <%s.h>\n#include \"%s.h\"\n",
					use[for_class][in_class].name.chars(),
					molClassNames[for_class]);
			continue;
		}
		fprintf(f, "#include \"");
		if (isMap(use[for_class][in_class].name))
			fprintf(f, "%s.",
				use[for_class][in_class].keyType.chars());
		if (!isPtr(for_class))
			fprintf(f, "%s.%s.h\"\n", molClassNames[for_class],
					use[for_class][in_class].name.chars());
		else
			fprintf(f, "%sPtr.%s.h\"\n", molClassNames[for_class],
					use[for_class][in_class].name.chars());
	}
}

static void
dump_constructors(FILE *f, MolClasses m, const char *head)
{
	if (constructors[m].empty()) {
		fprintf(f, "\t%s);\n", head);
		return;
	}
	for (Pix i = constructors[m].first(); i != 0; constructors[m].next(i)) {
		fprintf(f, "\t%s", head);
		if (constructors[m](i).arguments == 0)
			fprintf(f, ");\n");
		else
			fprintf(f, ", %s);\n",
					constructors[m](i).arguments.chars());
	}
}

static void
write_header(MolClasses c, FILE **fpp)
{
	String filename(molClassNames[c]);
	filename += ".h";
	if ((*fpp = fopen(filename.chars(), "w")) == NULL) {
		fprintf(stderr, "unable to open %s for writing\n",
							filename.chars());
		exit(1);
	}
	fprintf(*fpp,
		"#ifndef %s_h\n"
		"#define\t%s_h\n"
		"\n"
		"#include <mol/BaseMolecule.h>\n",
		molClassNames[c], molClassNames[c]);

	if (!includeFiles[c].empty()) {
		for (Pix i = includeFiles[c].first(); i != 0;
							includeFiles[c].next(i))
			fprintf(*fpp, "#include \"%s\"\n",
						includeFiles[c](i).chars());
	}
	dump_includes(*fpp, c);
}

static void
write_trailer(MolClasses c, FILE **fpp)
{
	fprintf(*fpp, "#endif\n");
	if (fclose(*fpp) == EOF) {
		fprintf(stderr, "fatal error writing %s.h.\n",
							molClassNames[c]);
		exit(1);
	}
	*fpp = NULL;
}

static void
write_instance(FILE *f, int for_class, int in_class)
{
	if (use[for_class][in_class].name.empty())
		return;
	fprintf(f, "\tfriend class\t%s;\n", molClassNames[for_class]);
	int is_ptr = isPtr(for_class);
	if (use[for_class][in_class].name.contains("_array", 0)) {
		fprintf(f, "\t%s\t%s%s[%s];\n",
			molClassNames[for_class], is_ptr ? "*" : "",
			use[for_class][in_class].instance.chars(),
			use[for_class][in_class].name.from(6).chars());
		return;
	}
	if (isTemplate(use[for_class][in_class].name)) {
		if (is_ptr)
			fprintf(f, "\ttypedef %s\t*%sPtr;\n",
				molClassNames[for_class],
				molClassNames[for_class]);
		fprintf(f, "\t%s<%s%s>\t%s;\n",
			use[for_class][in_class].name.chars(),
			molClassNames[for_class], is_ptr ? "Ptr" : "",
			use[for_class][in_class].instance.chars());
		return;
	}
	int is_map = isMap(use[for_class][in_class].name);
	fprintf(f, "\t%s%s%s%s\t%s;\n",
			is_map ? use[for_class][in_class].keyType.chars() : "",
			molClassNames[for_class], is_ptr ? "Ptr" : "",
			use[for_class][in_class].name.chars(),
			use[for_class][in_class].instance.chars());
}
static void
write_lookup(FILE *f, int for_class, int in_class)
{
	int is_ptr = isPtr(for_class);
	if (use[for_class][in_class].name.contains(imapSuffix)
	|| use[for_class][in_class].name.contains("_array", 0)) {
		// _array's, Plex and Vec classes are integrally "mapped"
		fprintf(f, "\t%s\t*lookup%s(int i) { return %s%s[i]; }\n",
			molClassNames[for_class],
			origClassNames[for_class],
			is_ptr ? "" : "&",
			use[for_class][in_class].instance.chars());
		return;
	}

	if (!isMap(use[for_class][in_class].name))
		return;
	fprintf(f, "\t%s\t*lookup%s(const %s &key) { Pix p = %s.seek(key); if (p == 0) return NULL; return %s%s.contents(p); }\n",
		molClassNames[for_class],
		origClassNames[for_class],
		use[for_class][in_class].keyType.chars(),
		use[for_class][in_class].instance.chars(),
		is_ptr ? "" : "&",
		use[for_class][in_class].instance.chars());
}

static void
writeBond()
{
	FILE	*f;
	String	s;

	write_header(Bond, &f);
	if (f == NULL)
		return;

	int isMolGraph = (use[Bond][Molecule].name == "MolGraph");
	if (isMolGraph)
		fprintf(f, "class %s;\nclass %s;\n", BondName, MoleculeName);
	fprintf(f, "\nclass %s: public BaseBond", BondName);
	if (isMolGraph)
		fprintf(f, ", public MolEdge<%s, %s, %s, %s>", AtomName,
			BondName, MoleculeName, RingName);
	if (members[Bond].baseClassList != 0)
		fprintf(f, ", %s", members[Bond].baseClassList.chars());
	fprintf(f, " {\n");
	fprintf(f, "\tvoid\toperator=(const %s &r);\t// disable\n", BondName);
	fprintf(f, "\t\t%s(const %s &b);\t// disable\n", BondName, BondName);
	if (!isMolGraph) {
		write_instance(f, Atom, Bond);
		s = expand_iterator(Atom, Bond);
		if (s != 0)
			fprintf(f, "%s\n", s.chars());
	}
	fprintf(f, "public:\n");
	s = String(BondName) + "(" + AtomName + " *a0, " + AtomName + " *a1";
	dump_constructors(f, Bond, s.chars());
	s = String(BondName) + "(" + AtomName + " *a0, " + AtomName
							+ " *a1, float order";
	dump_constructors(f, Bond, s.chars());
	fprintf(f, "\tvirtual\t~%s();\n", BondName);
	if (!isMolGraph)
		write_lookup(f, Atom, Bond);
	if (isMolGraph || !use[Atom][Bond].name.empty()) {
		fprintf(f, "\t%s\t*otherAtom(const %s *a) const {\n", AtomName,
								AtomName);
		if (isMolGraph) {
			fprintf(f, "\t\t\treturn otherVertex(a);\n");
		} else {
			fprintf(f, "\t\t\tif (atoms_[0] != a)\n");
			fprintf(f, "\t\t\t\treturn atoms_[0];\n");
			fprintf(f, "\t\t\tif (atoms_[1] != a)\n");
			fprintf(f, "\t\t\t\treturn atoms_[1];\n");
			fprintf(f, "\t\t\treturn NULL;\n");
		}
		fprintf(f, "\t\t}\n");
	}
	if (isMolGraph) {
		fprintf(f, "\tCIter<%s>\tciterAtom() const {\n", AtomName);
		fprintf(f, "\t\t\treturn vertexIter();\n");
		fprintf(f, "\t\t}\n");
		fprintf(f, "\t%s\t*lookupAtom(int i) const {\n", AtomName);
		fprintf(f, "\t\t\treturn vertex(i);\n");
		fprintf(f, "\t\t}\n");
	}
	if (members[Bond].memberText != 0)
		fprintf(f, "%s", members[Bond].memberText.chars());
	else
		fprintf(f, "%s", class_operators(Bond, "BaseBond").chars());
	fprintf(f, "};\n");
	write_trailer(Bond, &f);
}

static void
writeAtom()
{
	FILE	*f;
	String	s;

	write_header(Atom, &f);
	if (f == NULL)
		return;

	int isMolGraph = (use[Atom][Molecule].name == "MolGraph");
	if (isMolGraph)
		fprintf(f, "class %s;\nclass %s;\n", AtomName, BondName);

	fprintf(f, "\nclass %s: public BaseAtom", AtomName);
	if (isMolGraph)
		fprintf(f, ", public MolVertex<%s, %s, %s, %s>", AtomName,
					BondName, MoleculeName, RingName);
	if (members[Atom].baseClassList != 0)
		fprintf(f, ", %s", members[Atom].baseClassList.chars());
	fprintf(f, " {\n");
	fprintf(f, "\tvoid\toperator=(const %s &r);\t// disable\n", AtomName);
	fprintf(f, "\t\t%s(const %s &a);\t// disable\n", AtomName, AtomName);
	fprintf(f, "\tunsigned int\tflags;\n");
	if (!isMolGraph) {
		fprintf(f, "\t%s\t*const %s;\n", MoleculeName,
					use[Molecule][Atom].instance.chars());
		write_instance(f, Bond, Atom);
	}
	if (!use[Atom][Residue].name.empty())
		fprintf(f, "\t%s\t*const %s;\n", ResidueName,
					use[Residue][Atom].instance.chars());
	if (!isMolGraph) {
		s = expand_iterator(Bond, Atom);
		if (s != 0)
			fprintf(f, "%s\n", s.chars());
	}
	fprintf(f, "public:\n");
	s = String(AtomName) + "(" + MoleculeName + " *m";
	if (!use[Atom][Residue].name.empty())
		s += String(", ") + ResidueName + " *r";
	dump_constructors(f, Atom, s.chars());
	s = String(AtomName) + "(" + MoleculeName + " *m, ";
	if (!use[Atom][Residue].name.empty())
		s += String(ResidueName) + " *r, ";
	s += "Symbol name, AtomicSymbol element";
	dump_constructors(f, Atom, s.chars());
	s = String(AtomName) + "(" + MoleculeName + " *m, ";
	if (!use[Atom][Residue].name.empty())
		s += String(ResidueName) + " *r, ";
	s += "Symbol name, AtomicSymbol element, AtomType atomType";
	dump_constructors(f, Atom, s.chars());
	fprintf(f, "\tvirtual\t~%s();\n", AtomName);
	fprintf(f, "\tvoid\tnewCoord() const;\n");
	fprintf(f, "\t%s\t*getCoord() const;\n", CoordName);
	fprintf(f, "\t%s\t*getCoord(%s *) const;\n", CoordName, CoordSetName);
	fprintf(f, "\t%s\t*molecule() const { return ", MoleculeName);
	if (isMolGraph)
		fprintf(f, "graph()");
	else
		fprintf(f, "%s", use[Molecule][Atom].instance.chars());
	fprintf(f, "; }\n");
	if (!isMolGraph && !use[Bond][Atom].name.empty()) {
		// MolGraph provides its own isConnectedTo
		fprintf(f, "\t%s\t*isConnectedTo(%s *a) {\n", BondName,
								AtomName);
		fprintf(f, "\t\t\tfor (CIter<%s> bi = citerBond(); bi.ok();"
						" bi.next())\n", BondName);
		fprintf(f, "\t\t\t\tif (bi->otherAtom(this) == a)\n");
		fprintf(f, "\t\t\t\t\treturn bi;\n");
		fprintf(f, "\t\t\treturn 0;\n");
		fprintf(f, "\t\t}\n");
	}
	if (!use[Atom][Residue].name.empty())
		fprintf(f, "\t%s\t*residue() const { return %s; }\n",
			ResidueName, use[Residue][Atom].instance.chars());
	if (isMolGraph) {
		fprintf(f, "\tCIter<%s>\tciterBond() const {\n", BondName);
		fprintf(f, "\t\t\treturn edgeIter();\n");
		fprintf(f, "\t\t}\n");
	}
	if (members[Atom].memberText != 0)
		fprintf(f, "%s", members[Atom].memberText.chars());
	else
		fprintf(f, "%s", class_operators(Atom, "BaseAtom").chars());
	fprintf(f, "};\n");
	write_trailer(Atom, &f);
}

static void
writeRing()
{
	FILE	*f;

	write_header(Ring, &f);
	if (f == NULL)
		return;

	// TODO

	write_trailer(Ring, &f);
}

static void
writeResidue()
{
	FILE	*f;
	String	s;

	write_header(Residue, &f);
	if (f == NULL)
		return;

	fprintf(f, "\nclass %s: public BaseResidue", ResidueName);
	if (members[Residue].baseClassList != 0)
		fprintf(f, ", %s", members[Residue].baseClassList.chars());
	fprintf(f, " {\n\tunsigned int\tflags;\n\t%s\t*const %s;\n",
			MoleculeName, use[Molecule][Residue].instance.chars());
	write_instance(f, Atom, Residue);
	write_instance(f, Bond, Residue);
	s = expand_iterator(Atom, Residue);
	if (s != 0)
		fprintf(f, "%s\n", s.chars());
	s = expand_iterator(Bond, Residue);
	if (s != 0)
		fprintf(f, "%s\n", s.chars());
	fprintf(f, "public:\n");
	s = String(ResidueName) + "(" + MoleculeName
						+ " *m, Symbol t, MolResId rid";
	dump_constructors(f, Residue, s.chars());
	s = String(ResidueName) + "(" + MoleculeName
			+ " *m, Symbol t, Symbol chain, int pos, char insert";
	dump_constructors(f, Residue, s.chars());
	fprintf(f, "\tvirtual\t~%s();\n", ResidueName);
	fprintf(f, "\t%s\t*molecule() const { return %s; }\n", MoleculeName,
				use[Molecule][Residue].instance.chars());
	write_lookup(f, Atom, Residue);
	write_lookup(f, Bond, Residue);
	if (members[Residue].memberText != 0)
		fprintf(f, "%s", members[Residue].memberText.chars());
	else
		fprintf(f, "%s", class_operators(Residue, "BaseResidue").chars());
	fprintf(f, "};\n");
	write_trailer(Residue, &f);
}

static void
writeCoord()
{
	FILE	*f;
	String	s;

	write_header(Coord, &f);
	if (f == NULL)
		return;

	fprintf(f, "\nclass %s: public BaseCoord", CoordName);
	if (members[Coord].baseClassList != 0)
		fprintf(f, ", %s", members[Coord].baseClassList.chars());
	fprintf(f, " {\npublic:\n");
	s = String(CoordName) + '(';
	dump_constructors(f, Coord, s.chars());
	fprintf(f, "\tvirtual\t~%s();\n", CoordName);
	if (members[Coord].memberText != 0)
		fprintf(f, "%s", members[Coord].memberText.chars());
	else
		fprintf(f, "%s", class_operators(Coord, "BaseCoord").chars());
	fprintf(f, "};\n");
	write_trailer(Coord, &f);
}

static void
writeCoordSet()
{
	FILE	*f;
	String	s;

	write_header(CoordSet, &f);
	if (f == NULL)
		return;

	fprintf(f, "\nclass %s: public BaseCoordSet", CoordSetName);
	if (members[CoordSet].baseClassList != 0)
		fprintf(f, ", %s", members[CoordSet].baseClassList.chars());
	fprintf(f, " {\n\t%s\t*const %s;\n", MoleculeName,
				use[Molecule][CoordSet].instance.chars());
	fprintf(f, "\tfriend class\t%s;\n", AtomName);
	write_instance(f, Coord, CoordSet);
	s = expand_iterator(Coord, CoordSet);
	if (s != 0)
		fprintf(f, "%s\n", s.chars());
	fprintf(f, "public:\n");
	s = String(CoordSetName) + '(' + MoleculeName + " *m";
	if (isMap(use[CoordSet][Molecule].name))
		s += ", " + use[CoordSet][Molecule].keyType + " k";
	dump_constructors(f, CoordSet, s.chars());
	fprintf(f, "\tvirtual\t~%s();\n", CoordSetName);
	fprintf(f, "\t%s\t*molecule() const { return %s; }\n", MoleculeName,
				use[Molecule][CoordSet].instance.chars());
	write_lookup(f, Coord, CoordSet);
	if (members[CoordSet].memberText != 0)
		fprintf(f, "%s", members[CoordSet].memberText.chars());
	fprintf(f, "};\n");
	write_trailer(CoordSet, &f);
}

static void
writeMolecule()
{
	FILE	*f;
	String	s;

	write_header(Molecule, &f);
	if (f == NULL)
		return;

	int isMolGraph = (use[Molecule][Molecule].name == "MolGraph");
	fprintf(f, "\nclass %s: public BaseMolecule", MoleculeName);
	if (isMolGraph)
		fprintf(f, ", public MolGraph<%s, %s, %s, %s>", AtomName,
					BondName, MoleculeName, RingName);
	if (members[Molecule].baseClassList != 0)
		fprintf(f, ", %s", members[Molecule].baseClassList.chars());
	fprintf(f, " {\n");
	fprintf(f, "\tvoid\toperator=(const %s &r);\t// disable\n",
								MoleculeName);
	fprintf(f, "\t\t%s(const %s &b);\t// disable\n", MoleculeName,
								MoleculeName);
	fprintf(f, "\tunsigned int\tflags;\n");
	fprintf(f, "\t%s\t*activeCS;\n", CoordSetName);
	if (isMolGraph)
		fprintf(f, "\tfriend class\t%s;\n", AtomName);
	else {
		write_instance(f, Atom, Molecule);
		write_instance(f, Bond, Molecule);
	}
	write_instance(f, Residue, Molecule);
	write_instance(f, CoordSet, Molecule);
	if (!isMolGraph) {
		s = expand_iterator(Atom, Molecule);
		if (s != 0)
			fprintf(f, "%s\n", s.chars());
		s = expand_iterator(Bond, Molecule);
		if (s != 0)
			fprintf(f, "%s\n", s.chars());
	}
	s = expand_iterator(Residue, Molecule);
	if (s != 0)
		fprintf(f, "%s\n", s.chars());
	s = expand_iterator(CoordSet, Molecule);
	if (s != 0)
		fprintf(f, "%s\n", s.chars());
	fprintf(f, "public:\n");
	s = String(MoleculeName) + '(';
	dump_constructors(f, Molecule, s.chars());
	fprintf(f, "\tvirtual\t~%s();\n", MoleculeName);
	fprintf(f, "\tvoid\tactiveCoordSet(%s *a) { activeCS = a; }\n",
								CoordSetName);
	fprintf(f, "\t%s\t*activeCoordSet() { return activeCS; }\n",
								CoordSetName);
	if (!isMolGraph) {
		write_lookup(f, Atom, Molecule);
		write_lookup(f, Bond, Molecule);
	}
	write_lookup(f, Residue, Molecule);
	write_lookup(f, CoordSet, Molecule);
	if (isMolGraph) {
		fprintf(f,
		    "\tvoid\trotateBond(%s *b, const %s *a, float angle);\n",
							BondName, AtomName);
		fprintf(f, "\tCIter<%s>\tciterBond() const {\n", BondName);
		fprintf(f, "\t\t\treturn edgeIter();\n");
		fprintf(f, "\t\t}\n");
		fprintf(f, "\tCIter<%s>\tciterAtom() const {\n", AtomName);
		fprintf(f, "\t\t\treturn vertexIter();\n");
		fprintf(f, "\t\t}\n");
	}
	if (members[Molecule].memberText != 0)
		fprintf(f, "%s", members[Molecule].memberText.chars());
	fprintf(f, "};\n");
	write_trailer(Molecule, &f);
}
