#include <stdio.h>
#include "Mol/Molecule.h"

//
// Compute the bounding box for the atom
//
void
Atom::computeBBox(const Camera &camera, float fraction)
{
	const MolPos &coord = getCoord()->xyz;
	float x = coord[0];
	float y = coord[1];
	float z = coord[2];
	radius_ *= fraction;
	camera.project(x, y, z, projectedCenter_);
	projectedRadius_ = camera.projectLength(radius_, z);
	setBounds(projectedCenter_[0] - projectedRadius_,
			projectedCenter_[0] + projectedRadius_,
			projectedCenter_[1] - projectedRadius_,
			projectedCenter_[1] + projectedRadius_);
}

//
// Check if this atom covers or is covered by the given
// one in the projected image
//
int
Atom::depthCompare(const Camera &camera, Atom &a)
{
	if (!overlaps(a))
		return 0;
	FixedAVector<float, 2> tc(projectedCenter());
	FixedAVector<float, 2> ac(a.projectedCenter());
	float dsq = (projectedRadius_ + a.projectedRadius()) *
			(projectedRadius_ + a.projectedRadius());
	if ((tc - ac).sqlength() > dsq)
		return 0;
	return (getCoord()->xyz[2] < a.getCoord()->xyz[2]) ? -1 : 1;
}

//
// Check if this atom covers or is covered by the given
// bond in the projected image
//
int
Atom::depthCompare(const Camera &camera, Bond &bond)
{
	// If bounding boxes do not overlap, they do not overlap
	if (!overlaps(bond))
		return 0;

	// Get parameters
	const float *center = projectedCenter();
	const float radius = projectedRadius();
	const float *v0 = bond.projectedEnd(0);
	const float *v1 = bond.projectedEnd(1);
	const float width = bond.projectedWidth();

	// Compute the closest point on the line segment to the circle
	float dv1v0[2], dcv0[2];
	for (int i = 0; i < 2; i++) {
		dv1v0[i] = v1[i] - v0[i];
		dcv0[i] = center[i] - v0[i];
	}
	float t = (dcv0[0] * dv1v0[0] + dcv0[1] * dv1v0[1]) /
			(dv1v0[0] * dv1v0[0] + dv1v0[1] * dv1v0[1]);
	if (t < 0)
		t = 0;
	else if (t > 1)
		t = 1;
	float bpt[2] = { dv1v0[0] * t + v0[0], dv1v0[1] * t + v0[1] };

	// Compute the distance from the point to the circle center.
	// If it's more than radius + width, then they do not overlap.
	float dbc[2] = { bpt[0] - center[0], bpt[1] - center[1] };
	float lsq = dbc[0] * dbc[0] + dbc[1] * dbc[1];
	if (lsq > (radius + width) * (radius + width))
		return 0;

	float bcoord[3];
	if (bond.backProjection(camera, bpt, bcoord) < 0)
		return 0;
	return getCoord()->xyz[2] < bcoord[2] ? -1 : 1;
}

//
// Convert a projected point back into a Cartesian coordinate
// that falls on this atom
// The algorithm used is the intersection of a sphere and
// a parametric line in 3-space.
//
int
Atom::backProjection(const Camera &camera, const float p[2], float answer[3])
{
	const float *eye = camera.eye();
	float pt[3] = { p[0], p[1], eye[2] - camera.near() };
	const MolPos &center = getCoord()->xyz;
	float a = 0;
	float b = 0;
	float c = -radius_ * radius_;
	float dpe[3];
	for (int i = 0; i < 3; i++) {
		dpe[i] = pt[i] - eye[i];
		float dec = eye[i] - center[i];
		a += dpe[i] * dpe[i];
		b += dpe[i] * dec;
		c += dec * dec;
	}
	b = b * 2;
	float t;
	if (a == 0) {
		if (b == 0)
			return -1;
		t = -c / b;
	}
	else {
		float disc = b * b - 4 * a * c;
		if (disc < 0)
			return -1;
		t = (-b - sqrt(disc)) / (2 * a);
	}
	for (i = 0; i < 3; i++)
		answer[i] = dpe[i] * t + eye[i];
	return 0;
}

//
// Draw the atom in PostScript to stdout
//
virtual void
Atom::drawThis()
{
	printf("%.3f %.3f %.3f c\n", projectedCenter_[0], projectedCenter_[1],
		projectedRadius_);
	if (rgb_[0] == 1 && rgb_[1] == 1 && rgb_[2] == 1)
		printf("0 0 0 sc da\n");
	else
		printf("%.3f %.3f %.3f sc da\n", rgb_[0], rgb_[1], rgb_[2]);
}
