#include <stdio.h>
#include <math.h>
#include "Mol/Molecule.h"

inline float
max(float f1, float f2)
{
	return f1 > f2 ? f1 : f2;
}

inline float
square(float f)
{
	return f * f;
}

//
// Compute the terminating points for the bond
// Also compute bounding box
//
void
Bond::computeProjection(const Camera &camera, float sFrac)
{
	Atom *a0 = lookupAtom(0);
	Atom *a1 = lookupAtom(1);
	MolPos &c0 = a0->getCoord()->xyz;
	MolPos &c1 = a1->getCoord()->xyz;

	Geom3d::Vector d = c0 - c1;
	d.normalize();
	zFraction_ = fabs(d[2]);
	projectedZFraction_ = zFraction_ * sFrac;

	float chopFraction = sqrt(1 - sFrac * sFrac) * 0.9;
	float r = (a0->radius() + a1->radius()) / 2;
	float z = (end_[0][2] + end_[1][2]) / 2;
	width_ = sFrac * r;
	end_[0] = c0 - d * chopFraction * a0->radius();
	end_[1] = c1 + d * chopFraction * a1->radius();

	projectedWidth_ = camera.projectLength(width_, z);
	camera.project(end_[0][0], end_[0][1], end_[0][2], projectedEnd_[0]);
	camera.project(end_[1][0], end_[1][1], end_[1][2], projectedEnd_[1]);

	float dx = -(projectedEnd_[1][1] - projectedEnd_[0][1]);
	float dy = projectedEnd_[1][0] - projectedEnd_[0][0];
	float sdl = projectedWidth_ / sqrt(dx * dx + dy * dy);
	dx *= sdl;
	dy *= sdl;
	addPoint(projectedEnd_[0][0] + dx, projectedEnd_[0][1] + dy);
	addPoint(projectedEnd_[0][0] - dx, projectedEnd_[0][1] - dy);
	addPoint(projectedEnd_[1][0] + dx, projectedEnd_[1][1] + dy);
	addPoint(projectedEnd_[1][0] - dx, projectedEnd_[1][1] - dy);
}

//
// Check if this bond covers or is covered by the given
// bond in the projected image
//
int
Bond::depthCompare(const Camera &camera, Bond &b)
{
	if (!overlaps(b))
		return 0;

	const float *s0 = projectedEnd(0);
	const float *e0 = projectedEnd(1);
	const float *s1 = b.projectedEnd(0);
	const float *e1 = b.projectedEnd(1);
	const float d0[2] = { e0[0] - s0[0], e0[1] - s0[1] };
	const float d1[2] = { e1[0] - s1[0], e1[1] - s1[1] };
	float denom = d0[0] * d1[1] - d0[1] * d1[0];
	if (denom == 0) {
		// Parallel projection.  First check if the lines overlap
		// at all (lines have widths).  If not, just return.
		// Otherwise, return the one in front.
		float t = (d1[0] * (s1[0] - s0[0]) + d1[1] * (s1[1] - s0[1])) /
				(square(d0[0]) + square(d0[1]));
		float x = d0[0] * t + s0[0];
		float y = d0[1] * t + s0[1];
		float dsq = square(x - s1[0]) + square(y - s1[1]);
		float wsq = square(projectedWidth() + b.projectedWidth());
		if (dsq < wsq)
			return 0;
		const MolPos &cs0 = end(0);
		const MolPos &ce0 = end(1);
		const MolPos &cs1 = b.end(0);
		const MolPos &ce1 = b.end(1);
		float z0 = max(cs0[2], ce0[2]);
		float z1 = max(cs1[2], ce1[2]);
		return z0 < z1 ? -1 : 1;
	}

	// Compute the parametric coordinates of the intersection.
	// If either one falls outside the segment, then the segments
	// do not overlap.  (This is not strictly true, as each segment
	// also has a width.  However, with molecular data, this test
	// is probably fine.)
	float t = (s1[0] * d1[1] - s1[1] * d1[0] -
			s0[0] * d1[1] + s0[1] * d1[0]) / denom;
	if (t < 0 || t > 1)
		return 0;
	float u = (d0[0] * t + s0[0] - s1[0]) / d1[0];
	if (u < 0 || u > 1)
		return 0;

	// Compute the coordinates and compare z
	float v0[2] = { d0[0] * t + s0[0], d0[1] * t + s0[1] };
	float c0[3];
	if (backProjection(camera, v0, c0) < 0)
		return 0;
	float v1[2] = { d1[0] * u + s1[0], d1[1] * u + s1[1] };
	float c1[3];
	if (b.backProjection(camera, v1, c1) < 0)
		return 0;
	return c0[2] < c1[2] ? -1 : 1;
}

//
// Convert a projected point back into a Cartesian coordinate
// that falls on this bond
// The algorithm used is the intersection of two parametric lines
// There should actually be a check after the computation of "t"
// to make sure that there is a solution, but we're cheating here.
//
int
Bond::backProjection(const Camera &camera, const float p[2], float answer[3])
{
	const float *eye = camera.eye();
	float pt[3] = { p[0], p[1], eye[2] - camera.near() };
	const MolPos &v0 = lookupAtom(0)->getCoord()->xyz;
	const MolPos &v1 = lookupAtom(1)->getCoord()->xyz;
	int i, j;
	float det;
	for (i = 0; i < 3; i++)
		for (j = i + 1; j < 3; j++) {
			det = (v1[i] - v0[i]) * (pt[j] - eye[j]) -
				(v1[j] - v0[j]) * (pt[i] - eye[i]);
			if (det != 0)
				goto found;
		}
	return -1;
found:
	float num = (pt[j] - eye[j]) * (eye[i] - v0[i]) -
			(pt[i] - eye[i]) * (eye[j] - v0[j]);
	float t = num / det;
	for (int n = 0; n < 3; n++)
		answer[n] = (v1[n] - v0[n]) * t + v0[n];
	return 0;
}

//
// Draw the bond in PostScript to stdout
//
virtual void
Bond::drawThis()
{
	float dx = projectedEnd_[1][0] - projectedEnd_[0][0];
	float dy = projectedEnd_[1][1] - projectedEnd_[0][1];
	float t = sqrt(dx * dx + dy * dy);
	dx /= t;
	dy /= t;
	float angle = atan2(dy, dx) * 180 / M_PI;
	float zp = (1 + zFraction_ * zFraction_) / (2 * zFraction_);
	float radius = projectedWidth_ * zp;
	float da = asin(projectedWidth_ / radius) * 180 / M_PI;

	float cx = projectedEnd_[1][0] - radius * dx;
	float cy = projectedEnd_[1][1] - radius * dy;
	printf("%.3f %.3f %.3f %.3f %.3f a\n", cx, cy, radius,
		angle - da, angle + da);

	angle += 180;
	cx = projectedEnd_[0][0] + radius * dx;
	cy = projectedEnd_[0][1] + radius * dy;
	printf("%.3f %.3f %.3f %.3f %.3f a 0 sg db\n", cx, cy, radius,
		angle - da, angle + da);
}
