#include <stdio.h>
#include <stdlib.h>
#include <GetOpt.h>
#include "Mol/Molecule.h"
#include <pdbio/PDBio.h>
#include "Camera.h"
#include "BBox.h"

#define	RESOLUTION	72
inline float min(float x, float y) { return x < y ? x : y; }

typedef PDBio<Bond,Atom,Residue,Coord,CoordSet,Molecule> IOpdb;

static Camera camera;
static BBox bbox;
static int lastAtomSerial = -1;

static bool handleUserRecords(PDB *pdb, Molecule *mol, const intVoidPMap *map);
static void printProlog(int eps, int frame, float margin, int bFill, int sFill);
static void printEpilog(int eps, int frame);

//
// otfbs:
//	Sample program using OTF classes
//	Reads a PDB file and produces ball-and-stick PostScript image
//
int
main(int argc, char **argv)
{
	//
	// Initialize variables
	//
	int generateEPS = false;
	int frame = false;
	float margin = 2;
	float ballFraction = 0.25;
	float stickFraction = 0.5;
	int ballFill = false;
	int stickFill = false;

	//
	// Process command line arguments
	//
	GetOpt opt(argc, argv, "b:efm:s:F:");
	int o;
	while ((o = opt()) != EOF)
		switch (o) {
		  case 'b':
			ballFraction = atof(opt.optarg);
			if (ballFraction < 0 || ballFraction > 1) {
				fprintf(stderr, "%s: fraction must be between"
						" zero and one\n", argv[0]);
				return 1;
			}
			break;
		  case 'e':
			generateEPS = !generateEPS;
			break;
		  case 'f':
			frame = !frame;
			break;
		  case 'm':
			margin = atof(opt.optarg);
			break;
		  case 's':
			stickFraction = atof(opt.optarg);
			if (stickFraction < 0 || stickFraction > 1) {
				fprintf(stderr, "%s: fraction must be between"
						" zero and one\n", argv[0]);
				return 1;
			}
			break;
		  case 'F':
			const char *arg = opt.optarg;
			if (arg != NULL)
				do
					switch (*arg) {
					  case 'b':
					  case 'B':
						ballFill = !ballFill;
						break;
					  case 's':
					  case 'S':
						stickFill = !stickFill;
						break;
					}
				while (*arg++ != '\0');
			break;
		}
	const char *input = NULL;
	switch (argc - opt.optind) {
	  case 0:
		break;
	  case 1:
		input = argv[opt.optind];
		break;
	  default:
		fprintf(stderr, "Usage: %s [-efm] [-b fraction] [-s fraction]"
				" [-F[bs]] PDB_input\n", argv[0]);
		return 1;
	}

	//
	// Read PDB file
	//
	IOpdb pdbio;
	pdbio.whatToDo().set(IOpdb::HYDROGEN_BONDS);
	pdbio.whatToDo().set(IOpdb::SALT_BRIDGES);
	pdbio.readMask().set(PDB::USER);
	pdbio.readMask().set(PDB::ATOM);
	pdbio.readMask().set(PDB::HETATM);
	pdbio.readMask().set(PDB::END);
	pdbio.readMask().set(PDB::ENDMDL);
	pdbio.readPDB(handleUserRecords);
	pdbio.readPDBfile(input);
	if (input == NULL)
		input = "standard input";
	if (!pdbio.ok()) {
		fprintf(stderr, "%s: %s: %s\n", argv[0], input, pdbio.error());
		return 1;
	}
	if (!camera.eyeSet()) {
		fprintf(stderr, "%s: %s: no EYEPOS record\n", argv[0], input);
		return 1;
	}
	if (!camera.windowSet()) {
		fprintf(stderr, "%s: %s: no WINDOW record\n", argv[0], input);
		return 1;
	}

	//
	// Print prolog
	//
	printProlog(generateEPS, frame, margin, ballFill, stickFill);

	//
	// Compute bounding boxes
	//
	Molecule *mol = pdbio.molecule();
	CIter<Atom> atomList = mol->citerAtom();
	for (CIter<Atom> ai = atomList; ai.ok(); ai.next())
		ai->computeBBox(camera, ballFraction);
	CIter<Bond> bondList = mol->citerBond();
	for (CIter<Bond> bi = bondList; bi.ok(); bi.next())
		bi->computeProjection(camera, stickFraction);

	//
	// Compute partial ordering
	//
	for (ai = atomList; ai.ok(); ai.next()) {
		Atom &atom = *ai;
		CIter<Atom> ai2 = ai;
		const String &nm = ai->string();
		for (ai2.next(); ai2.ok(); ai2.next()) {
			const String &nm2 = ai2->string();
			switch (atom.depthCompare(camera, *ai2)) {
			  case 0:
				break;
			  case 1:
				if (atom.isAfter((Atom *) ai2) < 0)
					fprintf(stderr, "CYCLE: %s and %s\n",
						nm2.chars(), nm.chars());
#ifdef REPORT_ORDER
				else
					fprintf(stderr, "%s is after %s\n",
						nm.chars(), nm2.chars());
#endif
				break;
			  case -1:
				if (ai2->isAfter(&atom) < 0)
					fprintf(stderr, "CYCLE: %s and %s\n",
						nm2.chars(), nm.chars());
#ifdef REPORT_ORDER
				else
					fprintf(stderr, "%s is after %s\n",
						nm2.chars(), nm.chars());
#endif
				break;
			}
		}
		for (CIter<Bond> bi2 = bondList; bi2.ok(); bi2.next()) {
			const String &nm2 = bi2->string();
			switch (atom.depthCompare(camera, *bi2)) {
			  case 0:
				break;
			  case 1:
				if (atom.isAfter((Bond *) bi2) < 0)
					fprintf(stderr, "CYCLE: %s and %s\n",
						nm2.chars(), nm.chars());
#ifdef REPORT_ORDER
				else
					fprintf(stderr, "%s is after %s\n",
						nm.chars(), nm2.chars());
#endif
				break;
			  case -1:
				if (bi2->isAfter(&atom) < 0)
					fprintf(stderr, "CYCLE: %s and %s\n",
						nm2.chars(), nm.chars());
#ifdef REPORT_ORDER
				else
					fprintf(stderr, "%s is after %s\n",
						nm2.chars(), nm.chars());
#endif
				break;
			}
		}
	}
	for (bi = bondList; bi.ok(); bi.next()) {
		Bond &bond = *bi;
		CIter<Bond> bi2 = bi;
		const String &nm = bi->string();
		for (bi2.next(); bi2.ok(); bi2.next()) {
			const String &nm2 = bi2->string();
			switch (bond.depthCompare(camera, *bi2)) {
			  case 0:
				break;
			  case 1:
				if (bond.isAfter((Bond *) bi2) < 0)
					fprintf(stderr, "CYCLE: %s and %s\n",
						nm.chars(), nm2.chars());
#ifdef REPORT_ORDER
				else
					fprintf(stderr, "%s is after %s\n",
						nm.chars(), nm2.chars());
#endif
				break;
			  case -1:
				if (bi2->isAfter(&bond) < 0)
					fprintf(stderr, "CYCLE: %s and %s\n",
						nm2.chars(), nm.chars());
#ifdef REPORT_ORDER
				else
					fprintf(stderr, "%s is after %s\n",
						nm2.chars(), nm.chars());
#endif
				break;
			}
		}
	}

	//
	// Draw atoms and bonds
	//
	int nUndrawn = atomList.numRemaining() + bondList.numRemaining();
	while (nUndrawn > 0) {
		int n = 0;
		for (ai = atomList; ai.ok(); ai.next())
			if (!ai->drawn() && !ai->draw())
				n++;
		for (bi = bondList; bi.ok(); bi.next())
			if (!bi->drawn() && !bi->draw())
				n++;
		if (n == nUndrawn) {
			fprintf(stderr, "%s: (warning) geometric deadlock: "
				"%d atoms and bonds were not drawn\n",
				argv[0], nUndrawn);
			break;
		}
		nUndrawn = n;
	}

	//
	// Finish drawing
	//
	printEpilog(generateEPS, frame);
	return 0;
}

//
// handleUserRecords:
//	Process user records
//
static bool
handleUserRecords(PDB *pdb, Molecule *mol, const intVoidPMap *map)
{
	switch (pdb->type()) {
	  case PDB::ATOM:
	  case PDB::HETATM:
		lastAtomSerial = pdb->atom.serialNum;
		break;
	  case PDB::END:
	  case PDB::ENDMDL:
		lastAtomSerial = -1;
		break;
	  case PDB::USER:
	  {
		char buf[80];
		(void) strcpy(buf, pdb->user.text);
		char *p = strtok(buf, " \t");
		if (strcmp(p, "EYEPOS") == 0) {
			float eye[3];
			eye[0] = atof(strtok(NULL, " \t"));
			eye[1] = atof(strtok(NULL, " \t"));
			eye[2] = atof(strtok(NULL, " \t"));
			camera.setEye(eye);
			break;
		}
		else if (strcmp(p, "WINDOW") == 0) {
			float left = atof(strtok(NULL, " \t"));
			float right = atof(strtok(NULL, " \t"));
			float bottom = atof(strtok(NULL, " \t"));
			float top = atof(strtok(NULL, " \t"));
			float near = atof(strtok(NULL, " \t"));
			float far = atof(strtok(NULL, " \t"));
			camera.setWindow(left, right, bottom, top, near, far);
			break;
		}
		Atom *atom = (Atom *) (*map)[lastAtomSerial];
		if (atom == NULL)
			break;
		if (strcmp(p, "RADIUS") == 0)
			atom->setRadius(atof(strtok(NULL, " \t")));
		else if (strcmp(p, "COLOR") == 0) {
			(void) strtok(NULL, " \t");
			float r = atof(strtok(NULL, " \t"));
			float g = atof(strtok(NULL, " \t"));
			float b = atof(strtok(NULL, " \t"));
			atom->setRGB(r, g, b);
		}
		break;
	  }
	  default:
		break;
	}
	return false;
}

//
// printProlog:
//	Print the PostScript prolog
//
static void
printProlog(int eps, int frame, float margin, int bFill, int sFill)
{
	//
	// Print the generic part
	//
	fputs("%!PS-Adobe-2.1 EPSF-2.0\n", stdout);
	fputs("%%Title: (molecule)\n", stdout);
	fputs("%%Creator: (bs)\n", stdout);
	fputs("%%ColorUsage: Color\n", stdout);
	if (eps)
		fputs("%%BoundingBox: (atend)\n", stdout);
	else
		fputs("%%Pages: 1\n", stdout);
	fputs("%%EndComments\n", stdout);
	fputs("/m { moveto } bind def\n", stdout);
	fputs("/l { lineto } bind def\n", stdout);
	fputs("/s { stroke } bind def\n", stdout);
	fputs("/f { fill } bind def\n", stdout);
	fputs("/a { arc } bind def\n", stdout);
	fputs("/c { 0 360 a } bind def\n", stdout);
	fputs("/sg { setgray } bind def\n", stdout);
	fputs("/sc { setrgbcolor } bind def\n", stdout);
	printf("/da { closepath %s } bind def\n",
		bFill ? "fill" : "gsave 1 sg fill grestore s");
	printf("/db { closepath %s } bind def\n",
		sFill ? "fill" : "gsave 1 sg fill grestore s");
	fputs("%%EndProlog\n", stdout);
	if (!eps) {
		fputs("%%Page: 1 1\n", stdout);
		printf("0 setgray\n");
		printf("72 72 translate\n");
	}

	//
	// Compute the input-specific prolog
	//
	float xrange = camera.right() - camera.left();
	float yrange = camera.top() - camera.bottom();
	float scale;
	if (eps)
		scale = (RESOLUTION * 10) / min(xrange, yrange);
	else {
		float xscale = (RESOLUTION * 6.5 - margin * 2) / xrange;
		float yscale = (RESOLUTION * 9.0 - margin * 2) / yrange;
		scale = min(xscale, yscale);
	}
	bbox.setBounds(0, xrange * scale + margin * 2 + 0.5,
			0, yrange * scale + margin * 2 + 0.5);
	printf("%.3f %.3f m\n", bbox.left(), bbox.bottom());
	printf("%.3f %.3f l\n", bbox.left(), bbox.top());
	printf("%.3f %.3f l\n", bbox.right(), bbox.top());
	printf("%.3f %.3f l\n", bbox.right(), bbox.bottom());
	fputs("clip newpath\n", stdout);
	if (frame)
		printf("gsave\n");
	printf("%.3f dup translate\n", margin);
	printf("%.3f dup scale\n", scale);
	printf("currentlinewidth %.3f div 2 div setlinewidth\n", scale);
	printf("%.3f %.3f translate\n", -(camera.eye()[0] + camera.left()),
		-(camera.eye()[1] + camera.bottom()));
}

//
// printEpilog:
//	Print PostScript epilog
//
static void
printEpilog(int eps, int frame)
{

	//
	// Generate PostScript page trailer and epilog
	//
	if (frame) {
		printf("grestore\n");
		printf("%.3f %.3f m\n", bbox.left(), bbox.bottom());
		printf("%.3f %.3f l\n", bbox.left(), bbox.top());
		printf("%.3f %.3f l\n", bbox.right(), bbox.top());
		printf("%.3f %.3f l\n", bbox.right(), bbox.bottom());
		fputs("closepath stroke\n", stdout);
	}
	if (!eps)
		printf("showpage\n");
	fputs("%%Trailer\n", stdout);
	if (eps)
		printf("%%%%BoundingBox: %.3f %.3f %.3f %.3f\n",
			bbox.left(), bbox.bottom(), bbox.right(), bbox.top());
}
