#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <GetOpt.h>
#include "Mol/Molecule.h"
#include "mol/Geom3d.h"
#include <pdbio/PDBio.h>

typedef PDBio<Bond,Atom,Residue,Coord,CoordSet,Molecule> IOpdb;

static void computeAngles(Residue *prev, Residue *r);

inline float
radianToDegree(float r) {
	return r / M_PI * 180;
}

//
// phipsi:
//	Reads a PDB file and print the phi, psi and omega angles
//	Also print chi1 and chi2
//
int
main(int argc, char **argv)
{
	//
	// Initialize variables
	//
	bool verbose = false;

	//
	// Process command line arguments
	//
	GetOpt opt(argc, argv, "v");
	int o;
	while ((o = opt()) != EOF)
		switch (o) {
		  case 'v':
			verbose = true;
			break;
		}
	const char *input = NULL;
	switch (argc - opt.optind) {
	  case 0:
		break;
	  case 1:
		input = argv[opt.optind];
		break;
	  default:
		fprintf(stderr, "Usage: %s [-v] [PDB_input]\n", argv[0]);
		return 1;
	}

	//
	// Read PDB file
	//
	IOpdb pdbio;
	pdbio.readPDBfile(input);
	if (input == NULL)
		input = "standard input";
	if (!pdbio.ok()) {
		fprintf(stderr, "%s: %s: %s\n", argv[0], input, pdbio.error());
		return 1;
	}

	//
	// Compute bounding boxes
	//
	Molecule *mol = pdbio.molecule();
	Residue *prev = NULL;
	for (CIter<Residue> ri = mol->citerResidue(); ri.ok(); ri.next()) {
		Residue *r = ri;
		if (prev != NULL)
			computeAngles(prev, r);
		prev = r;
	}
	printf("\t\t     Phi      Psi    Omega     Chi1     Chi2\n");
	for (ri = mol->citerResidue(); ri.ok(); ri.next()) {
		const Residue *r = ri;
		printf("%6s (%s)\t", r->id().chars(), r->type().chars());
		if (r->hasPhi())
			printf("%8.1f", radianToDegree(r->phi()));
		else
			printf("      --");
		printf(" ");
		if (r->hasPsi())
			printf("%8.1f", radianToDegree(r->psi()));
		else
			printf("      --");
		printf(" ");
		if (r->hasOmega())
			printf("%8.1f", radianToDegree(r->omega()));
		else
			printf("      --");
		printf(" ");
		if (r->hasChi1())
			printf("%8.1f", radianToDegree(r->chi1()));
		else
			printf("      --");
		printf(" ");
		if (r->hasChi2())
			printf("%8.1f", radianToDegree(r->chi2()));
		else
			printf("      --");
		printf("\n");
	}

	//
	// Clean up and exit
	//
	return 0;
}

static char *xg[] = {
	"CG",	"SG",	"CG1",	"OG1",	NULL
};

static char *xd[] = {
	"CD",	"OD1",	"ND1",	"CD1",	"SD",	NULL
};

static void
computeAngles(Residue *prev, Residue *r)
{
	if (!r->id().sameChain(prev->id()))
		return;
	const Atom *prevNatom = prev->lookupAtom("N");
	if (prevNatom == NULL)
		return;
	const Atom *prevCAatom = prev->lookupAtom("CA");
	if (prevCAatom == NULL)
		return;
	const Atom *prevCatom = prev->lookupAtom("C");
	if (prevCatom == NULL)
		return;
	const Atom *Natom = r->lookupAtom("N");
	if (Natom == NULL)
		return;
	const Atom *CAatom = r->lookupAtom("CA");
	if (CAatom == NULL)
		return;
	const Atom *Catom = r->lookupAtom("C");
	if (Catom == NULL)
		return;
	const MolPos prevN = prevNatom->getCoord()->xyz;
	const MolPos prevCA = prevCAatom->getCoord()->xyz;
	const MolPos prevC = prevCatom->getCoord()->xyz;
	const MolPos N = Natom->getCoord()->xyz;
	const MolPos CA = CAatom->getCoord()->xyz;
	const MolPos C = Catom->getCoord()->xyz;
	prev->setPsi(Geom3d::dihedral(prevN, prevCA, prevC, N));
	prev->setOmega(Geom3d::dihedral(prevCA, prevC, N, CA));
	r->setPhi(Geom3d::dihedral(prevC, N, CA, C));

	const Atom *CBatom = r->lookupAtom("CB");
	if (CBatom == NULL)
		return;
	const MolPos CB = CBatom->getCoord()->xyz;

	const Atom *xGatom = NULL;
	const char *const *cpp;
	for (cpp = xg; *cpp != NULL; cpp++)
		if ((xGatom = r->lookupAtom(*cpp)) != NULL)
			break;
	if (xGatom == NULL)
		return;
	const MolPos xG = xGatom->getCoord()->xyz;
	r->setChi1(Geom3d::dihedral(N, CA, CB, xG));

	const Atom *xDatom = NULL;
	for (cpp = xd; *cpp != NULL; cpp++)
		if ((xDatom = r->lookupAtom(*cpp)) != NULL)
			break;
	if (xDatom == NULL)
		return;
	const MolPos xD = xDatom->getCoord()->xyz;
	r->setChi2(Geom3d::dihedral(CA, CB, xG, xD));
}
