#include <stdlib.h>
#include <string.h>
#define String GL_String
#define Coord GL_Coord
#include <gl/gl.h>
#include <gl/device.h>
#undef String
#undef Coord

#include "Mol/Molecule.h"
#include <pdbio/PDBio.h>
#include <GetOpt.h>

typedef PDBio<Bond, Atom, Residue, Coord, CoordSet, Molecule>	IOpdb;
Molecule	*cur_mol;
CIter<CoordSet>	cur_cs;

static void	set_bbox(Molecule *);
static void	set_view_params();
static void	display();
static void	build_display_list(Molecule *);

static long	width = 512, height = 512;
static Object	listId;

int
main(int argc, char **argv)
{
	GetOpt	getopt(argc, argv, "h:w:");
	int	c;
	IOpdb	pdb_io;

	pdb_io.whatToDo().set(IOpdb::SALT_BRIDGES);
	pdb_io.whatToDo().set(IOpdb::HYDROGEN_BONDS);

	while ((c = getopt()) != EOF) switch (c) {
	case 'h': height = atoi(getopt.optarg); break;
	case 'w': width = atoi(getopt.optarg); break;
	default:
		goto usage;
	}
	if (argc > getopt.optind + 1) {
usage:
		cerr <<  "usage: " << argv[0] << " [pdb-file]" << endl;
		return 1;
	}

	if (argc == getopt.optind)
		pdb_io.readPDBfile(NULL);
	else
		pdb_io.readPDBfile(argv[getopt.optind]);
	if (!pdb_io.okay()) {
		cerr <<  argv[0] << ": " << pdb_io.error() << endl;
		return 1;
	}
	if (pdb_io.molecules().length() != 1) {
		cerr << argv[0]
		<< ": can only handle PDB files with coordinate set ensembles"
			<< endl;
		return 1;
	}

	cur_mol = pdb_io.molecule();
	cur_cs = cur_mol->citerCoordSet();
	if (!cur_cs.ok()) {
		cerr << argv[0] << ": no coordinate sets!?" << endl;
		return 1;
	}
	cur_mol->activeCoordSet(cur_cs);

	//
	// Open window based on viewport size and set default parameters
	//
	foreground();
	prefsize(width, height);
	winopen("ensembview");
	winconstraints();
	doublebuffer();
	RGBmode();
	gconfig();
	subpixel(TRUE);
	linesmooth(SML_ON);
	blendfunction(BF_SA, BF_MSA);
	listId = genobj();

	set_bbox(cur_mol);
	set_view_params();

	build_display_list(cur_mol);
	display();

	qdevice(LEFTMOUSE);
	for (;;) {
		short	data;
		long	device;

		device = qread(&data);
		switch (device) {
		case LEFTMOUSE:		// advance on mouse up
			if (data)
				break;
			cur_cs.next();
			if (!cur_cs.ok())
				return 0;
			cur_mol->activeCoordSet(cur_cs);
			build_display_list(cur_mol);
			display();
			break;
		case REDRAW:
			set_view_params();
			display();
			break;
		}
	}
}

static void
calculate_cs_bbox(CoordSet *cs, float bbox[2][3])
{
	CIter<Coord> ci = cs->citerCoord();
	if (!ci.ok()) {
		bbox[0][0] = bbox[0][1] = bbox[0][2] = 0.0f;
		bbox[1][0] = bbox[1][1] = bbox[1][2] = 0.0f;
		return;
	}
	bbox[0][0] = bbox[1][0] = ci->xyz[0];
	bbox[0][1] = bbox[1][1] = ci->xyz[1];
	bbox[0][2] = bbox[1][2] = ci->xyz[2];
	for (ci.next(); ci.ok(); ci.next()) {
		if (ci->xyz[0] < bbox[0][0])
			bbox[0][0] = ci->xyz[0];
		else if (ci->xyz[0] > bbox[1][0])
			bbox[1][0] = ci->xyz[0];
		if (ci->xyz[1] < bbox[0][1])
			bbox[0][1] = ci->xyz[1];
		else if (ci->xyz[1] > bbox[1][1])
			bbox[1][1] = ci->xyz[1];
		if (ci->xyz[2] < bbox[0][2])
			bbox[0][2] = ci->xyz[2];
		else if (ci->xyz[2] > bbox[1][2])
			bbox[1][2] = ci->xyz[2];
	}
}

static float	win[6], eye[3], at[3];
static float	bbox[2][3];

static void
set_bbox(Molecule *m)
{
	float	next_bbox[2][3];

	CIter<CoordSet> csi = m->citerCoordSet();
	if (!csi.ok())
		return;
	calculate_cs_bbox(csi, bbox);
	for (csi.next(); csi.ok(); csi.next()) {
		calculate_cs_bbox(csi, next_bbox);
		if (next_bbox[0][0] < bbox[0][0])
			bbox[0][0] = next_bbox[0][0];
		if (next_bbox[0][1] < bbox[0][1])
			bbox[0][1] = next_bbox[0][1];
		if (next_bbox[0][2] < bbox[0][2])
			bbox[0][2] = next_bbox[0][2];
		if (next_bbox[1][0] > bbox[1][0])
			bbox[1][0] = next_bbox[1][0];
		if (next_bbox[1][1] > bbox[1][1])
			bbox[1][1] = next_bbox[1][1];
		if (next_bbox[1][2] > bbox[1][2])
			bbox[1][2] = next_bbox[1][2];
	}
}

static void
set_view_params()
{
	reshapeviewport();
	getsize(&width, &height);

	at[0] = (bbox[0][0] + bbox[1][0]) / 2;
	at[1] = (bbox[0][1] + bbox[1][1]) / 2;
	at[2] = (bbox[0][2] + bbox[1][2]) / 2;
	eye[0] = at[0];
	eye[1] = at[1];
	eye[2] = bbox[1][2];

	float win_width = bbox[1][0] - bbox[0][0];
	float win_height = bbox[1][1] - bbox[0][1];
	float win_depth = bbox[1][2] - bbox[0][2];
	float win_aspect = (float) width / (float) height;
	// compensate for aspect ratio
	if (win_width < win_aspect * win_height)
		win_width = win_aspect * win_height;
	else
		win_height = win_width / win_aspect;

	win[0] = -.55 * win_width;
	win[1] = -win[0];
	win[2] = -.55 * win_height;
	win[3] = -win[2];
	win[4] = 0;
	win[5] = win_depth;
}

static void
build_display_list(Molecule *m)
{
	makeobj(listId);
	cpack(0xff808080);
	for (CIter<Bond> bi = m->citerBond(); bi.ok(); bi.next()) {
		Coord	*c0 = bi->lookupAtom(0)->getCoord(),
			*c1 = bi->lookupAtom(1)->getCoord();
		bgnline();
		v3f(c0->xyz);
		v3f(c1->xyz);
		endline();
	}
	closeobj();
}

void
loadIdentity()
{
	static Matrix im = {
				{ 1, 0, 0, 0 },
				{ 0, 1, 0, 0 },
				{ 0, 0, 1, 0 },
				{ 0, 0, 0, 1 }
			};
	loadmatrix(im);
}

//
// Display function
//
static void
display(void)
{
	//
	// Draw background
	//
	cpack(0);
	clear();

	//
	// Set the viewing parameters
	//
	mmode(MPROJECTION);
	loadIdentity();
	ortho(win[0], win[1], win[2], win[3], win[4], win[5]);
	mmode(MVIEWING);
	loadIdentity();
	lookat(eye[0], eye[1], eye[2], at[0], at[1], at[2], 0);
	callobj(listId);

	//
	// Annotate scene with model number (MolCoordSetId)
	//
	mmode(MPROJECTION);
	loadIdentity();
	ortho2(0, width, 0, height);
	mmode(MVIEWING);
	loadIdentity();
	cpack(0xff0000ff);
	cmov2i(5, 5);
	charstr(cur_cs->id().chars());

	swapbuffers();
}
