#include "Mol/Molecule.h"
#include <pdbio/PDBio.h>
#include <GetOpt.h>
#include "math.h"

typedef PDBio<Bond, Atom, Residue, Coord, CoordSet, Molecule>	IOpdb;

static void	printAtom(Atom *a, void *closure);
static void	printBond(Bond *a, void *closure);
static void	printResidue(Residue *a, void *closure);

int
main(int argc, char **argv)
{
	GetOpt	getopt(argc, argv, "cdhpst:");
	int	c;
	bool	debug = false;
	bool	pdb_output = false;
	IOpdb	pdb_io(longtoBitSet(1 << IOpdb::ATOMS));

	while ((c = getopt()) != EOF) switch (c) {
	case 'd': debug = true; break;
	case 'p': pdb_output = true; break;
	case 'c': pdb_io.whatToDo().set(IOpdb::COVALENT_BONDS); break;
	case 'h': pdb_io.whatToDo().set(IOpdb::HYDROGEN_BONDS); break;
	case 's': pdb_io.whatToDo().set(IOpdb::SALT_BRIDGES); break;
	case 't': IOpdb::bondLengthTolerance(atof(getopt.optarg)); break;
	default: goto usage;
	}
	if (argc > getopt.optind + 1) {
usage:
		fprintf(stderr, "usage: %s [-h] [-s] [pdb-file]\n", argv[0]);
		return 1;
	}

	if (argc == getopt.optind)
		pdb_io.readPDBfile(NULL);
	else
		pdb_io.readPDBfile(argv[getopt.optind]);
	if (!pdb_io.okay()) {
		fprintf(stderr, "%s\n", pdb_io.error());
		return 1;
	}

	if (debug) {
		Molecule *m = pdb_io.molecule();
		fprintf(stderr, "\nAtoms:\n");
		for (CIter<Atom> ai = m.citerAtom(); ai.ok(); ai.next())
			printAtom(ai, NULL);
		fprintf(stderr, "\nBonds:\n");
		for (CIter<Bond> bi = m.citerBond(); bi.ok(); bi.next())
			printBond(bi, NULL);
		fprintf(stderr, "\nResidues:\n");
		for (CIter<Residue> ri = m.citerResidue(); ri.ok(); ri.next())
			printResidue(ri, NULL);
	}
	if (pdb_output)
		pdb_io.writePDBfile(NULL);
	return 0;
}

void
printAtom(Atom *a, void *closure)
{
	fprintf(stderr, "'%s'\t%d\t%s\n", a->name().chars(), a->atomicNum(),
						a->residue()->id().chars());
}

void
printBond(Bond *b, void *closure)
{
	Atom	*a0 = b->lookupAtom(0), *a1 = b->lookupAtom(1);
	fprintf(stderr, ":%s@%s ->", a0->residue()->id().chars(),
							a0->name().chars());
	fprintf(stderr, " :%s@%s", a1->residue()->id().chars(),
							a1->name().chars());
	Coord	*c0 = a0->getCoord();
	Coord	*c1 = a1->getCoord();
	float dist_sq = (c0->xyz[0] - c1->xyz[0]) * (c0->xyz[0] - c1->xyz[0])
			+ (c0->xyz[1] - c1->xyz[1]) * (c0->xyz[1] - c1->xyz[1])
			+ (c0->xyz[2] - c1->xyz[2]) * (c0->xyz[2] - c1->xyz[2]);
	fprintf(stderr, "  %g\n", sqrt(dist_sq));
}

void
printResidue(Residue *r, void *closure)
{
	if (r == NULL)
		fprintf(stderr, "missing!\n");
	else
		fprintf(stderr, "%s %s\n", r->type().chars(), r->id().chars());
}
