// Copyright 1992, U.C.S.F. Computer Graphics Laboratory
// $Id: AtomicSymbol.cc,v 1.4 94/10/13 17:41:46 gregc Exp $

#include <string.h>
#include <ctype.h>
#include "AtomicSymbol.h"

#ifdef _toupper
#define	toupper	_toupper
#endif
#ifdef _tolower
#define	tolower	_tolower
#endif

#ifndef NULL
#define	NULL	0
#endif

char const * const	AtomicSymbol::symbols[] = {
	"LP",  "H", "He", "Li", "Be",  "B",  "C",  "N",  "O",
	 "F", "Ne", "Na", "Mg", "Al", "Si",  "P",  "S", "Cl",
	"Ar",  "K", "Ca", "Sc", "Ti",  "V", "Cr", "Mn", "Fe",
	"Co", "Ni", "Cu", "Zn", "Ga", "Ge", "As", "Se", "Br",
	"Kr", "Rb", "Sr",  "Y", "Zr", "Nb", "Mo", "Tc", "Ru",
	"Rh", "Pd", "Ag", "Cd", "In", "Sn", "Sb", "Te",  "I",
	"Xe", "Cs", "Ba", "La", "Ce", "Pr", "Nd", "Pm", "Sm",
	"Eu", "Gd", "Tb", "Dy", "Ho", "Er", "Tm", "Yb", "Lu",
	"Hf", "Ta",  "W", "Re", "Os", "Ir", "Pt", "Au", "Hg",
	"Tl", "Pb", "Bi", "Po", "At", "Rn", "Fr", "Ra", "Ac",
	"Th", "Pa",  "U", "Np", "Pu", "Am", "Cm", "Bk", "Cf",
	"Es", "Fm", "Md", "No", "Lr",
};

AtomicSymbol::AS
AtomicSymbol::atomicNumber(const char *name)
{
	char const * const	*e;
	char			symbol[2];

	if (name == NULL)
		return LonePair;
	
	if (isdigit(name[0]) || name[0] == ' ') {
		if (islower(name[1]))
			symbol[0] = toupper(name[1]);
		else
			symbol[0] = name[1];
		symbol[1] = '\0';
	} else {
		if (islower(name[0]))
			symbol[0] = toupper(name[0]);
		else
			symbol[0] = name[0];
		if (isupper(name[1]))
			symbol[0] = tolower(name[1]);
		else
			symbol[0] = name[1];
	}

	if (symbol[1] == '\0')
		switch (symbol[0]) {
		case 'H': return H;
		case 'D': return D;	// deuterium
		case 'T': return T;	// tritium
		case 'B': return B;
		case 'C': return C;
		case 'N': return N;
		case 'O': return O;
		case 'F': return F;
		case 'P': return P;
		case 'S': return S;
		case 'V': return V;
		case 'Y': return Y;
		case 'I': return I;
		case 'W': return W;
		case 'U': return U;
		default: return LonePair;
		}

	for (e = symbols + 1; e < &symbols[Lr + 1]; e += 1)
		if (symbol[0] == (*e)[0] && symbol[1] == (*e)[1])
			return AS(e - symbols);
	return LonePair;
}


//	The following covalent bond radii are taken from: Elaine C. Meng
//	and Richard A. Lewis, "Determination of Molecular Topology and
//	Atomic Hybridization States from Heavy Atom Coordinates," Journal
//	of Computational Chemistry, Vol. 12, No. 7, pp. 891-898 (1991).

static const float	covalent[] = {
	0.0 /* */, 0.23 /* H */, 0.0 /* He */, 0.68 /* Li */,
	0.35 /* Be */, 0.83 /* B */, 0.68 /* C */, 0.68 /* N */,
	0.68 /* O */, 0.64 /* F */, 0.0 /* Ne */, 0.97 /* Na */,
	1.10 /* Mg */, 1.35 /* Al */, 1.20 /* Si */, 1.05 /* P */,
	1.02 /* S */, 0.99 /* Cl */, 0.0 /* Ar */, 1.33 /* K */,
	0.99 /* Ca */, 1.44 /* Sc */, 1.47 /* Ti */, 1.33 /* V */,
	1.35 /* Cr */, 1.35 /* Mn */, 1.34 /* Fe */, 1.33 /* Co */,
	1.50 /* Ni */, 1.52 /* Cu */, 1.45 /* Zn */, 1.22 /* Ga */,
	1.17 /* Ge */, 1.21 /* As */, 1.22 /* Se */, 1.21 /* Br */,
	0.0 /* Kr */, 1.47 /* Rb */, 1.12 /* Sr */, 1.78 /* Y */,
	1.56 /* Zr */, 1.48 /* Nb */, 1.47 /* Mo */, 1.35 /* Tc */,
	1.40 /* Ru */, 1.45 /* Rh */, 1.50 /* Pd */, 1.59 /* Ag */,
	1.69 /* Cd */, 1.63 /* In */, 1.46 /* Sn */, 1.46 /* Sb */,
	1.47 /* Te */, 1.40 /* I */, 0.0 /* Xe */, 1.67 /* Cs */,
	1.34 /* Ba */, 1.87 /* La */, 1.83 /* Ce */, 1.82 /* Pr */,
	1.81 /* Nd */, 1.80 /* Pm */, 1.80 /* Sm */, 1.99 /* Eu */,
	1.79 /* Gd */, 1.76 /* Tb */, 1.75 /* Dy */, 1.74 /* Ho */,
	1.73 /* Er */, 1.72 /* Tm */, 1.94 /* Yb */, 1.72 /* Lu */,
	1.57 /* Hf */, 1.43 /* Ta */, 1.37 /* W */, 1.35 /* Re */,
	1.37 /* Os */, 1.32 /* Ir */, 1.50 /* Pt */, 1.50 /* Au */,
	1.70 /* Hg */, 1.55 /* Tl */, 1.54 /* Pb */, 1.54 /* Bi */,
	1.68 /* Po */, 0.0 /* At */, 0.0 /* Rn */, 0.0 /* Fr */,
	1.90 /* Ra */, 1.88 /* Ac */, 1.79 /* Th */, 1.61 /* Pa */,
	1.58 /* U */, 1.55 /* Np */, 1.53 /* Pu */, 1.51 /* Am */,
};

static const int NumCovalent = sizeof covalent / sizeof covalent[0];

inline float
AtomicSymbol::bondRadius(AtomicSymbol a)
{
	if (a.number() < 0 || a.number() >= NumCovalent)
		return 0.0;
	else
		return covalent[a.number()];
}

float
AtomicSymbol::bondLength(AtomicSymbol a0, AtomicSymbol a1)
{
	return bondRadius(a0) + bondRadius(a1);
}
