// Copyright 1992, U.C.S.F. Computer Graphics Laboratory
// $Id: FixedAVector.h,v 1.13 95/01/05 13:21:13 gregc Exp $

#ifndef FIXED_AVECTOR_H
#define	FIXED_AVECTOR_H

#include <assert.h>

#include <iostream.h>

// This header file provides a template class for fixed length vectors
// of an arithmetic type.

#include <math.h>

template <class T, int SIZE>
class FixedAVector {
	T	v[SIZE];
public:
	FixedAVector() {}
	FixedAVector(const T vec[SIZE])
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] = vec[i];
	}

	// Since there are no pointers, memberwise copy is okay, and you
	// don't need an explicit copy constructor or operator=() function.

	bool operator==(const FixedAVector &r) const
	{
		for (int i = 0; i < SIZE; i += 1)
			if (v[i] != r.v[i])
				return 0;
		return 1;
	}

	bool operator!=(const FixedAVector &r) const
	{
		for (int i = 0; i < SIZE; i += 1)
			if (v[i] != r.v[i])
				return 1;
		return 0;
	}

	operator T *const ()
	{
		return v;
	}

	T &operator[](int index)
	{
		assert((unsigned) index < (unsigned) SIZE);
		return v[index];
	}

	T operator[](int index) const
	{
		assert((unsigned) index < (unsigned) SIZE);
		return v[index];
	}

	T &elem(int index)
	{
		return v[index];
	}

	T elem(int index) const
	{
		return v[index];
	}

	FixedAVector operator+(const FixedAVector &r) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] + r.v[i];
		return result;
	}

	void operator+=(const FixedAVector &r)
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] += r.v[i];
	}

	FixedAVector operator-() const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = - v[i];
		return result;
	}

	FixedAVector operator-(const FixedAVector &r) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] - r.v[i];
		return result;
	}

	void operator-=(const FixedAVector &r)
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] -= r.v[i];
	}

	FixedAVector multiply(const FixedAVector &r) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] * r.v[i];
		return result;
	}

	FixedAVector divide(const FixedAVector &r) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] / r.v[i];
		return result;
	}

	// dot product
	T operator*(const FixedAVector &r) const
	{
		T	result = 0;

		for (int i = 0; i < SIZE; i += 1)
			result += v[i] * r.v[i];
		return result;
	}

	FixedAVector operator+(T f) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] + f;
		return result;
	}

	FixedAVector operator-(T f) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] - f;
		return result;
	}

	FixedAVector operator*(T f) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] * f;
		return result;
	}

	FixedAVector operator/(T f) const
	{
		FixedAVector	result;

		for (int i = 0; i < SIZE; i += 1)
			result.v[i] = v[i] / f;
		return result;
	}

	void operator+=(T f)
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] += f;
	}

	void operator-=(T f)
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] -= f;
	}

	void operator*=(T f)
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] *= f;
	}

	void operator/=(T f)
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] /= f;
	}

	T sum() const
	{
		T	result = 0;

		for (int i = 0; i < SIZE; i += 1)
			result += v[i];
		return result;
	}

	T sqlength() const
	{
		T	result = 0;

		for (int i = 0; i < SIZE; i += 1)
			result += v[i] * v[i];
		return result;
	}

	double length() const
	{
		return sqrt(sqlength());
	}

	void negate()
	{
		for (int i = 0; i < SIZE; i += 1)
			v[i] = - v[i];
	}

	void normalize()
	{
		double	len = length();

		assert(len != 0.0);
		for (int i = 0; i < SIZE; i += 1)
			v[i] /= len;
	}

	void setLength(T newlen)
	{
		T	len = length();

		assert(len != 0.0);
		for (int i = 0; i < SIZE; i += 1)
			v[i] *= newlen / len;
	}
};

class FixedAVector<float, 3> {
	float	v[3];
public:
	FixedAVector() {}
	FixedAVector(const float vec[3])
	{
		v[0] = vec[0]; v[1] = vec[1]; v[2] = vec[2];
	}

	// Since there are no pointers, memberwise copy is okay, and you
	// don't need an explicit copy constructor or operator=() function.

	bool operator==(const FixedAVector &r) const
	{
		return v[0] == r.v[0] && v[1] == r.v[1] && v[2] == r.v[2];
	}

	bool operator!=(const FixedAVector &r) const
	{
		return v[0] != r.v[0] || v[1] != r.v[1] || v[2] != r.v[2];
	}

	operator float *const ()
	{
		return v;
	}

	float &operator[](int index)
	{
		assert((unsigned) index < (unsigned) 3);
		return v[index];
	}

	float operator[](int index) const
	{
		assert((unsigned) index < (unsigned) 3);
		return v[index];
	}

	float &elem(int index)
	{
		return v[index];
	}

	float elem(int index) const
	{
		return v[index];
	}

	FixedAVector operator+(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] + r.v[0];
		result.v[1] = v[1] + r.v[1];
		result.v[2] = v[2] + r.v[2];
		return result;
	}

	void operator+=(const FixedAVector &r)
	{
		v[0] += r.v[0]; v[1] += r.v[1]; v[2] += r.v[2];
	}

	FixedAVector operator-() const
	{
		FixedAVector	result;

		result.v[0] = - v[0];
		result.v[1] = - v[1];
		result.v[2] = - v[2];
		return result;
	}

	FixedAVector operator-(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] - r.v[0];
		result.v[1] = v[1] - r.v[1];
		result.v[2] = v[2] - r.v[2];
		return result;
	}

	void operator-=(const FixedAVector &r)
	{
		v[0] -= r.v[0]; v[1] -= r.v[1]; v[2] -= r.v[2];
	}

	FixedAVector multiply(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] * r.v[0];
		result.v[1] = v[1] * r.v[1];
		result.v[2] = v[2] * r.v[2];
		return result;
	}

	FixedAVector divide(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] / r.v[0];
		result.v[1] = v[1] / r.v[1];
		result.v[2] = v[2] / r.v[2];
		return result;
	}

	// dot product
	float operator*(const FixedAVector &r) const
	{
		return v[0] * r.v[0] + v[1] * r.v[1] + v[2] * r.v[2];
	}

	FixedAVector operator+(float f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] + f;
		result.v[1] = v[1] + f;
		result.v[2] = v[2] + f;
		return result;
	}

	FixedAVector operator-(float f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] - f;
		result.v[1] = v[1] - f;
		result.v[2] = v[2] - f;
		return result;
	}

	FixedAVector operator*(float f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] * f;
		result.v[1] = v[1] * f;
		result.v[2] = v[2] * f;
		return result;
	}

	FixedAVector operator/(float f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] / f;
		result.v[1] = v[1] / f;
		result.v[2] = v[2] / f;
		return result;
	}

	void operator+=(float f)
	{
		v[0] += f; v[1] += f; v[2] += f;
	}

	void operator-=(float f)
	{
		v[0] -= f; v[1] -= f; v[2] -= f;
	}

	void operator*=(float f)
	{
		v[0] *= f; v[1] *= f; v[2] *= f;
	}

	void operator/=(float f)
	{
		v[0] /= f; v[1] /= f; v[2] /= f;
	}

	float sum() const
	{
		return v[0] + v[1] + v[2];
	}

	float sqlength() const
	{
		return v[0] * v[0] + v[1] * v[1] + v[2] * v[2];
	}

	float length() const
	{
		return sqrtf(sqlength());
	}

	void negate()
	{
		v[0] = - v[0]; v[1] = - v[1]; v[2] = - v[2];
	}

	void normalize()
	{
		float	len = length();
		assert(len != 0.0);
		v[0] /= len; v[1] /= len; v[2] /= len;
	}

	void setLength(float newlen)
	{
		float	len = length();
		assert(len != 0.0);
		v[0] *= newlen / len;
		v[1] *= newlen / len;
		v[2] *= newlen / len;
	}
};

class FixedAVector<double, 3> {
	double	v[3];
public:
	FixedAVector() {}
	FixedAVector(const double vec[3])
	{
		v[0] = vec[0]; v[1] = vec[1]; v[2] = vec[2];
	}

	// Since there are no pointers, memberwise copy is okay, and you
	// don't need an explicit copy constructor or operator=() function.

	bool operator==(const FixedAVector &r) const
	{
		return v[0] == r.v[0] && v[1] == r.v[1] && v[2] == r.v[2];
	}

	bool operator!=(const FixedAVector &r) const
	{
		return v[0] != r.v[0] || v[1] != r.v[1] || v[2] != r.v[2];
	}

	operator double *const ()
	{
		return v;
	}

	double &operator[](int index)
	{
		assert((unsigned) index < (unsigned) 3);
		return v[index];
	}

	double operator[](int index) const
	{
		assert((unsigned) index < (unsigned) 3);
		return v[index];
	}

	double &elem(int index)
	{
		return v[index];
	}

	double elem(int index) const
	{
		return v[index];
	}

	FixedAVector operator+(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] + r.v[0];
		result.v[1] = v[1] + r.v[1];
		result.v[2] = v[2] + r.v[2];
		return result;
	}

	void operator+=(const FixedAVector &r)
	{
		v[0] += r.v[0]; v[1] += r.v[1]; v[2] += r.v[2];
	}

	FixedAVector operator-() const
	{
		FixedAVector	result;

		result.v[0] = - v[0];
		result.v[1] = - v[1];
		result.v[2] = - v[2];
		return result;
	}

	FixedAVector operator-(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] - r.v[0];
		result.v[1] = v[1] - r.v[1];
		result.v[2] = v[2] - r.v[2];
		return result;
	}

	void operator-=(const FixedAVector &r)
	{
		v[0] -= r.v[0]; v[1] -= r.v[1]; v[2] -= r.v[2];
	}

	FixedAVector multiply(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] * r.v[0];
		result.v[1] = v[1] * r.v[1];
		result.v[2] = v[2] * r.v[2];
		return result;
	}

	FixedAVector divide(const FixedAVector &r) const
	{
		FixedAVector	result;

		result.v[0] = v[0] / r.v[0];
		result.v[1] = v[1] / r.v[1];
		result.v[2] = v[2] / r.v[2];
		return result;
	}

	// dot product
	double operator*(const FixedAVector &r) const
	{
		return v[0] * r.v[0] + v[1] * r.v[1] + v[2] * r.v[2];
	}

	FixedAVector operator+(double f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] + f;
		result.v[1] = v[1] + f;
		result.v[2] = v[2] + f;
		return result;
	}

	FixedAVector operator-(double f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] - f;
		result.v[1] = v[1] - f;
		result.v[2] = v[2] - f;
		return result;
	}

	FixedAVector operator*(double f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] * f;
		result.v[1] = v[1] * f;
		result.v[2] = v[2] * f;
		return result;
	}

	FixedAVector operator/(double f) const
	{
		FixedAVector	result;

		result.v[0] = v[0] / f;
		result.v[1] = v[1] / f;
		result.v[2] = v[2] / f;
		return result;
	}

	void operator+=(double f)
	{
		v[0] += f; v[1] += f; v[2] += f;
	}

	void operator-=(double f)
	{
		v[0] -= f; v[1] -= f; v[2] -= f;
	}

	void operator*=(double f)
	{
		v[0] *= f; v[1] *= f; v[2] *= f;
	}

	void operator/=(double f)
	{
		v[0] /= f; v[1] /= f; v[2] /= f;
	}

	double sum() const
	{
		return v[0] + v[1] + v[2];
	}

	double sqlength() const
	{
		return v[0] * v[0] + v[1] * v[1] + v[2] * v[2];
	}

	double length() const
	{
		return sqrt(sqlength());
	}

	void negate()
	{
		v[0] = - v[0]; v[1] = - v[1]; v[2] = - v[2];
	}

	void normalize()
	{
		double	len = length();
		assert(len != 0.0);
		v[0] /= len; v[1] /= len; v[2] /= len;
	}

	void setLength(double newlen)
	{
		double	len = length();
		assert(len != 0.0);
		v[0] *= newlen / len;
		v[1] *= newlen / len;
		v[2] *= newlen / len;
	}
};

template <class T, int SIZE>
inline FixedAVector<T, SIZE>
lerp(const FixedAVector<T, SIZE> &l, const FixedAVector<T, SIZE> &r, float alpha)
{
	FixedAVector<T, SIZE>	result;

	for (int i = 0; i < SIZE; i += 1)
		result.elem(i) = l.elem(i) + (r.elem(i) - l.elem(i)) * alpha;
	return result;
}


template <class T, int SIZE>
inline FixedAVector<T, SIZE>
combine(const FixedAVector<T, SIZE> &l, const FixedAVector<T, SIZE> &r, float alpha, float beta)
{
	FixedAVector<T, SIZE>	result;

	for (int i = 0; i < SIZE; i += 1)
		result.elem(i) = alpha * l.elem(i) + beta * r.elem(i);
	return result;
}

template <class T, int SIZE>
inline double
distance(const FixedAVector<T, SIZE> &l, const FixedAVector<T, SIZE> &r)
{
	T sum = 0;
	for (int i = 0; i < SIZE; i += 1) {
		T diff = l.elem(i) - r.elem(i);
		sum += diff * diff;
	}
	return sqrt(sum);
}

inline float
distance(const FixedAVector<float, 3> &l, const FixedAVector<float, 3> &r)
{
	float sum = 0;
	for (int i = 0; i < 3; i += 1) {
		float diff = l.elem(i) - r.elem(i);
		sum += diff * diff;
	}
	return sqrtf(sum);
}

template <class T>
inline FixedAVector<T, 3>
cross(const FixedAVector<T, 3> &l, const FixedAVector<T, 3> &r)
{
	FixedAVector<T, 3>	result;

	result.elem(0) = (l.elem(1) * r.elem(2)) - (l.elem(2) * r.elem(1));
	result.elem(1) = (l.elem(2) * r.elem(0)) - (l.elem(0) * r.elem(2));
	result.elem(2) = (l.elem(0) * r.elem(1)) - (l.elem(1) * r.elem(0));
	return result;
}

template <class T, int SIZE>
inline ostream &
operator<<(ostream &os, const FixedAVector<T, SIZE> &r)
{
	os << r.elem(0);
	for (int i = 1; i < SIZE; i += 1)
		os << ' ' << r.elem(i);
	return os;
}

#endif /* FIXED_AVECTOR_H */
