// Copyright 1994, U.C.S.F. Computer Graphics Laboratory
// $Id: Geom3d.cc,v 1.3 95/01/05 13:47:19 gregc Exp $

#include "Geom3d.h"
#include <math.h>

// We follow the OpenGL/GraphicsGems nomenclature of column vectors for points.
//
// In graphics hardware, the affine transformations (Xform's) would be
// represented as 4 by 4 matrices whose last row would be (0, 0, 0, 1),
// points by a column vector whose last coordinate would be 1, and vectors
// would be column vectors whose last coordinate would 0.
//
// Due to the restrictions on the type of transformations allowed, the
// rotation part of a tranformation is always orthonormal.

#ifdef __GNUG__
template class FixedAVector<double, 3>;
template inline FixedAVector<double, 3> cross(const FixedAVector<double, 3> &, const FixedAVector<double, 3> &);
#endif

static inline void
swap(double &a, double &b)
{
	double t = a;
	a = b;
	b = t;
}

static inline bool
eq_zero(double f)
{
	return -1.0e-10f < f && f < 1.0e-10f;
}

void
Geom3d::Xform::setIdentity()
{
	rot[0][0] = 1; rot[0][1] = 0; rot[0][2] = 0;
	rot[1][0] = 0; rot[1][1] = 1; rot[1][2] = 0;
	rot[2][0] = 0; rot[2][1] = 0; rot[2][2] = 1;
	xlate.elem(0) = 0; xlate.elem(1) = 0; xlate.elem(2) = 0;
	inv_xlate = xlate;
}

void
Geom3d::Xform::setTranslate(double x, double y, double z)
{
	rot[0][0] = 1; rot[0][1] = 0; rot[0][2] = 0;
	rot[1][0] = 0; rot[1][1] = 1; rot[1][2] = 0;
	rot[2][0] = 0; rot[2][1] = 0; rot[2][2] = 1;
	xlate.elem(0) = x; xlate.elem(1) = y; xlate.elem(2) = z;
	inv_xlate = xlate;
	inv_xlate.negate();
}

void	
Geom3d::Xform::setTranslate(const Geom3d::Vector &xyz)
{
	rot[0][0] = 1; rot[0][1] = 0; rot[0][2] = 0;
	rot[1][0] = 0; rot[1][1] = 1; rot[1][2] = 0;
	rot[2][0] = 0; rot[2][1] = 0; rot[2][2] = 1;
	xlate = xyz;
	inv_xlate = xlate;
	inv_xlate.negate();
}

void
Geom3d::Xform::setXRotate(double angle)
{
	double s = sin(angle);
	double c = cos(angle);
	rot[0][0] = 1; rot[0][1] = 0; rot[0][2] = 0;
	rot[1][0] = 0; rot[1][1] = c; rot[1][2] = -s;
	rot[2][0] = 0; rot[2][1] = s; rot[2][2] = c;
	xlate.elem(0) = 0; xlate.elem(1) = 0; xlate.elem(2) = 0;
	inv_xlate = xlate;
}

void
Geom3d::Xform::setYRotate(double angle)
{
	double s = sin(angle);
	double c = cos(angle);
	rot[0][0] = c; rot[0][1] = 0; rot[0][2] = s;
	rot[1][0] = 0; rot[1][1] = 1; rot[1][2] = 0;
	rot[2][0] = -s; rot[2][1] = 0; rot[2][2] = c;
	xlate.elem(0) = 0; xlate.elem(1) = 0; xlate.elem(2) = 0;
	inv_xlate = xlate;
}

void
Geom3d::Xform::setZRotate(double angle)
{
	double s = sin(angle);
	double c = cos(angle);
	rot[0][0] = c; rot[0][1] = -s; rot[0][2] = 0;
	rot[1][0] = s; rot[1][1] = c; rot[1][2] = 0;
	rot[2][0] = 0; rot[2][1] = 0; rot[2][2] = 1;
	xlate.elem(0) = 0; xlate.elem(1) = 0; xlate.elem(2) = 0;
	inv_xlate = xlate;
}

void
Geom3d::Xform::setRotate(double x, double y, double z, double angle)
{
	double len = (x * x + y * y + z * z);
	if (eq_zero(len))
#if 0
		throw exception;
#else
		return;
#endif
	len = sqrt(len);
	x /= len;
	y /= len;
	z /= len;
	double s = sin(angle);
	double c = cos(angle);
	rot[0][0] = c + (1 - c) * x * x;
	rot[0][1] = c + (1 - c) * x * y + s * - z;
	rot[0][2] = c + (1 - c) * x * z + s * y;
	rot[1][0] = c + (1 - c) * y * x + s * z;
	rot[1][1] = c + (1 - c) * y * y;
	rot[1][2] = c + (1 - c) * y * z + s * - x;
	rot[2][0] = c + (1 - c) * z * x + s * - y;
	rot[2][1] = c + (1 - c) * z * y + s * x;
	rot[2][2] = c + (1 - c) * z * z;
	xlate.elem(0) = 0; xlate.elem(1) = 0; xlate.elem(2) = 0;
	inv_xlate = xlate;
}

void
Geom3d::Xform::setRotate(const Vector &xyz, double angle)
{
	setRotate(xyz.elem(0), xyz.elem(1), xyz.elem(2), angle);
}

// This function is like the one from the E & S library which produces
// a matrix to rotate a vector to the Z axis.
//
// This version is much enhanced over the Pic Sys original, in that
// you give it both ends of the bond to be rotated about and it
// returns both the "m_lookat" tensor and its inverse.
//
// Martin Pensak:  1977
//
// Stolen and hacked for the PS300 by Conrad Huang 24feb84
// Stolen again for the IRIS 23jan89
// Stolen and hacked again for C++ by Greg Couch 9sep94

void
Geom3d::Xform::setZAlign(const Geom3d::Point &p0, const Geom3d::Point &p1)
{
	double a = p1.elem(0) - p0.elem(0);
	double b = p1.elem(1) - p0.elem(1);
	double c = p1.elem(2) - p0.elem(2);
	double l = a * a + c * c;
	double d = l + b * b;
	if (eq_zero(d))
		return;
	l = sqrt(l);
	d = sqrt(d);

	rot[0][1] = 0;
	rot[1][1] = l / d;
	if (eq_zero(l)) {
		rot[0][0] = 1;
		rot[0][2] = 0;
		rot[1][0] = 0;
		rot[1][2] = - b / d;
	} else {
		rot[0][0] = c / l;
		rot[0][2] = - a / l;
		rot[1][0] = -(a * b) / (l * d);
		rot[1][2] = -(b * c) / (l * d);
	}
	rot[2][0] = a / d;
	rot[2][1] = b / d;
	rot[2][2] = c / d;

	/* now set up the translations */
	a = p0.elem(0);
	b = p0.elem(1);
	c = p0.elem(2);
	inv_xlate.elem(0) = a;
	inv_xlate.elem(1) = b;
	inv_xlate.elem(2) = c;

	xlate.elem(0) = -(a * rot[0][0] + b * rot[0][1] + c * rot[0][2]);
	xlate.elem(1) = -(a * rot[1][0] + b * rot[1][1] + c * rot[1][2]);
	xlate.elem(2) = -(a * rot[2][0] + b * rot[2][1] + c * rot[2][2]);
}

#define	update_inverse(t) \
	(t).inv_xlate.elem(0) = - ((t).rot[0][0] * (t).xlate.elem(0) \
		+ (t).rot[1][0] * (t).xlate.elem(1) \
		+ (t).rot[2][0] * (t).xlate.elem(2)); \
	(t).inv_xlate.elem(1) = - ((t).rot[0][1] * (t).xlate.elem(0) \
		+ (t).rot[1][1] * (t).xlate.elem(1) \
		+ (t).rot[2][1] * (t).xlate.elem(2)); \
	(t).inv_xlate.elem(2) = - ((t).rot[0][2] * (t).xlate.elem(0) \
		+ (t).rot[1][2] * (t).xlate.elem(1) \
		+ (t).rot[2][2] * (t).xlate.elem(2));

void
Geom3d::Xform::setLookAt(const Geom3d::Point &p0, const Geom3d::Point &p1,
							const Geom3d::Point &p2)
{
	/*
	 * Compute the rotational part of lookat matrix
	 */
	Vector	p01 = p1 - p0;
	p01.normalize();
	Vector	p02 = p2 - p0;
	Vector x = cross(p02, p01);
	x.normalize();
	Vector y = cross(p01, x);
	y.normalize();

	for (int i = 0; i < 3; i++) {
		rot[0][i] = x.elem(i);
		rot[1][i] = y.elem(i);
		rot[2][i] = p01.elem(i);
	}

	/*
	 * Compute the translation component of matrices
	 */
	xlate[0] = 0.0f; xlate[1] = 0.0f; xlate[2] = 0.0f;
	xlate = (Vector) (FixedAVector<double, 3>) apply(p0);
	xlate.negate();
	update_inverse(*this);
}

void
Geom3d::Xform::invert()
{
	swap(rot[0][1], rot[1][0]);
	swap(rot[0][2], rot[2][0]);
	swap(rot[1][2], rot[2][1]);
	swap(xlate.elem(0), inv_xlate.elem(0));
	swap(xlate.elem(1), inv_xlate.elem(1));
	swap(xlate.elem(2), inv_xlate.elem(2));
}

void
Geom3d::Xform::translate(double x, double y, double z)
{
	// TODO: optimize like zRotate
	Xform	t;
	t.setTranslate(x, y, z);
	multiply(t);
}

void	
Geom3d::Xform::translate(const Geom3d::Vector &xyz)
{
	// TODO: optimize like zRotate
	Xform	t;
	t.setTranslate(xyz);
	multiply(t);
}

void
Geom3d::Xform::xRotate(double angle)
{
	// TODO: optimize like zRotate
	Xform	t;
	t.setXRotate(angle);
	multiply(t);
}

void
Geom3d::Xform::yRotate(double angle)
{
	// TODO: optimize like zRotate
	Xform	t;
	t.setYRotate(angle);
	multiply(t);
}

void
Geom3d::Xform::zRotate(double angle)
{
#if 1
	Xform	t;
	t.setZRotate(angle);
	multiply(t);
#else
	// TODO: optimize axis rotation
	double tmp[2][3];

	double s = sin(angle);
	double c = cos(angle);
	tmp[0][0] = rot[0][0] * c + rot[1][0] * s;
	tmp[0][1] = rot[0][1] * c + rot[1][1] * s;
	tmp[0][2] = rot[0][2] * c + rot[1][2] * s;
	tmp[1][0] = rot[0][0] * - s + rot[1][0] * c;
	tmp[1][1] = rot[0][1] * - s + rot[1][1] * c;
	tmp[1][2] = rot[0][2] * - s + rot[1][2] * c;
	rot[0][0] = tmp[0][0];
	rot[0][1] = tmp[0][1];
	rot[0][2] = tmp[0][2];
	rot[1][0] = tmp[1][0];
	rot[1][1] = tmp[1][1];
	rot[1][2] = tmp[1][2];
	update_inverse(*this);
#endif
}

void
Geom3d::Xform::rotate(double x, double y, double z, double angle)
{
	// TODO: optimize like zRotate
	Xform	t;
	t.setRotate(x, y, z, angle);
	multiply(t);
}

void
Geom3d::Xform::rotate(const Geom3d::Vector &xyz, double angle)
{
	// TODO: optimize like zRotate
	Xform	t;
	t.setRotate(xyz, angle);
	multiply(t);
}

Geom3d::Point
Geom3d::Xform::apply(const Geom3d::Point &p) const
{
	Point	result;

	for (int i = 0; i < 3; i += 1) {
		result.elem(i) = xlate.elem(i);
		for (int j = 0; j < 3; j += 1)
			result.elem(i) += rot[i][j] * p.elem(j);
	}
	return result;
}

Geom3d::Vector
Geom3d::Xform::apply(const Geom3d::Vector &v) const
{
	Vector	result;

	// Vectors are transformed by the inverse tranpose, which in this
	// case (orthonormal rotation matrix in Xform, implicit zero
	// homogeneous coordinate in Vector) is just the matrix.
	for (int i = 0; i < 3; i += 1) {
		result.elem(i) = 0;
		for (int j = 0; j < 3; j += 1)
			result.elem(i) += v.elem(i) * rot[i][j];
	}
	return result;
}

void
Geom3d::Xform::multiply(const Geom3d::Xform &op)
{
	Xform	tmp;

	for (int i = 0; i < 3; i += 1)
		for (int j = 0; j < 3; j += 1) {
			tmp.rot[i][j] = 0;
			for (int k = 0; k < 3; k += 1)
				tmp.rot[i][j] += rot[i][k] * op.rot[k][j];
		}
	for (int j = 0; j < 3; j += 1) {
		tmp.xlate.elem(j) = xlate.elem(j);
		for (int k = 0; k < 3; k += 1)
			tmp.xlate.elem(j) += rot[j][k] * op.xlate.elem(k);
	}
	update_inverse(tmp);
	*this = tmp;
}

double
Geom3d::angle(const Geom3d::Vector &v0, const Geom3d::Vector &v1)
{
	double acc = v0 * v1;
	double d0 = v0.length();
	double d1 = v1.length();
	if (d0 <= 0.0 || d1 <= 0.0)
		return 0.0;
	acc /= (d0 * d1);
	if (acc > 1.0)
		acc = 1.0;
	else if (acc < -1.0)
		acc = -1.0;
	return acos(acc);
}

double
Geom3d::angle(const Geom3d::Point &p0, const Geom3d::Point &p1,
						const Geom3d::Point &p2)
{
	return angle(p0 - p1, p2 - p1);
}

double
Geom3d::dihedral(const Geom3d::Point &p0, const Geom3d::Point &p1,
			const Geom3d::Point &p2, const Geom3d::Point &p3)
{
	Vector v10 = p1 - p0;
	Vector v12 = p1 - p2;
	Vector v23 = p2 - p3;
	Vector t = cross(v10, v12);
	Vector u = cross(v23, v12);
	Vector v = cross(u, t);
	double w = v * v12;
	double acc = angle(u, t);
	if (w < 0.0)
		acc = -acc;
	return acc;
}
