#include "MGraph.h"
#include "MVertex.h"
#include "MEdge.h"
#include "MRing.h"
#include "alloc.h"
#include "misc.h"

MGraph::MGraph(unsigned size) // defaults to 7250
{
	Edges = (MEdge **) mg_malloc(size * sizeof(MEdge *));
	Vertices = (MVertex **) mg_malloc(size * sizeof(MVertex *));
	AllocatedEdges = AllocatedVertices = size;
	AllocatedPseudoEdges = (unsigned *) NULL;
	PseudoEdges = (MEdge ***) NULL;
	NumEdges = NumVertices = NumPseudoArrays = 0;
	NumPseudoEdges = (int *) NULL;
	Roots = (RootInfo *) NULL;
	Rings = (MRing **) NULL;
	NumRings = NumRoots = 0;
	AllocatedRings = AllocatedRoots = 0;
	RecomputeRoots = RecomputeRings = true;
	BeingDestroyed = false;
}

MGraph::~MGraph(void)
{
	BeingDestroyed = true;
	
	while (--NumEdges >= 0)
		delete Edges[NumEdges];
	
	while (--NumPseudoArrays >= 0) {
		while (--NumPseudoEdges[NumPseudoArrays] >= 0)
			delete PseudoEdges[NumPseudoEdges[NumPseudoArrays]];
		mg_free((void *)PseudoEdges[NumPseudoArrays]);
	}
	
	while (--NumVertices >= 0)
		delete Vertices[NumVertices];

	mg_free((void *)NumPseudoEdges);

	mg_free((void *)Roots);
	
	while(--NumRings >= 0)
		delete Rings[NumRings];
	mg_free((void *)Rings);
}

void MGraph::delVertex(const MVertex *v)
{
	int vertex;

	for (vertex = 0; vertex < NumVertices; vertex++)
		if (Vertices[vertex] == v)
			break;
	Vertices[vertex] = Vertices[--NumVertices];

	RecomputeRoots = RecomputeRings = true;
}

void MGraph::delEdge(const MEdge *e)
{
	int	*numedges, edge;
	MEdge	**edges;

	if (e->isPseudo()) {
		numedges = &NumPseudoEdges[e->pseudoType()];
		edges = PseudoEdges[e->pseudoType()];
	} else {
		numedges = &NumEdges;
		edges = Edges;
	}
	for (edge = 0; edge < *numedges; edge++)
		if (edges[edge] == e)
			break;
	edges[edge] = edges[--(*numedges)];

	RecomputeRoots = RecomputeRings = true;
}

void MGraph::addVertex(MVertex *v)
{
	if (NumVertices == AllocatedVertices) {
		int	newnumvertices = (int)(NumVertices * 1.5 + 1);
		Vertices = (MVertex **) mg_realloc((void *) Vertices,
		    newnumvertices * sizeof(MVertex *));
		AllocatedVertices = newnumvertices;
	}
	Vertices[NumVertices++] = v;
	RecomputeRoots = RecomputeRings = true;
}

void MGraph::addEdge(MEdge *e)
{
	if (e->isPseudo()) {
		if (e->pseudoType() >= NumPseudoArrays) {
			NumPseudoEdges =
			  (int *) mg_realloc((void *)NumPseudoEdges,
			    e->pseudoType() * sizeof(int));
			AllocatedPseudoEdges = (unsigned *)
			  mg_realloc((void *)AllocatedPseudoEdges,
			    e->pseudoType() * sizeof(unsigned));
			while (NumPseudoArrays <= e->pseudoType()) {
			  int numalloc = (int)(AllocatedEdges * 0.1 + 1);
			  PseudoEdges[NumPseudoArrays] = (MEdge **)
			    mg_malloc(numalloc * sizeof(MEdge *));
			  AllocatedPseudoEdges[NumPseudoArrays] = numalloc;
			  NumPseudoEdges[NumPseudoArrays++] = 0;
			}
			if (NumPseudoEdges[e->pseudoType()] ==
			  AllocatedPseudoEdges[e->pseudoType()]) {
				int newnumedges = (int)(NumEdges * 1.5 + 1);
				PseudoEdges[e->pseudoType()] = (MEdge **)
				  mg_realloc((void *)
				  PseudoEdges[e->pseudoType()],
				  newnumedges * sizeof(MEdge *));
				AllocatedPseudoEdges[e->pseudoType()] =
				  newnumedges;
			}
			PseudoEdges[e->pseudoType()]
			  [NumPseudoEdges[e->pseudoType()]++] = e;
		}
	} else {
		if (NumEdges == AllocatedEdges) {
			int newnumedges = (int)(NumEdges * 1.5 + 1);
			Edges = (MEdge **) mg_realloc((void *)Edges,
			  newnumedges * sizeof(MEdge *));
			AllocatedEdges = newnumedges;
		}
		Edges[NumEdges++] = e;
	}
	RecomputeRoots = RecomputeRings = true;
}

const MGraph::RootInfo *MGraph::roots(void)
{
	if (RecomputeRoots)
		traversalOrganize();
	return Roots;
}

// organize bonds in traversal order, determining roots also
void MGraph::traversalOrganize(void)
{
	MEdge **newEdges = (MEdge **) mg_malloc(NumEdges * sizeof(MEdge *));
	MVertex **newVertices = (MVertex **) mg_malloc(NumVertices
	  * sizeof(MVertex *));
	for (MVertex **vi = Vertices + NumVertices - 1; vi >= Vertices; vi--) {
		(*vi)->Visited = false;
	}
	for (MEdge **ei = Edges + NumEdges - 1; ei >= Edges; ei--) {
		(*ei)->Visited = false;
	}
	NumRoots = 0;
	mg_free(Roots);
	AllocatedRoots = 0;
	for (vi = Vertices + NumVertices - 1; vi >= Vertices; vi--) {
		MVertex *v = *vi;
		if (v->Visited) 
			continue;
		NumRoots++;
		if (NumRoots > AllocatedRoots) {
			AllocatedRoots = 2 * AllocatedRoots + 1;
			Roots = (RootInfo *) mg_realloc(Roots,
			  AllocatedRoots * sizeof(RootInfo));
		}
		RootInfo *ri = Roots + NumRoots - 1;
		ri->root = v;
		if (NumRoots > 1) {
			RootInfo *pri = Roots + NumRoots - 2;
			ri->edgeIndex = pri->edgeIndex + pri->size.numEdges;
			ri->vertexIndex = pri->vertexIndex + pri->size.numVertices;
		} else
			ri->edgeIndex = ri->vertexIndex = 0;
		ri->size = traverse(v, NULL, newEdges + ri->edgeIndex,
		  newVertices + ri->vertexIndex, v);
	}
	AllocatedEdges = NumEdges;
	AllocatedVertices = NumVertices;
	mg_free(Edges);
	mg_free(Vertices);
	Edges = newEdges;
	Vertices = newVertices;
}

MGraph::GraphSize MGraph::traverse(MVertex *root, MEdge *from, MEdge **edges,
  MVertex **vertices, MVertex *trueRoot)
{
	GraphSize gs;
	gs.numEdges = 0;
	gs.numVertices = 1;
	root->Visited = true;
	if (root == trueRoot)
		root->From = NULL; 
	root->Root = trueRoot;
	*vertices = root;
	vertices++;
	int curedge = root->numEdges() - 1;
	MEdge *const*e = root->edges() + curedge;
	for (; curedge >= 0; curedge--, e--) {
		if ((*e)->Visited)
			continue;
		if ((*e)->getBreakPoint() == root)
			continue;
		(*e)->Visited = true;
		*edges = *e;
		edges++;
		gs.numEdges++;
		(*e)->From = from;
		MVertex *const*ends = (*e)->vertices();
		MVertex *otherend = ends[0] == root ? ends[1] : ends[0];
		if (!otherend->Visited && (*e)->getBreakPoint() != otherend) {
			GraphSize branchSize = traverse(otherend, *e, edges,
			  vertices, trueRoot);
			otherend->From = root;
			gs.numEdges += branchSize.numEdges;
			gs.numVertices += branchSize.numVertices;
			edges += branchSize.numEdges;
			vertices += branchSize.numVertices;
		}
	}
	RecomputeRoots = false;
	return gs;
}

void MGraph::useAsRoot(MVertex *newRoot)
{
	if (RecomputeRoots)
		traversalOrganize();
	
	if (newRoot->Root == newRoot)
		return;
	
	MVertex *oldRoot = newRoot->Root;
	for (int i = NumRoots - 1; i >= 0; i--) {
		if (Roots[i].root == oldRoot)
			break;
	}
	
	MVertex **Vstart = Vertices + Roots[i].vertexIndex;
	MVertex **Vend = Vstart + Roots[i].size.numVertices;
	for (MVertex **vi = Vstart; vi < Vend; vi++) {
		(*vi)->Visited = false;
	}
	MEdge **Estart = Edges + Roots[i].edgeIndex;
	MEdge **Eend = Edges + Roots[i].size.numEdges;
	for (MEdge **ei = Estart; ei < Eend; ei++) {
		(*ei)->Visited = false;
	}
	(void) traverse(newRoot, NULL, Edges + Roots[i].edgeIndex,
	  Vertices + Roots[i].vertexIndex, newRoot);
	Roots[i].root = newRoot;
}

MGraph::VertexGroupList *MGraph::vertexAggregates(int aggSize, int *numFound) const
{
	*numFound = 0;

	if (aggSize < 1)
		return (VertexGroupList *) NULL;

	bool *vertexMarks = (bool *) mg_calloc(2*NumVertices, sizeof(bool));

	for (int i = 0; i < NumVertices; i++)
		Vertices[i]->AggregateInfo = vertexMarks + 2*i;

	VertexGroupList *head = NULL;
	MVertex **aggregate = new MVertex *[aggSize];
	for (MVertex **vi = Vertices + NumVertices - 1; vi >= Vertices; vi--) {
		aggregate[aggSize-1] = *vi;
		(*vi)->AggregateInfo[MVertex::CheckedAsEndNode] = true;
		findAggregates(aggregate, &head, aggSize, aggSize-1, numFound);
	}
	mg_free(vertexMarks);

	return head;
}

void MGraph::findAggregates(MVertex **aggregate, VertexGroupList **head,
  int aggSize, int remainingAggSize, int *numFound) const
{
	MVertex *v = aggregate[remainingAggSize];

	if (remainingAggSize == 0) {
		if (!v->AggregateInfo[MVertex::CheckedAsEndNode]) {
			(*numFound)++;
			VertexGroupList *vg = new VertexGroupList;
			vg->group = new MVertex *[aggSize];
			memcpy(vg->group, aggregate,
			  aggSize * sizeof(MVertex *));
			vg->next = *head;
			*head = vg;
		}
		return;
	}

	remainingAggSize--;

	v->AggregateInfo[MVertex::InCurrentAggregate] = true;

	for (MEdge **ei = v->Edges + v->NumEdges - 1; ei >= v->Edges; ei--) {
		MVertex *other = (*ei)->otherVertex(v);

		if (other->AggregateInfo[MVertex::InCurrentAggregate])
			continue;
		
		aggregate[remainingAggSize] = other;
		findAggregates(aggregate, head, aggSize,
		  remainingAggSize, numFound);
	}
	v->AggregateInfo[MVertex::InCurrentAggregate] = false;
}
