#ifndef MGRAPH_H
#define MGRAPH_H

#include "common.h"

class MGraph
{
protected:
	struct GraphSize {
		int numEdges, numVertices;
	};
	struct RootInfo {
		MVertex *root;
		int edgeIndex, vertexIndex; 
		GraphSize size;
	};
	struct VertexGroupList {
		MVertex **group;
		struct VertexGroupList *next;
	};
private:
	MEdge		**Edges;
	MVertex	**Vertices;
	unsigned	AllocatedEdges, AllocatedVertices;
	unsigned	*AllocatedPseudoEdges;
	MEdge		***PseudoEdges;
	int		NumEdges, NumVertices, NumPseudoArrays;
	int		*NumPseudoEdges;
	RootInfo	*Roots;
	MRing		**Rings;
	int		NumRings, NumRoots;
	unsigned	AllocatedRings, AllocatedRoots;

	void		traversalOrganize();
	GraphSize	traverse(MVertex *root, MEdge *from, MEdge **edges,
			  MVertex **vertices, MVertex *trueRoot);
	void		findAggregates(MVertex **agg, VertexGroupList **head,
			  int aggSize, int remainingAggSize, int *numFound) const;

	int		RingMaxEdges;
	  //  current list of rings was computed by looking for rings with no
	  //    no more than 'RingMaxEdges' edges.  If a ring search request
	  //    for a different maximum number of edges occurs, a new search is
	  //    required even if 'RecomputeRings' is false.

	bool	RecomputeRoots, RecomputeRings;
          // there has been an edge addition/deletion or an edge has been
          //   marked untraversable, so rings and/or root vertices need to be
          //   recomputed.  (Kept track of separately since ring recomputation
          //   needed much less frequently than root recomputation, probably)

	bool	BeingDestroyed;
         // so that graph components can tell that entire graph is being
         // destroyed and that they can forego some bookkeepping functions


	friend MEdge;
	friend MVertex;
	friend MRing;

	void		addVertex(MVertex *v);
	void		addEdge(MEdge *e);
	void		delVertex(const MVertex *v);
	void		delEdge(const MEdge *e);
	void		setRootsInvalid(void) { RecomputeRoots = true; }

	bool	beingDestroyed(void) const { return BeingDestroyed; }

protected:
	void		validateRoots(void); // make sure roots are correct
	
	const
	RootInfo	*roots(void);
	int		numRoots(void);
	void		useAsRoot(MVertex *newRoot);
	VertexGroupList *vertexAggregates(int aggSize, int *numFound) const;

public:
			MGraph(unsigned size = 7250);
	virtual	~MGraph(void);

	MVertex	*const *vertices(void) const
				{ return (MVertex *const *) Vertices; }
	int		numVertices(void) const
				{ return NumVertices; }
	
	MEdge		*const *edges(void) const
				{ return (MEdge *const *) Edges; }
	int		numEdges(void) const
				{ return NumEdges; }
	
	int		numPseudoTypes(void) const
				{ return NumPseudoArrays; }
	MEdge		*const *pseudoEdges(int type) const
				{ return (MEdge *const *) PseudoEdges[type]; }
	int		numPseudoEdges(int type) const
				{ return NumPseudoEdges[type]; }
};


inline int MGraph::numRoots(void)
{
	if (RecomputeRoots)
		(void) roots();
	return(NumRoots);
}

inline void MGraph::validateRoots(void)
{
	if (RecomputeRoots)
		traversalOrganize();
}
#endif
