#include "MolGraph.h"
#include "MolVertex.h"
#include "MolEdge.h"
#include <stdio.h>
#include <pdb++.h>
#include "alloc.h"

class Atom;
class Bond;
class Molecule;
class Ring;

class Molecule : public MolGraph<Atom,Bond,Molecule,Ring> {
public:
	Molecule(): MolGraph<Atom,Bond,Molecule,Ring>() {
		printf("Building molecule.\n");
	}
	~Molecule(void) {
		printf("Destroying molecule.\n");
	}
};

class Atom : public MolVertex<Atom,Bond,Molecule,Ring> {
	char	name_[20];
public:
	Atom(PDB &record, Molecule *m): MolVertex<Atom,Bond,Molecule,Ring>(m) {
		sprintf(name_, ":%d%c@%s",
		  record.atom.residue.seqNum,
		  record.atom.residue.chainId == '\0' ? ' ' :
		  record.atom.residue.chainId,
		  record.atom.name);
		char *from, *to;
		from = to = name_;
		while (*from != '\0') {
			if (*from == ' ')
				from++;
			else {
				*to = *from;
				from++, to++;
			}
		}
		*to = *from;
	}
	~Atom(void) { }
	char *name(void) { return name_; };
	void print(void) { printf("%s", name_); }
	int isNamed(char *label) { return strcmp(label, name_) == 0; };
};

class Bond : public MolEdge<Atom,Bond,Molecule,Ring> {
public:
	Bond(Atom *a1, Atom *a2): MolEdge<Atom,Bond,Molecule,Ring>(a1, a2) {
	}
	~Bond(void) {
	}
};

class Ring {
public:
	Ring(void) {};
	~Ring(void) {};
};

void printbond(Bond *);
void printatom(Atom *);

main(int argc, char **argv)
{
	Molecule	m;
	FILE		*pdbfile;
	char		buf[256];
	int		allocated = 0;
	Atom		**atoms = NULL, *first = NULL;
	Bond		*start;

	if (argc != 2) {
		fprintf(stderr, "Usage %s pdbfile\n", argv[0]);
		exit(1);
	}

	if ((pdbfile = fopen(argv[1], "r")) == NULL) {
		fprintf(stderr, "Cannot read %s\n", argv[1]);
		exit(1);
	}

	while (fgets(buf, sizeof buf, pdbfile) != NULL) {
		PDB	record(buf);

		switch (record.type()) {
		  case PDB::ATOM:
		  case PDB::HETATM:
			Atom *cur = new Atom(record, &m);
			if (record.atom.serialNum >= allocated) {
				allocated = record.atom.serialNum * 2 + 1;
				atoms = (Atom **) mg_realloc(atoms,
				  allocated * sizeof(Atom *));
			}
			atoms[record.atom.serialNum] = cur;
			if (first == NULL)
				first = cur;
			break;
		  case PDB::CONECT:
			for (int bond = 0; bond < 4; bond++) {
				if (record.conect.covalent[bond] >
				  record.conect.serialNum) {
					Bond *b = new Bond(
					  atoms[record.conect.covalent[bond]],
					  atoms[record.conect.serialNum]);
					if (record.conect.serialNum == 2
					  && record.conect.covalent[bond] == 5)
						start = b;
				}
			}
			break;
		}
	}
	printf("Molecule data read.\n");
	for (CIter<Atom> rootAtom = m.rootIter(); rootAtom.ok();
	  (void) rootAtom.next()) {
		for (CIter<Atom> travAtom = m.traverseVertices(rootAtom);
		  travAtom.ok(); (void) travAtom.next()) {
			printatom(&*travAtom);
		} 
		for (CIter<Bond> travBond = m.traverseEdges(rootAtom);
		  travBond.ok(); (void) travBond.next()) {
			printbond(&*travBond);
		}
	}

	CIter<Atom*const*> angles = m.vertexGroupIter(3);
	CIter<Atom*const*> diheds = m.vertexGroupIter(4);
	CIter<Atom*const*> fives = m.vertexGroupIter(5);
	CIter<Atom*const*> sixes = m.vertexGroupIter(6);
	printf("Found %d angles, %d dihedrals, %d five-mers, %d six-mers\n",
	  angles.numRemaining(), diheds.numRemaining(),
	  fives.numRemaining(), sixes.numRemaining());
	printf("5-mers:\n");
	for (; fives.ok(); fives.next()) {
		Atom *const*agg = *fives;
		printf("\t%s - %s - %s - %s - %s\n", agg[0]->name(),
		  agg[1]->name(), agg[2]->name(), agg[3]->name(), agg[4]->name());
	}
}

void printbond(Bond *b)
{
	printf("Traversing bond from atom "),
	b->vertex(0)->print(),
	printf(" to atom ");
	b->vertex(1)->print(),
	printf("\n");
}

void printatom(Atom *a)
{
	printf("Visiting atom "), a->print();
	a->from() == NULL ? printf(" <root>\n") : 
	(printf(" (from "), a->from()->print(), printf(")\n"));
}
