//
//	Copyright (c) 1992 The Regents of the University of California.
//	All rights reserved.
//
//	Redistribution and use in source and binary forms are permitted
//	provided that the above copyright notice and this paragraph are
//	duplicated in all such forms and that any documentation,
//	advertising materials, and other materials related to such
//	distribution and use acknowledge that the software was developed
//	by the University of California, San Francisco.  The name of the
//	University may not be used to endorse or promote products derived
//	from this software without specific prior written permission.
//	THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
//	IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
//	WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
//
//	$Id: pdb++.h,v 1.9 94/09/06 15:01:58 gregc Exp $
//
//	Based on Brookhaven National Laboratory Protein Data Bank, Feb 1992
//
//	C structure declarations
//

#ifndef PDB_H
#define	PDB_H

#include <iostream.h>

#ifdef __GNUG__
#pragma interface
#endif

class PDB {
public:
	enum {	BufLen = 73 };			// PDB record length (72 + '\0')
						// Using enum to get around bug:
						// ... static const int x = 0;
	typedef char	Date[10];
	typedef char	AName[5];		// atom name - NO2*
	typedef char	RName[5];		// residue name - ALA
	typedef char	PName[5];		// pdb name - 9lyz
	typedef char	Id[4];			// generic short id field
	typedef double	Real;			// size of floating point

	struct Residue {			// residue info
		RName	name;
		char	chainId;
		int	seqNum;
		char	insertCode;
	};

	//
	//	structures declarations for each record type
	//

	struct Unknown {
		char	junk[BufLen];
	};
	struct Aggrgt {
		int	serialNum;
		int	numComponents;
		int	cmpontSerialNums[14];
	};
	struct Agrdes {
		int	num;
		char	text[60];
	};
	struct Anisou {
		int	serialNum;
		AName	name;
		char	altLoc;
		Residue	residue;
		int	u[6];
	};
	struct Atom {
		int	serialNum;
		AName	name;
		char	altLoc;
		Residue	residue;
		Real	xyz[3];
		Real	occupancy, tempFactor;
		int	ftnoteNum;
	};
	struct Author {
		char	data[61];
		char	continuation;
	};
	typedef Agrdes	Cmpdes;
	struct Cmpont {
		int	seqNum;
		Residue	residues[2];
	};
	typedef Author	Compnd;
	struct Conect {
		int	serialNum;
		int	covalent[4];
		struct {
			int	hydrogen[2];
			int	salt;
		}	bonds[2];
	};
	struct Cryst1 {
		Real	a, b, c;
		Real	alpha, beta, gamma;
		char	spaceGrp[12];
		int	z;
	};
	// no structure for END
	// no structure for ENDMDL
	typedef Author	Expdta;
	struct Formul {
		int	component;
		RName	hetId;
		int	continuation;
		char	exclude;	// * to exclude
		char	formula[52];
	};
	typedef Agrdes	Ftnote;
	struct Header {
		char	classification[41];
		Date	timestamp;
		PName	id;
		char	type;
	};
	struct Helix {
		int	serialNum;
		Id	id;
		Residue	residues[2];
		int	type;
		char	comment[31];
	};
	struct Het {
		Residue	hetGrp;
		int	numAtoms;
		char	text[41];
	};
	typedef Atom	Hetatm;
	typedef Author	Jrnl;
	struct Master {
		int	numRemark;
		int	numFtnote;
		int	numHet;
		int	numHelix;
		int	numSheet;
		int	numTurn;
		int	numSite;
		int	numTransform;
		int	numCoordinate;
		int	numTer;
		int	numConect;
		int	numSeqres;
	};
	struct Model {
		int	num;
	};
	struct Mtrix {
		int	rowNum;
		int	serialNum;
		Real	m1, m2, m3, v;
		int	given;
	};
	typedef Agrdes	Mtxdes;
	struct Obslte {
		int	continuation;
		Date	timestamp;
		PName	oldId;
		PName	idMap[8];
	};
	struct Origx {
		int	rowNum;
		Real	o1, o2, o3, t;
	};
	typedef Ftnote	Remark;
	struct Revdat {
		int	modification;
		int	continuation;
		Date	timestamp;
		char	id[8];
		char	modType;
		char	corrections[31];
	};
	struct Scale {
		int	rowNum;
		Real	s1, s2, s3, u;
	};
	struct Seqres {
		int	serialNum;
		char	chainId;
		int	count;
		RName	names[13];
	};
	struct Sheet {
		int	strandNum;
		Id	id;
		int	count;
		Residue	residues[2];
		int	sense;
		struct {
			AName	name;
			Residue	residue;
		}		atoms[2];
	};
	typedef Atom	Sigatm;
	typedef Anisou	Siguij;
	struct Site {
		int	seqNum;
		Id	id;
		int	count;
		Residue	residues[4];
	};
	typedef Author	Source;
	struct Sprsde {
		int	continuation;
		Date	timestamp;
		PName	id;
		PName	supersede[8];
	};
	struct Ssbond {
		int	seqNum;
		Residue	residues[2];
		char	comment[31];
	};
	typedef Agrdes	Symdes;
	struct Symop {
		int	rowNum;
		int	serialNum;
		Real	s1, s2, s3, t;
	};
	struct Ter {
		int	serialNum;
		Residue	residue;
	};
	struct Trnsfm {
		int	resultSerialNum;
		int	applySerialNum;
		int	sourceSerialNum;
	};
	struct Turn {
		int	seqNum;
		Id	id;
		Residue	residues[2];
		char	comment[31];
	};
	struct Tvect {
		int	serialNum;
		Real	t1, t2, t3;
		char	comment[31];
	};
	struct User {
		char	subtype[3];
		char	text[67];
	};

	enum RecordType { UNKNOWN, ANISOU, ATOM, AUTHOR, COMPND, CONECT, CRYST1,
		END, FORMUL, FTNOTE, HEADER, HELIX, HET, HETATM, JRNL, MASTER,
		MTRIX, OBSLTE, ORIGX, REMARK, REVDAT, SCALE, SEQRES, SHEET,
		SIGATM, SIGUIJ, SITE, SOURCE, SPRSDE, SSBOND, TER, TURN, TVECT,
		USER, MODEL, ENDMDL, EXPDTA, SYMDES, SYMOP, MTXDES, CMPDES,
		CMPONT, TRNSFM, AGRDES, AGGRGT };
	const int	NUM_TYPES = AGGRGT + 1;
private:
	RecordType	rtype;
public:
	union {
		Unknown	unknown;
		Aggrgt	aggrgt;
		Agrdes	agrdes;
		Anisou	anisou;
		Atom	atom;
		Author	author;
		Cmpdes	cmpdes;
		Cmpont	cmpont;
		Compnd	compnd;
		Conect	conect;
		Cryst1	cryst1;
		// no End structure
		// no Endmdl structure
		Expdta	expdta;
		Formul	formul;
		Ftnote	ftnote;
		Header	header;
		Helix	helix;
		Het	het;
		Hetatm	hetatm;
		Jrnl	jrnl;
		Master	master;
		Model	model;
		Mtrix	mtrix;
		Mtxdes	mtxdes;
		Obslte	obslte;
		Origx	origx;
		Remark	remark;
		Revdat	revdat;
		Scale	scale;
		Seqres	seqres;
		Sheet	sheet;
		Sigatm	sigatm;
		Siguij	siguij;
		Site	site;
		Source	source;
		Sprsde	sprsde;
		Ssbond	ssbond;
		Symdes	symdes;
		Symop	symop;
		Ter	ter;
		Trnsfm	trnsfm;
		Turn	turn;
		Tvect	tvect;
		User	user;
	};

			PDB() { type(UNKNOWN); }
			PDB(RecordType t) { type(t); }
			PDB(const char *buf);
	RecordType	type() const { return rtype; }
	void		type(RecordType t);
	const char	*chars() const;
	static RecordType	getType(const char *buf);
	static int	sscanf(const char *, const char *, ...);
	static int	sprintf(char *, const char *, ...);

	friend istream	&operator>>(istream &s, PDB &p);
};

inline ostream &
operator<<(ostream &s, const PDB &p)
{
	s << p.chars();
	return s;
}

# endif // PDB_H
