// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef __GNUG__
#pragma implementation
#endif
#include "VoidP.CHSet.h"

// A CHSet is implemented as an array (tab) of buckets, each of which
// contains a pointer to a list of VoidPCHNodes.  Each node contains a
// pointer to the next node in the list, and a pointer to the VoidP.
// The end of the list is marked by a next node pointer which is odd
// when considered as an integer (least significant bit = 1).  The
// assumption is that CHNodes will all begin on even addresses.  If
// the odd pointer is right-shifted by one bit, it becomes the index
// within the tab array of the next bucket (that is, bucket i has
// next bucket pointer 2*(i+1)+1).

// The bucket pointers are initialized by the constructor and
// used to support the next(Pix&) method.

// This implementation is not portable to machines with different
// pointer and integer sizes, or on which CHNodes might be aligned on
// odd byte boundaries, but allows the same pointer to be used for
// chaining within a bucket and to the next bucket.


static inline int goodCHptr(VoidPCHNode* t)
{
  return ((((unsigned)t) & 1) == 0);
}

static inline VoidPCHNode* index_to_CHptr(int i)
{
  return (VoidPCHNode*)((i << 1) + 1);
}

static inline int CHptr_to_index(VoidPCHNode* t)
{
  return ( ((unsigned) t) >> 1);
}

VoidPCHSet::VoidPCHSet(unsigned int sz)
{
  tab = (VoidPCHNode**)(new VoidPCHNodePtr[size = sz]);
  for (unsigned int i = 0; i < size; ++i) tab[i] = index_to_CHptr(i+1);
  count = 0;
}

VoidPCHSet::VoidPCHSet(const VoidPCHSet& a)
{
  tab = (VoidPCHNode**)(new VoidPCHNodePtr[size = a.size]);
  for (unsigned int i = 0; i < size; ++i) tab[i] = index_to_CHptr(i+1);
  count = 0;
  for (Pix p = a.first(); p; a.next(p)) add(a(p));
}


Pix VoidPCHSet::seek(VoidP  key) const
{
  unsigned int h = VoidPHASH(key) % size;

  for (VoidPCHNode* t = tab[h]; goodCHptr(t); t = t->tl)
    if (VoidPEQ(key, t->hd))
      return Pix(t);

  return 0;
}


Pix VoidPCHSet::add(VoidP  item)
{
  unsigned int h = VoidPHASH(item) % size;

  for (VoidPCHNode* t = tab[h]; goodCHptr(t); t = t->tl)
    if (VoidPEQ(item, t->hd))
      return Pix(t);

  ++count;
  t = new VoidPCHNode(item, tab[h]);
  tab[h] = t;
  return Pix(t);
}


void VoidPCHSet::del(VoidP  key)
{
  unsigned int h = VoidPHASH(key) % size;

  VoidPCHNode* t = tab[h]; 
  VoidPCHNode* trail = t;
  while (goodCHptr(t))
  {
    if (VoidPEQ(key, t->hd))
    {
      if (trail == t)
        tab[h] = t->tl;
      else
        trail->tl = t->tl;
      delete t;
      --count;
      return;
    }
    trail = t;
    t = t->tl;
  }
}


void VoidPCHSet::clear()
{
  for (unsigned int i = 0; i < size; ++i)
  {
    VoidPCHNode* p = tab[i];
    tab[i] = index_to_CHptr(i+1);
    while (goodCHptr(p))
    {
      VoidPCHNode* nxt = p->tl;
      delete(p);
      p = nxt;
    }
  }
  count = 0;
}

Pix VoidPCHSet::first() const
{
  for (unsigned int i = 0; i < size; ++i) if (goodCHptr(tab[i])) return Pix(tab[i]);
  return 0;
}

void VoidPCHSet::next(Pix& p) const
{
  if (p == 0) return;
  VoidPCHNode* t = ((VoidPCHNode*)p)->tl;
  if (goodCHptr(t))
    p = Pix(t);
  else
  {
    for (unsigned int i = CHptr_to_index(t); i < size; ++i) 
    {
      if (goodCHptr(tab[i]))
      {
        p =  Pix(tab[i]);
        return;
      }
    }
    p = 0;
  }
}

int VoidPCHSet::operator == (const VoidPCHSet& b) const
{
  if (count != b.count)
    return 0;
  else
  {
    VoidPCHNode* p;
    for (unsigned int i = 0; i < size; ++i)
      for (p = tab[i]; goodCHptr(p); p = p->tl)
        if (b.seek(p->hd) == 0)
          return 0;
    for (i = 0; i < b.size; ++i)
      for (p = b.tab[i]; goodCHptr(p); p = p->tl)
        if (seek(p->hd) == 0)
          return 0;
    return 1;
  }
}

int VoidPCHSet::operator <= (const VoidPCHSet& b) const
{
  if (count > b.count)
    return 0;
  else
  {
    for (unsigned int i = 0; i < size; ++i)
      for (VoidPCHNode* p = tab[i]; goodCHptr(p); p = p->tl)
        if (b.seek(p->hd) == 0)
          return 0;
    return 1;
  }
}

void VoidPCHSet::operator |= (VoidPCHSet& b)
{
  if (&b == this || b.count == 0)
    return;
  for (unsigned int i = 0; i < b.size; ++i)
    for (VoidPCHNode* p = b.tab[i]; goodCHptr(p); p = p->tl)
      add(p->hd);
}

void VoidPCHSet::operator &= (VoidPCHSet& b)
{
  for (unsigned int i = 0; i < size; ++i)
  {
    VoidPCHNode* t = tab[i]; 
    VoidPCHNode* trail = t;
    while (goodCHptr(t))
    {
      VoidPCHNode* nxt = t->tl;
      if (b.seek(t->hd) == 0)
      {
        if (trail == tab[i])
          trail = tab[i] = nxt;
        else
          trail->tl = nxt;
        delete t;
        --count;
      }
      else
        trail = t;
      t = nxt;
    }
  }
}

void VoidPCHSet::operator -= (VoidPCHSet& b)
{
  for (unsigned int i = 0; i < size; ++i)
  {
    VoidPCHNode* t = tab[i]; 
    VoidPCHNode* trail = t;
    while (goodCHptr(t))
    {
      VoidPCHNode* nxt = t->tl;
      if (b.seek(t->hd) != 0)
      {
        if (trail == tab[i])
          trail = tab[i] = nxt;
        else
          trail->tl = nxt;
        delete t;
        --count;
      }
      else
        trail = t;
      t = nxt;
    }
  }
}

int VoidPCHSet::OK()
{
  int v = tab != 0;
  int n = 0;
  for (unsigned int i = 0; i < size; ++i)
  {
    for (VoidPCHNode* p = tab[i]; goodCHptr(p); p = p->tl) ++n;
    v &= CHptr_to_index(p) == i + 1;
  }
  v &= count == n;
  if (!v) error("invariant failure");
  return v;
}
