#########################################################################
#                                                                       #
#            (C) Copyright 1995 The Board of Trustees of the            #
#                        University of Illinois                         #
#                         All Rights Reserved                           #
#                                                                       #
#########################################################################

############################################################################
# RCS INFORMATION:
#
# 	$RCSfile: vectors.tcl,v $
# 	$Author: dalke $	$Locker:  $		$State: Exp $
#	$Revision: 1.1 $	$Date: 1997/03/23 10:21:27 $
#
############################################################################
# DESCRIPTION:
#  These routines handle the vector and matrix manipulations needed for
# doing 3D transformations.
#
############################################################################
# This is part of the VMD installation.
# For more information about VMD, see http://www.ks.uiuc.edu/Research/vmd

# a vector is a n element list of numbers (in column form)
# a matrix is a 4x4 matrix represeneted as a 4 element list of 4 
#   4 elements, in row major form
#
# This requires TclX


set M_PI 3.14159265358979323846

# Function:  veczero
# Returns :    the zero vector, {0 0 0}
proc veczero {} {
    return {0 0 0}
}


# Function:  vecadd {v1} {v2} ... {vn}
# Returns :   sum of the vectors, as a vector
proc vecadd {x y args} {
    if {[llength $x] != [llength $y]} {
	error "vecadd: two vectors don't have the same size"
    }
    if {$args == "" } {
	set retval ""
	for {set a [lvarpop x]; set b [lvarpop y]} {$a != ""} {
	        set a [lvarpop x]; set b [lvarpop y]} {
	    lappend retval [expr $a + $b]
	}
	return $retval
   }
   set result [vecadd $x $y]
   while {[set next [lvarpop args]] != ""} {
      set result [vecadd $result $next]
   }
   return $result
}

# Function:  vecsub {vector x} {vector y} 
# Returns :   v1 - v2, as a vector
proc vecsub {x y} {
    set retval ""
    if {[llength $x] != [llength $y]} {
	error "vecsub: x and y have different lengths"
    }
    for {set a [lvarpop x]; set b [lvarpop y]} {$a != ""} {
	    set a [lvarpop x]; set b [lvarpop y]} {
	lappend retval [expr  $a - $b]
    }
   return $retval
}

# Function: vecscale {constant c} {vector v}
# Returns :   product c * v
proc vecscale {c v} {
    set ret {}
    if [catch {
	if {[llength $c] == 1} {
	    foreach term $v {
		lappend ret [expr $c * $term]
	    } 
	} else {
	    foreach term $c {
		lappend ret [expr $v * $term]
	    }
	}
    }] {return [list vecscale: syntax error in $c * $v]}
    return $ret
}

# Function: vecdot {vector x} {vector y}
# Returns :  the vector dot product v1 * v2
proc vecdot {x y} {
    if {[llength $x] != [llength $y]} {
	error "vecdot needs vectors of the same size: $x : $y"
    }
    set ret 0
    foreach t1 $x t2 $y {
	set ret [expr $ret + $t1 * $t2]
    }
    return $ret
}

# Function: veccross {v1} {v2}
# Returns :  cross product of v1 and v2
proc veccross {x y} {
    lassign $x x1 x2 x3
    lassign $y y1 y2 y3
    set ret {}
    lappend ret [expr   $x2 * $y3 - $y2 * $x3]
    lappend ret [expr - $x1 * $y3 + $y1 * $x3]
    lappend ret [expr   $x1 * $y2 - $y1 * $x2]
    return $ret
}

# Function:  veclength {v}
#  Returns:    the vector length
proc veclength {v} {
    set retval 0
    foreach term $v {
	set retval [expr $retval + $term * $term]
    }
    return [sqrt $retval]
}

# Function:  veclength2 {v}
#  Returns:    the square of the vector length
proc veclength2 {v} {
    set retval 0
    foreach term $v {
	set retval [expr $retval + $term * $term]
    }
    return $retval
}

# Function:  vecnorm {v}
#  Returns:    the normal vector pointing along v
proc vecnorm {v} {
    set sum 0
    foreach term $v {
	set sum [expr $sum + $term * $term]
    }
    set sum [sqrt $sum]
    set retval {}
    foreach term $v {
	lappend retval [expr $term / $sum]
    }
    return $retval
}

# Function:  vecinvert {v}
#  Returns:    a vector with all terms inverted
proc vecinvert {v} {
    set ret {}
    foreach i $v {
	lappend ret [expr -$i]
    }
    return $ret
}


# Function: vectrans {matrix} {vector}
# Returns :  the vector = {matrix} * {vector}
#   If the matrix is 4x4 and the vector is length 3, the 4th element is 0
proc vectrans {m v} {
    if { [llength $v] == 3 } {
	lappend v 0
	return [lrange [vectrans $m $v] 0 2]
    }
    lassign $m m1 m2 m3 m4
    set ret {}
    if [catch {
	lappend ret [vecdot $m1 $v]
	lappend ret [vecdot $m2 $v]
	lappend ret [vecdot $m3 $v]
	lappend ret [vecdot $m4 $v]
    } ] {
error "vec/coordtrans: either the matrix or vector were incorrectly formed"
    }
    return $ret
}

# Function: coordtrans {matrix} {vector}
# Returns :  the vector = {matrix} * {vector}
#   If the matrix is 4x4 and the vector is length 3, the 4th element is 1
proc coordtrans {m v} {
    if { [llength $v] == 3} {
	lappend v 1
	return [lrange [vectrans $m $v] 0 2]
    }
    return [vectrans $m $v]
}


# Function: transidentity
#  Returns:   the identity matrix
proc transidentity { } {
 return "{1.0 0.0 0.0 0.0} {0.0 1.0 0.0 0.0} {0.0 0.0 1.0 0.0} {0.0 0.0 0.0 1.0}"
}

# Function: transtranspose {matrix}
# Returns :  the transpose of the matrix, as a matrix -- must be 4x4
proc transtranspose {m} {
    lassign $m m1 m2 m3 m4
    lassign $m1  m11 m12 m13 m14
    lassign $m2  m21 m22 m23 m24
    lassign $m3  m31 m32 m33 m34
    lassign $m4  m41 m42 m43 m44
    set retval {}
    lappend retval [concat $m11 $m21 $m31 $m41]
    lappend retval [concat $m12 $m22 $m32 $m42]
    lappend retval [concat $m13 $m23 $m33 $m43]
    lappend retval [concat $m14 $m24 $m34 $m44]
    return $retval
}
    

# Function: transmult {matrix} {matrix} ... {matrix}
# Returns :   matrix product of everything
proc transmult {mx my args} {
    if {$args == ""} {
	set myT [transtranspose $my]
	lassign $mx mx1 mx2 mx3 mx4
	lassign $myT my1 my2 my3 my4
	set retval {}
	if [catch {
	    lappend retval [concat [vecdot $mx1 $my1] [vecdot $mx1 $my2] [vecdot $mx1 $my3] [vecdot $mx1 $my4] ]
	    lappend retval [concat [vecdot $mx2 $my1] [vecdot $mx2 $my2] [vecdot $mx2 $my3] [vecdot $mx2 $my4] ]
	    lappend retval [concat [vecdot $mx3 $my1] [vecdot $mx3 $my2] [vecdot $mx3 $my3] [vecdot $mx3 $my4] ]
	    lappend retval [concat [vecdot $mx4 $my1] [vecdot $mx4 $my2] [vecdot $mx4 $my3] [vecdot $mx4 $my4] ]
	}] {
	    error "transmult: cannot multiply a poorly formed matrix"
	}
	return $retval
    }
# otherwise, accumulate the multiplications
    set result [transmult $mx $my]
    while {[set next [lvarpop args]] != ""} {
	set result [transmult $result $next]
    }
    return $result
}


# Function:  find_rotation_value <list reference>
#  Returns:    value of the rotation in radians with the appropriate
#              list elements removed
proc find_rotation_value {varname} {
    global M_PI
    upvar $varname a
    if {![info exists a]} {
	error "find_rotation_value: don't know upvar $varname"
    }

    set amount [expr [lvarpop a] + 0.0]
    set units [lvarpop a]
    if {$units == "rad" || $units == "radians" || $units == "radian"} {
	# set amount $amount
    } elseif {$units == "pi"} {
	set amount [expr $amount * $M_PI]
    } elseif {$units == "deg" || $units == "degrees" || $units == "degree"} {
	set amount [expr $amount / 180.0 * $M_PI]
    } else {
	if {$units != ""} {
	    lvarpush a $units
	}
	# default is degrees
	set amount [expr $amount / 180.0 * $M_PI]
    }
    return $amount
}



# Function:  transaxis {'x' | 'y' | 'z'} amount { | deg | rad | pi }
#  Returns:    the matrix to rotate "amount" radians about the given axis
# the default angle measurement is "degrees"
proc transaxis {axis args} {
    global M_PI
    if { $axis != "x" && $axis != "y" && $axis != "z" } {
	error "transaxis must get either x, y, or z, not $axis"
    }
    set amount [find_rotation_value args]
    if { $args != ""} {
	error "Unknown angle measurement '$args' in transaxis"
    }

    set cos [cos $amount]
    set mcos [expr -$cos]
    set sin [sin $amount]
    set msin [sin -$amount]
    if { $axis == "x" } {
	set retval           "{1.0 0.0 0.0 0.0}"
	lappend retval [concat 0.0 $cos $msin 0.0]
	lappend retval [concat 0.0 $sin $cos 0.0]
	lappend retval        {0.0 0.0 0.0 1.0}
	return $retval
    }
    if { $axis == "y" } {
	set retval {}
	lappend retval [concat  $cos 0.0  $sin 0.0]
	lappend retval          {0.0 1.0  0.0 0.0}
	lappend retval [concat  $msin 0.0  $cos 0.0]
	lappend retval          {0.0 0.0  0.0 1.0}
	return $retval
    }
    if { $axis == "z" } {
	set retval {}
	lappend retval [concat  $cos $msin 0.0 0.0]
	lappend retval [concat  $sin $cos 0.0 0.0]
	lappend retval          {0.0 0.0 1.0 0.0}
        lappend retval          {0.0 0.0 0.0 1.0}
	return $retval
    }
}


# Function:  transvec <vector>
#  Returns:    the matrix needed to bring 'x' along the given vector
proc transvec {v} {
    lassign $v x y z
    if {[catch {atan2 "$y,$x"}] != 0} {
	# then wants the z axis
	if {$z == 0} {
	    error "transvect can't take the 0 vector"
	}
	if {$z > 0} {
	    return "{0.0 0.0 -1.0 0.0} {0.0 1.0 0.0 0.0} {1.0 0.0 0.0 0.0} {0.0 0.0 0.0 1.0}"
	}
	return "{0.0 0.0 1.0 0.0} {0.0 1.0 0.0 0.0} {-1.0 0.0 0.0 0.0} {0.0 0.0 0.0 1.0}"
    }
    set theta [atan2 "$y,$x"]
    set length [sqrt [expr $x * $x + $y * $y]]
    set phi [atan2 "$z,$length"]
    set m1 [transaxis y -$phi rad]
    set m2 [transaxis z $theta rad]
    set retval [transmult $m2 $m1]
    return $retval
}

# Function:  transvecinv <vector>
#  Returns:    the matrix needed to bring a given vector along 'x'
proc transvecinv {v} {
    lassign $v x y z
    if {[catch {atan2 "$y,$x"}] != 0} {
	# then wants the z axis
	if {$z == 0} {
	    error "transvectinc can't take the 0 vector"
	}
	if {$z > 0} {
	    return "{0.0 0.0 1.0 0.0} {0.0 1.0 0.0 0.0} {-1.0 0.0 0.0 0.0} {0.0 0.0 0.0 1.0}"
	}
	return "{0.0 0.0 -1.0 0.0} {0.0 1.0 0.0 0.0} {1.0 0.0 0.0 0.0} {0.0 0.0 0.0 1.0}"
    }
    set theta [atan2 "$y,$x"]
    set length [sqrt [expr $x * $x + $y * $y]]
    set phi [atan2 "$z,$length"]
    set m1 [transaxis y $phi rad]
    set m2 [transaxis z -$theta rad]
    set retval [transmult $m1 $m2]
    return $retval
}


# Function: transoffset <vector>
#  Returns:  the matrix needed to translate by vector
proc transoffset {v} {
    lassign $v x y z
    return "{1.0 0.0 0.0 $x} {0.0 1.0 0.0 $y} {0.0 0.0 1.0 $z} {0.0 0.0 0.0 1.0}"
}


# Function:  transabout <vector>  amount { | deg | rad | pi }
#  Returns:   rotation matrix the given amount around the given axis
proc transabout {axis args} {
    lassign $args amount units
    set transf [transvec $axis]
    set transfinv [transvecinv $axis]
    set rot [transaxis x $amount $units]
    return [transmult $transf $rot $transfinv]
}

# Function: trans
#  this has lots of options
# 
proc trans {args} {
    set origin {0.0 0.0 0.0}
    set offset {0.0 0.0 0.0}
    set axis {1.0 0.0 0.0}
    set amount 0
    set rotmat [transidentity]


    while { [set keyword [lvarpop args]] != ""} {
	if { $keyword == "origin" } {
	    set origin [lvarpop args]
	    continue
	}
	if { $keyword == "offset" } {
	    set offset [lvarpop args]
	    continue
	}
	if { $keyword == "center" } {
	    set offset [lvarpop args]
	    set origin $offset
	    continue
	}
	# alias 'x' to 'axis x', 'y' to 'axis y', 'z' to 'axis z'
	if { $keyword == "x" || $keyword == "y" || $keyword == "z"} {
	    lvarpush args $keyword
	    set keyword "axis"
	}
	if { $keyword == "axis" } {
	    set axis [lvarpop args]
	    if {$axis == "x"} {
		set axis {1.0 0.0 0.0}
	    } elseif {$axis == "y"} {
		set axis {0.0 1.0 0.0}
	    } elseif {$axis == "z"} {
		set axis {0.0 0.0 1.0}
	    } elseif {[llength $axis] != 3} {
		error "transform: axis must be 'x', 'y', 'z' or a vector, not $axis"
	    }
	    # find out how much to rotate
	    set amount [find_rotation_value args]

	    # and apply to the current rotation matrix
	    set rotmat [transmult [transabout $axis $amount rad] $rotmat]
	    set axis {1.0 0.0 0.0}
	    set amount 0.0
	    continue
	}
	if { $keyword == "bond" } {
	    set v1 [lvarpop args]
	    set v2 [lvarpop args]
	    set origin $v1
	    set offset $v1
	    set axis [vecsub $v2 $v1]
	    # find out how much to rotate
	    set amount [find_rotation_value args]
#	    puts "Axis is: $axis"
	    set rotmat [transabout $axis $amount rad]
#	    puts "Rotmat is: $rotmat"
	    continue
	}
	if { $keyword == "angle" } {
	    set v1 [lvarpop args]
	    set v2 [lvarpop args]
	    set v3 [lvarpop args]
	    set origin $v2
	    set offset $v2
	    set axis [veccross [vecsub $v2 $v1] [vecsub $v3 $v2]]
	    if {[veclength $axis] <= 0.0} {
		if {[veclength [veccross [vecnorm [vecsub $v1 $v2] {1.0 0.0 0.0}]]] < 0.01} {
		    set axis {0.0 0.0 1.0}
		    puts "Warning: transform found degenerate 'angle'; using z axis"
		} else {
		    set axis {1.0 0.0 0.0}
		    puts "Warning: transform found degenerate 'angle'; using x axis"
		}
	    } else {
		set axis [vecnorm $axis]
	    }
	    # find out how much to rotate
	    set amount [find_rotation_value args]
	    set rotmat [transabout $axis $amount rad]
	    continue
	}
	error "Unknown command for 'transform': $keyword"

    }
    # end of while loop
    set origmat [transoffset [vecinvert $origin]]
    set offsetmat [transoffset $offset]
#    puts "Orig: $origmat"
#    puts "Offset: $offsetmat"
#    puts "Rotmat: $rotmat"
#    puts [list Result: [transmult $offsetmat $rotmat $origmat]]
    return [transmult $offsetmat $rotmat $origmat]
}
# end of "transform"

# Function: trans_from_rotate
# Returns a transformation matrix given a 3x3 rotation matrix
proc trans_from_rotate {rotate} {
  lassign $rotate a b c
  return "{$a 0} {$b 0} {$c 0} {0 0 0 1}"
}

# Function: trans_to_rotate
#  Returns: the upper left 3x3 rotation component
proc trans_to_rotate {trans_matrix} {
  lassign $trans_matrix a b c
  lassign $a a1 a2 a3
  lassign $b b1 b2 b3
  lassign $c c1 c2 c3
  return "{$a1 $a2 $a3} {$b1 $b2 $b3} {$c1 $c2 $c3}"
}

# Function: trans_from_offset
#  Returns: the transformation corresponding to an offset vector
proc trans_from_offset {offset} {
    return [transoffset $offset]
}

# Function: trans_to_offset
#  Returns: the transformation offset of the given matrix
proc trans_to_offset {trans_matrix} {
    return [coordtrans $trans_matrix {0 0 0}]
}
# set nothing "Okay!"
