/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomSel.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.40 $      $Date: 1997/01/09 02:50:11 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for selecting atoms.
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h1/dalke/vmd/src/RCS/AtomSel.C,v 1.40 1997/01/09 02:50:11 dalke Exp $";
#endif

#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include "Global.h"
#include "AtomSel.h"
#include "DrawMolecule.h"
#include "SymbolTable.h"
#include "ParseTree.h"
#include "MoleculeList.h"

#ifdef VMDTCL
#include "tcl.h"
#include "UIText.h"
#include "TclCommands.h"
#endif

///// Regex error catching functions
  // (I hate doing it this way, but that's the way this class works,
  //  and I didn't plan ahead for it)
#include "builtin.h"  // from GNU C++ class library
// is 0 (FALSE) if there was an error
static atomsel_regex_error = 1;

// contains the error message
static GString atomsel_regex_message;

// reset the error value to 1
void atomsel_regex_reset(void)
{
   atomsel_regex_error = 1;
   atomsel_regex_message = "";
}
// returns the value of the error value.  Also, if there is a pointer,
// sets it to the error string
int atomsel_regex_check(const char ** s)
{
   if (s) {
      *s = atomsel_regex_message;
   }
   return atomsel_regex_error;
}


//_VOLATILE_VOID atomsel_regex_error_fctn(const char *s1, const char *s2)
void atomsel_regex_error_fctn(const char *s1, const char *s2)
{
   atomsel_regex_message = s1;
   atomsel_regex_message += " : " ;
   atomsel_regex_message += s2;
   atomsel_regex_error = 0; // FALSE means error
}

// hold the previous error handle
static two_arg_error_handler_t atomsel_regex_oldfctn = NULL;

// swaps the normal error handler with the temporary one, and
// vice versa
static void atomsel_regex_swap(void)
{
   if (atomsel_regex_oldfctn == NULL ) {
      atomsel_regex_oldfctn = lib_error_handler;
      lib_error_handler = atomsel_regex_error_fctn;
   } else {
      lib_error_handler = atomsel_regex_oldfctn;
      atomsel_regex_oldfctn = NULL;
   }
}
////////////////////////////////////////////

//////
//   These functions get the information from the appropriate
// molecule, as defined in 'atom_sel_mol
// This is a 'DrawMolecule' since I want to call 'force_recalc' 
static DrawMolecule *atom_sel_mol;
static AtomSel *atom_sel;

// 'all'
static int atomsel_all(int num, int *flgs)
{
   for (int i=num-1; i>=0; i--) {
      flgs[i] = TRUE;
   }
   return 1;
}
// 'none'
static int atomsel_none(int num, int *flgs)
{
   for (int i=num-1; i>=0; i--) {
      flgs[i] = FALSE;
   }
   return 1;
}

#define generic_atom_data(fctnname, datatype, field)			      \
static int fctnname(int num, datatype *data, int *flgs)			      \
{									      \
   for (int i=0; i<num; i++) {						      \
      if (flgs[i]) {							      \
	 data[i] = atom_sel_mol -> atom(i)->field;			      \
      }									      \
   }									      \
   return 1;								      \
}

#define generic_set_atom_data(fctnname, datatype, field, extra)		      \
static int fctnname(int num, datatype *data, int *flgs)			      \
{									      \
  int i;								      \
  for (i=0; i<num; i++) {						      \
    if (flgs[i]) {							      \
      atom_sel_mol -> atom(i) -> field = data[i];			      \
    }									      \
  }									      \
  extra;								      \
  return 1;								      \
}

// 'name'
generic_atom_data(atomsel_name, GString, namestr);
// 'type'
generic_atom_data(atomsel_type, GString, typestr);
// 'index'
generic_atom_data(atomsel_index, int, index);
// 'fragment'
generic_atom_data(atomsel_fragment, int, fragment);
// 'numbonds'
generic_atom_data(atomsel_numbonds, int, bonds);
// 'residue'
generic_atom_data(atomsel_residue, int, uniq_resid);
// 'resname'
generic_atom_data(atomsel_resname, GString, resnamestr);
// 'chain'
generic_atom_data(atomsel_chain, GString, chainstr);
// 'segname'
generic_atom_data(atomsel_segname, GString, segnamestr);

// 'radius'
generic_atom_data(atomsel_radius, double, radius());
generic_set_atom_data(atomsel_set_radius, double, extra[ATOMRAD],
		      atom_sel_mol -> force_recalc(DrawMolItem::COL_REGEN));

// 'mass'
generic_atom_data(atomsel_mass, double, mass());
generic_set_atom_data(atomsel_set_mass, double, extra[ATOMMASS],
		      atom_sel_mol -> force_recalc(DrawMolItem::COL_REGEN));

// 'charge'
generic_atom_data(atomsel_charge, double, charge());
generic_set_atom_data(atomsel_set_charge, double, extra[ATOMCHARGE],
		      atom_sel_mol -> force_recalc(DrawMolItem::COL_REGEN));

// 'beta'
generic_atom_data(atomsel_beta, double, beta());
generic_set_atom_data(atomsel_set_beta, double, extra[ATOMBETA],
		      atom_sel_mol -> force_recalc(DrawMolItem::COL_REGEN));
// 'occupancy?'
generic_atom_data(atomsel_occupancy, double, occup());
generic_set_atom_data(atomsel_set_occupancy, double, extra[ATOMOCCUP], 
		      atom_sel_mol -> force_recalc(DrawMolItem::COL_REGEN));

// User forces
// 'user force in the x direction'
#define forces_changed                                           \
{for(i=0; i<num; i++) if (flgs[i]) atom_sel_mol -> atom(i) ->    \
	changed_userforce = 1;}

generic_atom_data(atomsel_user_force_x, double, extra[ATOM_USERFORCEX]);
generic_set_atom_data(atomsel_set_user_force_x, double, 
		      extra[ATOM_USERFORCEX], forces_changed);
// 'user force in the y direction'
generic_atom_data(atomsel_user_force_y, double, extra[ATOM_USERFORCEY]);
generic_set_atom_data(atomsel_set_user_force_y, double, 
		      extra[ATOM_USERFORCEY], forces_changed);
// 'user force in the z direction'
generic_atom_data(atomsel_user_force_z, double, extra[ATOM_USERFORCEZ]);
generic_set_atom_data(atomsel_set_user_force_z, double, 
		      extra[ATOM_USERFORCEZ], forces_changed);
// has the data changed?
generic_atom_data(atomsel_user_forces_changed, int, changed_userforce);
generic_set_atom_data(atomsel_modify_user_forces, int, changed_userforce,0);

// 'resid'
// this is unusual as I convert from string to an int
static int atomsel_resid(int num, int *data, int *flgs)
{
   for (int i=0; i<num; i++) {
      if (flgs[i]) {
	 data[i] = atoi(atom_sel_mol -> atom(i)->residstr);
      }
   }
   return 1;
}

// return a list of indicies of atoms bound to the given atom
static int atomsel_bondlist(int num, GString *data, int *flgs)
{
  for (int i=0; i<num; i++) {  // get the turned on atoms
    if (flgs[i]) {
      MolAtom *atom = atom_sel_mol -> atom(i);
      char s[20];
      for (int j=0; j<atom->bonds; j++) {  // get the list of atoms to
	if (j) data[i] += " ";             // which this atom is bound
	sprintf(s, "%d", atom->bondTo[j]);
	data[i] += s;
      }
    }
  }
  return 1;
}

#define generic_atom_boolean(fctnname, comparison)			      \
static int fctnname(int num, int *flgs)					      \
{									      \
   for (int i=0; i<num; i++) {						      \
      if (flgs[i]) {							      \
	 flgs[i] = atom_sel_mol -> atom(i) -> comparison;		      \
      }									      \
   }									      \
   return 1;								      \
}

// 'backbone'
generic_atom_boolean(atomsel_backbone, atomType != MolAtom::NORMAL);
// 'protein'
generic_atom_boolean(atomsel_protein, residueType == MolAtom::PROTEIN);
// 'nucleic'
generic_atom_boolean(atomsel_nucleic, residueType == MolAtom::NUCLEIC);
// 'water'
generic_atom_boolean(atomsel_water, residueType == MolAtom::WATERS);

#define generic_sstruct_boolean(fctnname, comparison)                    \
static int fctnname(int num, int *flgs)                                  \
{                                                                        \
  atom_sel_mol -> need_secondary_structure(1);                           \
  for (int i=0; i<num; i++) {                                            \
    Residue::SStruct s;                                                  \
    if (flgs[i]) {                                                       \
      s = atom_sel_mol -> residue(                                       \
				  atom_sel_mol -> atom(i) -> uniq_resid  \
				  ) -> sstruct;                          \
      if (!comparison) {                                                 \
	flgs[i] = 0;                                                     \
      }                                                                  \
    }                                                                    \
  }                                                                      \
  return 1;                                                              \
}

// once I define a structure, I don't need to recompute; hence the 
//   need_secondary_structure(0);
#define generic_set_sstruct_boolean(fctnname, newval)                    \
static int fctnname(int num, int *data, int *flgs)                       \
{                                                                        \
  atom_sel_mol -> need_secondary_structure(0);                           \
  for (int i=0; i<num; i++) {                                            \
    if (flgs[i] && data[i]) {                                            \
	atom_sel_mol -> residue( atom_sel_mol -> atom(i) -> uniq_resid ) \
	  -> sstruct = Residue:: newval;                                 \
    }                                                                    \
  }                                                                      \
  return 1;                                                              \
}


static void recursive_find_sidechain_atoms(BaseMolecule *mol, int *sidechain,
					   int atom_index)
{
  // Have I been here before?
  if (sidechain[atom_index] == 2) return;
  // Is this a backbone atom
  MolAtom *atom = mol -> atom(atom_index);
  if (atom -> atomType != MolAtom::NORMAL) return;
  
  // mark this atom
  sidechain[atom_index] = 2;

  // try the atoms to which this is bonded
  for (int i=0; i<atom->bonds; i++) {
    recursive_find_sidechain_atoms(mol, sidechain, atom->bondTo[i]);
  }
}

// give an array where 1 indicates an atom on the sidechain, find the
// connected atoms which are also on the sidechain
static void find_sidechain_atoms(BaseMolecule *mol, int *sidechain)
{
  for (int i=0; i<mol->nAtoms; i++) {
    if (sidechain[i] == 1) {
      recursive_find_sidechain_atoms(mol, sidechain, i);
    }
  }
}


// 'sidechain' is tricky.  I start from a protein CA, pick a bond which
// isn't along the backbone (and discard it once if it is a hydrogen),
// and follow the atoms until I stop at another backbone atom or run out
// of atoms
static int atomsel_sidechain(int num, int *flgs)
{
  int *seed = new int[num];
  {  // generate a list of the "CB" atoms (or whatever they are)
    for (int i=0; i<num; i++) {
      seed[i] = 0;
    }
  }
  // get the CA and HA2 name index
  int CA = atom_sel_mol -> atomNames.typecode("CA");
  // for each protein
  for (int pfrag=atom_sel_mol->pfragList.num()-1; pfrag>=0; pfrag--) {
    // get a residue
    for (int res = atom_sel_mol->pfragList[pfrag]->num()-1; res >=0; res--) {
      // find the CA
      int atomid = atom_sel_mol -> find_atom_in_residue(CA, res);
      if (atomid < 0) {
	msgErr << "atomselection: sidechain: cannot find CA" << sendmsg;
	continue;
      }
      // find at most two neighbors which are not on the backbone
      MolAtom *atom = atom_sel_mol -> atom(atomid);
      int b1 = -1, b2 = -1;
      int i;
      for (i=0; i<atom->bonds; i++) {
	if (atom->bondType[i] == MolAtom::NORMAL) {
	  if (b1 == -1) {
	    b1 = atom->bondTo[i];
	  } else {
	    if (b2 == -1) {
	      b2 = atom->bondTo[i];
	    } else {
	      msgErr << "atomselection: sidechain: protein residue index " 
		     << res << ", C-alpha index " << i << " has more than "
		     << "two non-backbone bonds; ignoring the others"
		     << sendmsg;
	    }
	  }
	}
      }
      if (b1 == -1) continue;
      if (b2 != -1) {  // find the right one
	// first check the number of atoms and see if I have a lone H
	int c1 = atom_sel_mol -> atom(b1) -> bonds;
	int c2 = atom_sel_mol -> atom(b2) -> bonds;
	if (c1 == 1 && c2 > 1) {
	  b1 = b2;
	} else if (c2 == 1 && c1 > 1) {
	  b1 = b1;
	} else if (c1 ==1 && c2 == 1) {
	  // check the masses
	  float m1 = atom_sel_mol -> atom(b1) -> mass();
	  float m2 = atom_sel_mol -> atom(b2) -> mass();
	  if (m1 > 2.3 && m2 <= 2.3) {
	    b1 = b2;
	  } else if (m2 > 2.3 && m1 <= 2.3) {
	    b1 = b1;
	  } else if (m1 <= 2.0 && m2 <= 2.3) {
	    // should have two H's, find the "first" of these
	    if (strcmp(atom_sel_mol -> atom(b1) -> namestr,
		       atom_sel_mol -> atom(b2) -> namestr) > 0) {
	      b1 = b2;
	    } // else b1 = b1
	  } else {
	    msgErr << "atomselect: sidechain:  protein residue index " 
		   << res << ", C-alpha index " << i << " has sidechain-like "
		   << "atom (indicies " << b1 << " and " << b2 << "), and "
		   << "I cannot determine which to call a sidechain -- "
		   << "I'll guess" << sendmsg;
	    if (strcmp(atom_sel_mol -> atom(b1) -> namestr,
		       atom_sel_mol -> atom(b2) -> namestr) > 0) {
	      b1 = b2;
	    }
	  } // punted
	} // checked connections and masses
      } // found the right one; it is b1.
      seed[b1] = 1;

    } // loop over residues
  } // loop over protein fragments

  // do the search for all the sidechain atoms (returned in seed)
  find_sidechain_atoms(atom_sel_mol, seed);
  // set the return values
  {
    for (int i=0; i<num; i++) {
      if (flgs[i]) flgs[i] = (seed[i] != 0);
    }
  }
  return 1;
}


// which of these are helices?
generic_sstruct_boolean(atomsel_helix, (s == Residue::HELIX_ALPHA ||
					s == Residue::HELIX_3_10  ||
					s == Residue::HELIX_PI));
generic_sstruct_boolean(atomsel_alpha_helix, (s == Residue::HELIX_ALPHA));
generic_sstruct_boolean(atomsel_3_10_helix, (s == Residue::HELIX_3_10));
generic_sstruct_boolean(atomsel_pi_helix, (s == Residue::HELIX_PI));

// Makes the residue into a HELIX_ALPHA
generic_set_sstruct_boolean(atomsel_set_helix, HELIX_ALPHA);
// Makes the residue into a HELIX_3_10
generic_set_sstruct_boolean(atomsel_set_3_10_helix, HELIX_3_10);
// Makes the residue into a HELIX_PI
generic_set_sstruct_boolean(atomsel_set_pi_helix, HELIX_PI);

// which of these are beta sheets?
generic_sstruct_boolean(atomsel_sheet, (s == Residue::BETA ||
					s == Residue::BRIDGE ));
generic_sstruct_boolean(atomsel_extended_sheet, (s == Residue::BETA));
generic_sstruct_boolean(atomsel_bridge_sheet, (s == Residue::BRIDGE ));

// Makes the residue into a BETA
generic_set_sstruct_boolean(atomsel_set_sheet, BETA);
generic_set_sstruct_boolean(atomsel_set_bridge_sheet, BRIDGE);

// which of these are coils?
generic_sstruct_boolean(atomsel_coil, (s == Residue::COIL));

// Makes the residue into COIL
generic_set_sstruct_boolean(atomsel_set_coil, COIL);

// which of these are TURNS?
generic_sstruct_boolean(atomsel_turn, (s == Residue::TURN));

// Makes the residue into TURN
generic_set_sstruct_boolean(atomsel_set_turn, TURN);

// return the sstruct information as a 1 character string
static int atomsel_sstruct(int num, GString *data, int *flgs)
{
  atom_sel_mol -> need_secondary_structure(1);
  for (int i=0; i<num; i++) {
    if (flgs[i]) {
      switch( atom_sel_mol 
	      -> residue( atom_sel_mol -> atom(i) -> uniq_resid ) 
	      -> sstruct) {
      case Residue::HELIX_ALPHA: data[i] = 'H'; break;
      case Residue::HELIX_3_10 : data[i] = 'G'; break;
      case Residue::HELIX_PI   : data[i] = 'I'; break;
      case Residue::BETA       : data[i] = 'E'; break;
      case Residue::BRIDGE     : data[i] = 'B'; break;
      case Residue::TURN       : data[i] = 'T'; break;
      default:
      case Residue::COIL       : data[i] = 'C'; break;
      }
    }
  }
  return 1;
}
// set the secondary structure based on a string value
static int atomsel_set_sstruct(int num, GString *data, int *flgs)
{
  char c;
  atom_sel_mol -> need_secondary_structure(0);
  for (int i=0; i<num; i++) {
    if (flgs[i]) {
      if (data[i].length() == 0) {
	msgErr << "cannot set a 0 length secondary structure string"
	       << sendmsg;
      } else {
	c = ((const char *) data[i])[0];
	if (data[i].length() > 1) {
	  while (1) {
 if (!strcasecmp((const char *) data[i], "helix")) { c = 'H'; break;}
 if (!strcasecmp((const char *) data[i], "alpha")) { c = 'H'; break;}
 if (!strcasecmp((const char *) data[i], "alpha_helix")) { c = 'H'; break;}
 if (!strcasecmp((const char *) data[i], "alphahelix"))  { c = 'H'; break;}
 if (!strcasecmp((const char *) data[i], "alpha helix")) { c = 'H'; break;}

 if (!strcasecmp((const char *) data[i], "pi"))        { c = 'I'; break;}
 if (!strcasecmp((const char *) data[i], "pi_helix"))  { c = 'I'; break;}
 if (!strcasecmp((const char *) data[i], "pihelix"))   { c = 'I'; break;}
 if (!strcasecmp((const char *) data[i], "pi helix"))  { c = 'I'; break;}

 if (!strcasecmp((const char *) data[i], "310"))       { c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "310_helix")) { c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "3_10"))      { c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "3"))         { c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "310 helix")) { c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "3_10_helix")){ c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "3_10 helix")){ c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "3 10 helix")){ c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "helix_3_10")){ c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "helix 3 10")){ c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "helix3_10")) { c = 'G'; break;}
 if (!strcasecmp((const char *) data[i], "helix310"))  { c = 'G'; break;}

 if (!strcasecmp((const char *) data[i], "beta"))      { c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "betasheet")) { c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "beta_sheet")){ c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "beta sheet")){ c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "sheet"))     { c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "strand"))    { c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "beta_strand"))  { c = 'E'; break;}
 if (!strcasecmp((const char *) data[i], "beta strand"))  { c = 'E'; break;}

 if (!strcasecmp((const char *) data[i], "turn"))  { c = 'T'; break;}

 if (!strcasecmp((const char *) data[i], "coil"))     { c = 'C'; break;}
 if (!strcasecmp((const char *) data[i], "unknown"))  { c = 'C'; break;}
 c = 'X';
 break;
	  } // while (1)
	}
	// and set the value
	Residue::SStruct s = Residue::COIL;
	switch ( c ) {
	case 'H':
	case 'h': s = Residue::HELIX_ALPHA; break;
	case '3':
	case 'G':
	case 'g': s = Residue::HELIX_3_10; break;
	case 'P':  // so you can say 'pi'
	case 'p':
	case 'I':
	case 'i': s = Residue::HELIX_PI; break;
	case 'S':  // for "sheet"
	case 's':
	case 'E':
	case 'e': s = Residue::BETA; break;
	case 'B':
	case 'b': s = Residue::BRIDGE; break;
	case 'T':
	case 't': s = Residue::TURN; break;
	case 'L': // L is from PHD and may be an alternate form
	case 'l':
	case 'C':
	case 'c': s = Residue::COIL; break;
	default: {
	  msgErr << "Unknown sstruct assignment: '"
	    << (const char *) data[i] << "'" << sendmsg;
	  s = Residue::COIL; break;
	}
	}
	atom_sel_mol 
	  -> residue( atom_sel_mol -> atom(i) -> uniq_resid ) 
	  -> sstruct = s;
      }
    }
  }
  return 1;
}


//// specialized function to turn on all atoms in a given residue
//      and leave the rest alone.
// It is slower this way, but easier to understand
static void mark_atoms_given_residue(int residue, int *on)
{
  ResizeArray<int> *atoms = &(atom_sel_mol->residueList[residue]->atoms);
  for (int i= atoms->num()-1; i>=0; i--) {
     on[(*atoms)[i]] = TRUE;
  }
}


// macro for either protein or nucleic fragments
#define fragment_data(fctn, fragtype)					      \
static int fctn(int num, int *data, int *)				      \
{									      \
   int *tmp = new int[num];						      \
   int i;								      \
   for (i=num-1; i>=0; i--) {  /* clear the arrays */		      \
      tmp[i] = 0;							      \
      data[i] = -1;  /* default is -1 for 'not a [np]frag' */		      \
   }									      \
   /* for each fragment */						      \
   for ( i=atom_sel_mol->fragtype.num()-1; i>=0; i--) {			      \
      /* for each residues of the fragment */				      \
      int j;								      \
      for (j=atom_sel_mol->fragtype[i]->num()-1; j>=0; j--) {		      \
	 /* mark the atoms in the fragment */				      \
	 mark_atoms_given_residue((*atom_sel_mol->fragtype[i])[j], tmp);      \
      }									      \
      /* and label them with the appropriate number */			      \
      for (j=num-1; j>=0; j--) {					      \
	 if (tmp[j]) {							      \
	    data[j] = i;						      \
	    tmp[j] = 0;							      \
	 }								      \
      }									      \
   }									      \
   delete [] tmp;							      \
   return 1;								      \
}

fragment_data(atomsel_pfrag, pfragList);
fragment_data(atomsel_nfrag, nfragList);

// this is the only function which uses the 'which_frame' field
// to get the right value of on x,y,z
#define position_data(fctn, offset)					      \
static int fctn(int num, double *data, int *flgs)			      \
{									      \
  float *r = NULL;                                                            \
  if (atom_sel_mol -> is_current()) {                                         \
    int num = atom_sel_mol -> Animation::num();                               \
    switch (atom_sel -> which_frame) {                                        \
     case AtomSel::TS_LAST: r = atom_sel_mol -> item(num-1) -> pos; break;    \
     case AtomSel::TS_NOW : r = atom_sel_mol -> current() -> pos; break;      \
     default: {                                                               \
       if (!atom_sel_mol -> item(atom_sel -> which_frame) -> pos) {           \
         r = atom_sel_mol -> item(num-1) -> pos;                              \
       } else {                                                               \
         r = atom_sel_mol -> item(atom_sel -> which_frame) -> pos;            \
       }                                                                      \
     }                                                                        \
    }                                                                         \
  }                                                                           \
  if (!r) return 0;                                                           \
  for (int i=num-1; i>=0; i--) {					      \
    if (flgs[i]) {							      \
      data[i] = r[3*i + offset];					      \
    }								              \
  }									      \
  return 1;								      \
}
#define set_position_data(fctn, offset)					      \
static int fctn(int num, double *data, int *flgs)			      \
{									      \
  float *r = NULL;                                                            \
  if (atom_sel_mol -> is_current()) {                                         \
    int num = atom_sel_mol -> Animation::num();                               \
    switch (atom_sel -> which_frame) {                                        \
     case AtomSel::TS_LAST: r = atom_sel_mol -> item(num-1) -> pos; break;    \
     case AtomSel::TS_NOW : r = atom_sel_mol -> current() -> pos; break;      \
     default: {                                                               \
       if (!atom_sel_mol -> item(atom_sel -> which_frame) -> pos) {           \
         r = atom_sel_mol -> item(num-1) -> pos;                              \
       } else {                                                               \
         r = atom_sel_mol -> item(atom_sel -> which_frame) -> pos;            \
       }                                                                      \
     }                                                                        \
    }                                                                         \
  }                                                                           \
  if (!r) return 0;                                                           \
  for (int i=num-1; i>=0; i--) {					      \
    if (flgs[i]) {							      \
      r[3*i + offset] = data[i];					      \
    }								              \
  }									      \
  atom_sel_mol -> force_recalc();					      \
  return 1;								      \
}

position_data(atomsel_xpos, 0);
position_data(atomsel_ypos, 1);
position_data(atomsel_zpos, 2);

set_position_data(atomsel_set_xpos, 0);
set_position_data(atomsel_set_ypos, 1);
set_position_data(atomsel_set_zpos, 2);

static double atomsel_square(double x)
{
   return x*x;
}

// this is different than the previous.  It allows me to search for a
// given regex sequence.  For instace, given the protein sequence
//   WAPDTYLVAPDAQD
// the selection: sequence APDT
//  will select only the 2nd through 5th terms
// the selection: sequence APD
//  will select 2nd through 4th, and 9th to 11th.
// the selection: sequence "A.D"
//  will get 2-4 and 9-14
// and so on.
//   If a residue name is not normal, it becomes an X

// I am handed a list of strings for this selection
//  (eg, sequence APD "A.D" A to D)
// Since there are no non-standard residue names (ie, no '*')
// I'll interpret everything as a regex.  Also, phrases like
// "X to Y" are interpreted as "X.*Y"

static int atomsel_sequence(int argc, const char **argv, int *types,
			    int num, int *flgs)
{
   int i;
   // make a temporary array for marking the selected atoms
   int *selected = new int[num];
   for (i=0; i<num; i++) {
      selected[i] = FALSE;
   }
   // make the list of regex'es
   atomsel_regex_reset();
   GRegex **regex = (GRegex **) malloc( argc * sizeof(GRegex *));
   int num_r = 0;
   {
      GString pattern;
      atomsel_regex_swap();
      for (i=0; i<argc; i++) {
	 pattern  = argv[i];
	 if (types[i] >= 3) {  // get the next term (if a "to" element)
	    pattern += ".*";
	    pattern += argv[++i];
	 }
	 regex[num_r] = new GRegex(pattern);
//	 msgInfo << "Doing pattern " << pattern << sendmsg;
	 const char *err = NULL;
	 if (atomsel_regex_check(&err)) {
	    num_r ++;
	 } else {
	    msgErr << err << sendmsg;
	    delete regex[num_r];
	 }
      }
   } // constructed the regex array
   atomsel_regex_swap();
   if (num_r == 0) {
      return 0;
   }

   // construct a list of sequences from each protein (pfraglist)
   // and nucleic acid (nfragList)
   for (int fragcount=0; fragcount <2; fragcount++) {
      int pcount = atom_sel_mol->pfragList.num();
      int ncount = atom_sel_mol->nfragList.num();
      for (i=0; i< (fragcount == 0 ? pcount : ncount); i++) {
	 int size = (fragcount == 0 ? atom_sel_mol->pfragList[i]->num()
	                            : atom_sel_mol->nfragList[i]->num());
	 char *s = new char[size];
	 char *t = s;
	 int *mark = new int[size];
	 
	 for (int j=0; j<size; j++) {
	    int residuenum = ((fragcount == 0) ?
	          (*atom_sel_mol->pfragList[i])[j] :  
	          (*atom_sel_mol->nfragList[i])[j]);
	    int atomnum = (atom_sel_mol->residueList[residuenum]->atoms[0]);
	    MolAtom *atom = atom_sel_mol -> atom(atomnum);
	    char *resname = atom->resnamestr;
	    mark[j] = FALSE;
	    if (fragcount == 0) {
	       // protein translations
	       if (!strcasecmp( resname, "GLY")) {*t++ = 'G'; continue;}
	       if (!strcasecmp( resname, "ALA")) {*t++ = 'A'; continue;}
	       if (!strcasecmp( resname, "VAL")) {*t++ = 'V'; continue;}
	       if (!strcasecmp( resname, "PHE")) {*t++ = 'F'; continue;}
	       if (!strcasecmp( resname, "PRO")) {*t++ = 'P'; continue;}
	       if (!strcasecmp( resname, "MET")) {*t++ = 'M'; continue;}
	       if (!strcasecmp( resname, "ILE")) {*t++ = 'I'; continue;}
	       if (!strcasecmp( resname, "LEU")) {*t++ = 'L'; continue;}
	       if (!strcasecmp( resname, "ASP")) {*t++ = 'D'; continue;}
	       if (!strcasecmp( resname, "GLU")) {*t++ = 'E'; continue;}
	       if (!strcasecmp( resname, "LYS")) {*t++ = 'K'; continue;}
	       if (!strcasecmp( resname, "ARG")) {*t++ = 'R'; continue;}
	       if (!strcasecmp( resname, "SER")) {*t++ = 'S'; continue;}
	       if (!strcasecmp( resname, "THR")) {*t++ = 'T'; continue;}
	       if (!strcasecmp( resname, "TYR")) {*t++ = 'Y'; continue;}
	       if (!strcasecmp( resname, "HIS")) {*t++ = 'H'; continue;}
	       if (!strcasecmp( resname, "CYS")) {*t++ = 'C'; continue;}
	       if (!strcasecmp( resname, "ASN")) {*t++ = 'N'; continue;}
	       if (!strcasecmp( resname, "GLN")) {*t++ = 'Q'; continue;}
	    } else {
	       // nucleic acid translations
	       if (!strcasecmp( resname, "ADE")) {*t++ = 'A'; continue;}
	       if (!strcasecmp( resname, "A")) {*t++ = 'A'; continue;}
	       if (!strcasecmp( resname, "THY")) {*t++ = 'T'; continue;}
	       if (!strcasecmp( resname, "T")) {*t++ = 'T'; continue;}
	       if (!strcasecmp( resname, "CYT")) {*t++ = 'C'; continue;}
	       if (!strcasecmp( resname, "C")) {*t++ = 'C'; continue;}
	       if (!strcasecmp( resname, "GUA")) {*t++ = 'G'; continue;}
	       if (!strcasecmp( resname, "G")) {*t++ = 'G'; continue;}
	    }
	    // then I have no idea
	    *t++ = 'X';
	 }  // end loop 'j'; constructed the sequence for this protein
	 *t = 0; // terminate the string
	 
//	 msgInfo << "sequence " << i << " is: " << s << sendmsg;
	 // which of the residues match the regex(es)?
	 for (int r=0; r<num_r; r++) {
	    int len, start = 0, offset;
	    while ((offset = (regex[r] -> search(s, strlen(s), len, start)))
		   != -1) {
	       // then there was a match from offset to offset+len
//	       msgInfo << "start " << start << " offset " << offset << " len "
//		       << len << sendmsg;
	       for (int loop=offset; loop<offset+len; loop++) {
		  mark[loop] = 1;
	       }
	       start = offset+len;
	    }
	 }
	 
	 // the list of selected residues is in mark
	 // turn on the right atoms
	 for (int marked=0; marked<size; marked++) {
	    if (mark[marked]) {
	       int residuenum = (fragcount == 0 ?
				 (*atom_sel_mol->pfragList[i])[marked] :
				 (*atom_sel_mol->nfragList[i])[marked]);
	       for (int atomloop=0;
		    atomloop< atom_sel_mol->residueList[residuenum]->
		    atoms.num(); atomloop++) {
		  selected[atom_sel_mol->residueList[residuenum]->
			   atoms[atomloop]]=TRUE;
	       }
	    }
	 }
	 delete [] mark;
	 delete [] s;
      } // end loop i over the fragments
   }  // end loop 'fragcount'


   // get rid of the compiled regex's
   for (i=0; i<num_r; i++) {
      delete regex[i];
   }
   // copy the 'selected' array into 'flgs'
   for (i=0; i<num; i++) {
      flgs[i] = flgs[i] && selected[i];
   }
   return 1;
}

// Parse the annmm selection format
//  see: http://scop.mrc-lmb.cam.ac.uk/std/annmm/
//
// This is called as "annmm <text> <text> ... <text>" where 'text' is:
//   [model$][chain:][segid%][mer][#het][^alt][/atomtype]
//

// grab the "name" term from the front of the string.  This term
// ends in the character 'ch'
#define annmm_suffix(name, ch) {   \
  terms[name] = strchr(sptr, ch);  \
  if (terms[name]) {               \
    tmp = terms[name];             \
    terms[name] = sptr;            \
    sptr = tmp + 1;                \
    *tmp = 0;                      \
  }                                \
}

// grab the "name" term from the end of the string.  This term begins
// with the character 'ch'
#define annmm_prefix(name, ch) {   \
  terms[name] = strrchr(sptr, ch); \
  if (terms[name]) {               \
    *terms[name] = 0;              \
    terms[name]++;                 \
  }                                \
}
// convert the single term annmm string to the VMD equivalent
// replace ',' with ' '
// replace '-' with ' to '
// pass all other characters through
#define expand_macro(selstr, name, name2) {      \
  if (terms[name]) {                      \
    result += selstr;                     \
    char *temps;                          \
    for (temps = terms[name]; *temps; temps++) {  \
      if (*temps == ',') {                \
	result += " ";                    \
      } else if (*temps == '-') {         \
	result += " to ";                 \
      } else {                            \
	result += *temps;                 \
      }                                   \
    }                                     \
    result += name2;                      \
    result += " && ";                     \
  }                                       \
}    

// convert the annmm string into the usual VMD scripting language
static int atomsel_annmm(int argc, const char **argv, int */*types*/,
			 int num, int *flgs)
{
  int i, size = 0, tmpsize;
  char *s = NULL, *sptr;

  GString result; //
  char *terms[7];
  enum {MODEL, CHAIN, SEGID, MER, HET, ALT, ATOMTYPE};
  // for each of the annmm specs
  for (int annmm=0; annmm<argc; annmm++) {

    // make 's' point to a copy of the current annmm element
    // (the copy is needed since I modify the values)
    if (!s) {
      size = strlen(argv[annmm]) + 10; // extra space to reduce new'ing
      s = new char[size];
    } else {
      tmpsize = strlen(argv[annmm]) + 1;
      if (tmpsize > size) {
	delete [] s;
	size = tmpsize;
	s = new char[size];
      }
    }
    strcpy(s, argv[annmm]);

    // An annmm 'AnnElement' can contain several terms combined via a '|'
    // so go through each one of those
    char *start = s, *end;
    do {
      end = strchr(start, '|'); // end points to a '|' or is NULL
      if (end) {
	*end = 0;
      }
      sptr = start;

      char *tmp;
      annmm_suffix(MODEL, '$');    // [model$]
      annmm_suffix(CHAIN, ':');    // [chain:]
      annmm_suffix(SEGID, '%');    // [segid%]
      annmm_prefix(ATOMTYPE, '/'); // [/atomtype]
      annmm_prefix(ALT, '^');      // [^alt]
      annmm_prefix(HET, '#');      // [#het]
      if (sptr[0]) {               // [mer]
	terms[MER] = sptr;
      } else {
	terms[MER] = NULL;
      }
      // The terms either contain data or are NULL

      // Everything is now broken into the smallest parts, so
      // reconstruct it in the standard VMD selection language
      if (! (terms[0]||terms[1]||terms[2]||terms[3]||terms[4]||
	     terms[5]||terms[6]) ) {
	// then there is a blank entry
	result = "all || ";
      } else {
	result += "(";
	expand_macro("chain ", CHAIN, "");
	expand_macro("segname ", SEGID, "");
	// the het and mer elements use different namespaces
	if (terms[MER] && terms[HET]) {
	  result += '(';
	  expand_macro("protein resid ", MER, "");
	  result += " || ";
	  expand_macro("(not protein and resid ", HET, ")");
	  result += ") && ";
	} else {
	  expand_macro("protein resid ", MER, "");
	  expand_macro("(not protein and resid ", HET, ")");
	}
	expand_macro("name ", ATOMTYPE, "");
	// take off the last 4 characters (the " && ")
	result.del(int(result.length() - 4), 4);
	// and get ready for the next element
	result += ") || ";
      }

      // advance to the next '|' region
      start = end + 1;
    } while (end);
  }
  // delete the temp. string space
  if (s) {
    delete [] s;
  }
  // take off the last 4 characters (the " || ")
  result.del(int(result.length() - 4), 4);
  //  msgInfo << "Result is : " << (char *) (const char *) result << sendmsg;


  // Reparse using the existing selection language
  AtomSel *atomSel = new AtomSel(moleculeList);
  if (atomSel -> change( (char *) (const char *) result) ==
      AtomSel::NO_PARSE) {
    msgErr << "annmm: cannot understand:";
    for (i=0; i<argc; i++) {
      msgErr << " " << argv[i];
    }
    msgErr << sendmsg;
    delete atomSel;
    return 0;
  }
  if (atomSel->find(atom_sel_mol) < 0) {
    delete atomSel;
    msgErr << "annmm: unknown search error during: " <<
	   (const char *) result << sendmsg;
    return 0;
  }

  // and save the results
  for (i=0; i< num; i++) {
    flgs[i] = flgs[i] && atomSel -> on[i];
  }
  delete atomSel;
  return 1;
}


// access tcl variables/variable array with boolean values.  If these
// are atom seletions, return TRUE for those which are on, FALSE for
// those that are off.  These work on '@values'
static int atomsel_tcl_bool_variables(int num, int *flgs)
{
  // get the Tcl variable, if there is one
  char *selstr = (char *)NULL;
#ifdef VMDTCL
  int is_array = 0;
  selstr = Tcl_GetVar(uiText -> tclInterp, 
			    (char *) parsetree_nodestring+1, 0);
  // see if this is an array
  if (!selstr) {
    selstr = Tcl_GetVar2(uiText -> tclInterp,
			    (char *) parsetree_nodestring+1, "0", 0);
    if (selstr) is_array = 1;
  }
  // if they didn't work, is this a global array?
  if (!selstr) {
    selstr = Tcl_GetVar(uiText -> tclInterp, 
			    (char *) parsetree_nodestring+1, TCL_GLOBAL_ONLY);
  }
  if (!selstr) {
    selstr = Tcl_GetVar2(uiText -> tclInterp,
			 (char *) parsetree_nodestring+1, "0",
			 TCL_GLOBAL_ONLY);
    if (selstr) is_array = 1;
  }
#endif
  AtomSel *sel = (AtomSel *) NULL;
  // one last chance, could be a direct 'atomselect%d reference'
  sel = tcl_commands_get_sel((char *) parsetree_nodestring+1);

  if (sel == NULL && selstr == NULL) {
    msgErr << "selection: couldn't find Tcl variable " << parsetree_nodestring
	   << sendmsg;
    for (int i=num-1; i>=0; i--) { // couldn't find anything
      *flgs++ = 0;
    }
    return 0;
  }
    
  
#ifdef VMDTCL
  if (!sel) {  // get the atom selection from the given reference
    sel = tcl_commands_get_sel(selstr);
  }
  if (sel == NULL) {
    // this isn't an atom selection, so get the array or scalar value
    if (is_array) {
      // arrays are referenced from 0 to n-1; 0 is used if it doesn't exist
      char tmp_index[10];
      for (int i=0; i<num; i++) {      // go down the list
	if (flgs[i]) {                 //  of turned on atoms
	  sprintf(tmp_index, "%d", i); //  get the array value
	  if (!atoi(Tcl_GetVar2(uiText -> tclInterp,
			(char *) parsetree_nodestring+1, tmp_index, 0))) {
	    flgs[i] = 0;
	  }
	}
      } // end of going down list
      // end of array code
    } else {
      // is scalar value
      int f = (atoi(selstr) != 0);    // since I already have it, I use it
      for (int i=0; i<num; i++) {
	if (flgs[i]) {
	  flgs[i] = f;
	}
      }
    }
    return 0;
  }
#endif
  if (sel == NULL) {
    msgErr << "selection: Tcl variable " << parsetree_nodestring
	   << " is not a selection" << sendmsg;
    for (int i=num-1; i>=0; i--) { // couldn't find anything
      *flgs++ = 0;
    }
    return 0;
  }

  // make sure they have the same number of atoms
  if (sel -> num_atoms != atom_sel -> num_atoms) {
    msgErr << "selection for molecule " << atom_sel -> molid 
	   << " has " << atom_sel -> num_atoms 
	   << " atoms but the Tcl selection variable" << sendmsg;
    msgErr << parsetree_nodestring << " for molecule " << sel -> molid
	   << " has " << sel -> num_atoms 
	   << ": cannot intermix them" << sendmsg;
    for (int i=num-1; i>=0; i--) {
      *flgs++ = 0;
    }
    return 0;
  }
  // make sure it is the same molecule id
  if (sel -> molid != atom_sel -> molid ) {
    msgWarn << "The Tcl variable " << parsetree_nodestring << " from molecule "
	    << sel -> molid << " is being used" << sendmsg;
    msgWarn << "for molecule " << atom_sel -> molid 
	    << ".  They have the same number of atoms so it will work." 
	    << sendmsg;
  }
  // and do the assignment
  {
    int *ptr = sel->on;
    int i;
    for (i=num-1; i>=0; i--) {
      if (*flgs) *flgs = *ptr;
      flgs++;
      ptr++;
    }
  }
  return 1;

}

// access tcl variables/variable array with NON-boolean values.  If these
// are atom seletions, return "1" for those which are on, "0" for
// those that are off.  These work on '$values'
static int atomsel_tcl_variables(int num, GString *str, int *flgs)
{
  // get the Tcl variable, if there is one
  char *selstr = (char *)NULL;
#ifdef VMDTCL
  int is_array = 0;
  selstr = Tcl_GetVar(uiText -> tclInterp, 
			    (char *) parsetree_nodestring+1, 0);
  // see if this is an array
  if (!selstr) {
    selstr = Tcl_GetVar2(uiText -> tclInterp,
			    (char *) parsetree_nodestring+1, "0", 0);
    if (selstr) is_array = 1;
  }
  // if they didn't work, is this a global array?
  if (!selstr) {
    selstr = Tcl_GetVar(uiText -> tclInterp, 
			    (char *) parsetree_nodestring+1, TCL_GLOBAL_ONLY);
  }
  if (!selstr) {
    selstr = Tcl_GetVar2(uiText -> tclInterp,
			 (char *) parsetree_nodestring+1, "0",
			 TCL_GLOBAL_ONLY);
    if (selstr) is_array = 1;
  }
    
#endif
  AtomSel *sel = (AtomSel *) NULL;
  // one last chance, could be a direct 'atomselect%d reference'
  sel = tcl_commands_get_sel((char *) parsetree_nodestring+1);

  if (sel == NULL && selstr == NULL) {
    msgErr << "selection: couldn't find Tcl variable " << parsetree_nodestring
	   << sendmsg;
    for (int i=num-1; i>=0; i--) { // couldn't find anything
      *str++ = "";
    }
    return 0;
  }
    
#ifdef VMDTCL
  if (!sel) {  // get the atom selection from the given reference
    sel = tcl_commands_get_sel(selstr);
  }
  if (sel == NULL) {
    // this isn't an atom selection, so get the array or scalar value
    if (is_array) {
      // arrays are referenced from 0 to n-1; 0 is used if it doesn't exist
      char tmp_index[10];
      for (int i=0; i<num; i++) {      // go down the list
	if (flgs[i]) {                 //  of turned on atoms
	  sprintf(tmp_index, "%d", i); //  get the array value
	  str[i] = Tcl_GetVar2(uiText -> tclInterp,
			       (char *) parsetree_nodestring+1, tmp_index, 0);
	}
      } // end of going down list
      // end of array code
    } else {
      // is scalar value
      // since I already have it, I use it
      for (int i=0; i<num; i++) {
	if (flgs[i]) {
	  str[i] = selstr;
	}
      }
    }
    return 0;
  }
#endif
  if (sel == NULL) {
    msgErr << "selection: Tcl variable " << parsetree_nodestring
	   << " is not a selection" << sendmsg;
    for (int i=num-1; i>=0; i--) { // couldn't find anything
      *str++ = "";
    }
    return 0;
  }

  // make sure they have the same number of atoms
  if (sel -> num_atoms != atom_sel -> num_atoms) {
    msgErr << "selection for molecule " << atom_sel -> molid 
	   << " has " << atom_sel -> num_atoms 
	   << " atoms but the Tcl selection variable" << sendmsg;
    msgErr << parsetree_nodestring << " for molecule " << sel -> molid
	   << " has " << sel -> num_atoms 
	   << ": cannot intermix them" << sendmsg;
    for (int i=num-1; i>=0; i--) {
      *str++ = "";
    }
    return 0;
  }
  // make sure it is the same molecule id
  if (sel -> molid != atom_sel -> molid ) {
    msgWarn << "The Tcl variable " << parsetree_nodestring << " from molecule "
	    << sel -> molid << " is being used" << sendmsg;
    msgWarn << "for molecule " << atom_sel -> molid 
	    << ".  They have the same number of atoms so it will work." 
	    << sendmsg;
  }
  // and do the assignment
  {
    int i;
    for (i=num-1; i>=0; i--) {
      if (flgs[i]) str[i] = sel -> on[i] ? "1" : "0";
    }
  }
  return 1;

}

/************ support for RasMol selections ******************/
//// parse the rasmol primitive
// the full form of a primitive is (seems to be)
//   {[<resname>]}{<resid>}{:<chain>}{.<atom name>}
// if resname is only alpha, the [] can be dropped
// if chain is alpha, the : can be dropped
// resname only contains * if it is the first one ?
// ? cannot go in the resid
// * can only replace the whole field
static int atomsel_rasmol_primitive(int argc, const char **argv, int *,
				    int num, int *flgs)
{
  // for each word, (ignoring the quote flags)
  for (int word=0; word<argc; word++) {
    const char *rnm0 = argv[word]; // resname start
    const char *rnm1;              // and end position
    const char *rid0, *rid1;
    if (*rnm0 == '*') {
      rnm1 = rnm0 + 1;
      rid0 = rnm1;
    } else if (*rnm0 == '[') {
      rnm0++;
      rnm1 = rnm0;
      while (*rnm1 && *rnm1 != ']') { // find trailing bracket
	rnm1 ++;
      }
      if (rnm1 == rnm0) {  // for cases like [] and "["
	rid0 = rnm1;
      } else {
	if (*rnm1==']') {  // for cases like [so4]
	  rid0 = rnm1+1;
	} else {           // for (incorrect) cases like [so4
	  rid0 = rnm1;
	}
      }
    } else { // then must be alpha or ?
      rnm1 = rnm0;
      while (isalpha(*rnm1) || *rnm1 == '?') {  // find first non-alpha
	rnm1++;
      }
      rid0 = rnm1;
    }
    // got the resname

    // parse the resid
    rid1 = rid0;
    if (*rid1 == '*') {
      rid1++;
    } else {
      while (isdigit(*rid1)) {
	rid1++;
      }
    }

    // if this is the : delimiter, skip over it
    const char *chn0, *chn1;
    if (*rid1 == ':') {
      chn0 = rid1 + 1;
    } else {
      chn0 = rid1;
    }

    // get the chain
    // seek the . or end of string
    chn1 = chn0;
    while (*chn1 && *chn1 != '.') {
      chn1++;
    }

    const char *nm0, *nm1;
    if (*chn1 == '.') {
      nm0 = chn1 + 1;
    } else {
      nm0 = chn1;
    }
    nm1 = nm0;
    // seek the end of string
    while (*nm1) {
      nm1++;
    }
    

    // save the info into strings
    GString resname, resid, chain, name;
    const char *s;
    for (s=rnm0; s<rnm1; s++) {
      resname += *s;
    }
    for (s=rid0; s<rid1; s++) {
      resid += *s;
    }
    for (s=chn0; s<chn1; s++) {
      chain += *s;
    }
    for (s=nm0; s<nm1; s++) {
      name += *s;
    }
    //    msgInfo << "resname: " << (const char *) resname << sendmsg;
    //    msgInfo << "resid: " << (const char *) resid << sendmsg;
    //    msgInfo << "chain: " << (const char *) chain << sendmsg;
    //    msgInfo << "name: " << (const char *) name << sendmsg;

    // convert to the VMD regex ( ? => .? and * => .*)
    //   (however, if there is a * for the whole field, delete the field)
    if (resname == "*") resname = "";
    if (resid == "*") resid = "";
    if (chain == "*") chain = "";
    if (name == "*") name = "";
    resname.gsub("?", ".?"); resname.gsub("*", ".*");
    resid.gsub("?", ".?"); resid.gsub("*", ".*");
    chain.gsub("?", ".?"); chain.gsub("*", ".*");
    name.gsub("?", ".?"); name.gsub("*", ".*");
    // make everything upcase
    resname.upcase();
    resid.upcase();
    chain.upcase();
    name.upcase();

    // construct a new search
    GString search;
    if (resname != "") {
      search = "resname ";
      search += '"';
      search += resname;
      search += '"';
    }
    if (resid != "") {
      if (search != "") {
	search += " and resid ";
      } else {
	search = "resid ";
      }
      search += '"';
      search += resid;
      search += '"';
    }
    // if the chain length > 1, it is a segname
    int is_segname = chain.length() > 1;
    if (chain != "") {
      if (search != "") {
	search += (is_segname ? " and segname " : " and chain ");
      } else {
	search = (is_segname ? "segname " : "chain ");
      }
      search += '"';
      search += chain;
      search += '"';
    }
    if (name != "") {
      if (search != "") {
	search += " and name ";
      } else {
	search = "name ";
      }
      search += '"';
      search += name;
      search += '"';
    }
    msgInfo << "Search = " << search << sendmsg;

    if (search == "") {
      search = "all";
    }
    // and do the search
    AtomSel *atomSel = new AtomSel(moleculeList);
    if (atomSel -> change( (char *) (const char *) search) ==
	AtomSel::NO_PARSE) {
      msgErr << "rasmol: cannot understand: " << argv[word] << sendmsg;
      delete atomSel;
      continue;
    }
    if (atomSel -> find(atom_sel_mol) < 0) {
      delete atomSel;
      msgErr << "rasmol: unknown search error during: " <<
	argv[word] << sendmsg;
      continue;
    }
    
    // save the results
    {
      for (int i=0; i<num; i++) {
	flgs[i] = flgs[i] && atomSel -> on[i];
      }
    }
  }
  return 1;
}

// rasmol sets (the ones that already exist are marked *)
//    AT              Acidic          Acyclic
//    Aliphatic       Alpha           Amino
//    Aromatic      * Backbone        Basic
//    Bonded          Buried          CG
//    Charged         Cyclic          Cystine
//  * Helix           Hetero          Hydrogen
//    Hydrophobic     Ions            Large
//    Ligand          Medium          Neutral
//  * Nucleic         Polar         * Protein
//    Purine          Pyrimidine      Selected
//  * Sheet         * Sidechain       Small
//    Solvent         Surface       * Turn
//  * Water
#define generic_atomsel_replacement(fctnname,newselstr) \
static int fctnname(int num, int *flgs) {               \
  AtomSel *atomSel = new AtomSel(moleculeList);         \
  atomSel -> change(newselstr);                         \
  atomSel -> find(atom_sel_mol);                        \
  for (int i=0; i<num; i++) {                           \
    flgs[i] = flgs[i] && atomSel -> on[i];              \
  }                                                     \
  delete atomSel;                                       \
  return 1;                                             \
}

generic_atomsel_replacement(atomsel_rasmol_at, "resname ADE A THY T")
generic_atomsel_replacement(atomsel_rasmol_acidic, "resname ASP GLU")
generic_atomsel_replacement(atomsel_rasmol_acyclic, 
			    "protein and not cyclic")
generic_atomsel_replacement(atomsel_rasmol_aliphatic,
			    "resname ALA GLY ILE LEU VAL")
generic_atomsel_replacement(atomsel_rasmol_alpha, "protein and name CA")
generic_atomsel_replacement(atomsel_rasmol_amino, "protein")
generic_atomsel_replacement(atomsel_rasmol_aromatic,
			    "resname HIS PHE TRP TYR")
generic_atomsel_replacement(atomsel_rasmol_basic, "resname ARG HIS LYS")
generic_atomsel_replacement(atomsel_rasmol_bonded, "numbonds > 0")
generic_atomsel_replacement(atomsel_rasmol_buried, 
			    "resname ALA LEU VAL ILE PHE CYS MET TRP")
generic_atomsel_replacement(atomsel_rasmol_cg, "resname CYT C GUA G")
generic_atomsel_replacement(atomsel_rasmol_charged, "basic or acidic");
generic_atomsel_replacement(atomsel_rasmol_cyclic, 
			    "resname HIS PHE PRO TRP TYR")

/* cystine */

// best I can do
generic_atomsel_replacement(atomsel_rasmol_hetero, "not (protein or nucleic)")
generic_atomsel_replacement(atomsel_rasmol_hydrogen, "name \"[0-9]?H.*\"")
generic_atomsel_replacement(atomsel_rasmol_hydrophobic,
			    "resname ALA LEU VAL ILE PRO PHE MET TRP")
/* ions */

generic_atomsel_replacement(atomsel_rasmol_large,
			    "protein and not (small or medium)")

/* ligand */

generic_atomsel_replacement(atomsel_rasmol_medium, 
			    "resname VAL THR ASP ASN PRO CYS ASX PCA HYP")
generic_atomsel_replacement(atomsel_rasmol_neutral, 
	    "resname VAL PHE GLN TYR HIS CYS MET TRP ASX GLX PCA HYP")

generic_atomsel_replacement(atomsel_rasmol_polar, 
			    "protein and not hydrophobic")

generic_atomsel_replacement(atomsel_rasmol_purine, "resname ADE A GUA G")
generic_atomsel_replacement(atomsel_rasmol_pyrimidine, 
			    "resname CYT C THY T URI U")

/* selected -- cannot implement; different philosophies */

generic_atomsel_replacement(atomsel_rasmol_small, "resname ALA GLY SER")
/* solvent */
generic_atomsel_replacement(atomsel_rasmol_surface,
			    "protein and not buried");

////////// allow calls like 'asp38'
// These are three characters followed by at least 1 number
// The letters are resname, the numbers resid
#ifdef FIXED_THE_PROBLEM_IN_THE_PARSER_SO_THIS_WORKS_CORRECTLY
static int atomsel_resname_resid(int num, int *flgs)
{
  char resname[10];
  char resid[10];
  // get the resname and resid
  // resname is up to the first digit
  int i = 0;
  while (i < 8) {  // 8 is on the safe side
    resname[i] = toupper(((char *) parsetree_nodestring)[i]);
    if (isdigit(resname[i])) {
      resname[i] = 0;
      break;
    }
    i++;
  }
  resname[i] = 0;  // ensure termination
  // now get the number (this is easier: I deal with a normal string)
  strncpy(resid, ((char *) parsetree_nodestring) + i, 9);
  resid[9] = 0;    // ensure termination
  
  // do the search with the new parameters
  char newselstr[60];
  sprintf(newselstr, "resname %s and resid %s", resname, resid);
  AtomSel *atomSel = new AtomSel(moleculeList);
  atomSel -> change(newselstr);
  atomSel -> find(atom_sel_mol);
  for (i=0; i<num; i++) {
    flgs[i] = flgs[i] && atomSel -> on[i];
  }
  delete atomSel;
  return 1;
}
#endif


// called in Global.C . This sets up the function pointers for the
// seletion commands in the global variable
void atomSelParser_init(void)
{
   atomSelParser.add_keyword("name", "name", atomsel_name, NULL);
   atomSelParser.add_keyword("type", "type", atomsel_type, NULL);
   
   atomSelParser.add_keyword("index", "index", 
			     atomsel_index, NULL);
   atomSelParser.add_keyword("residue", "residue",
                             atomsel_residue, NULL);
   atomSelParser.add_keyword("resname",
			     "resname", atomsel_resname, NULL);
   atomSelParser.add_keyword("resid", "resid", 
			     atomsel_resid, NULL);
   atomSelParser.add_keyword("chain",
			     "chain", atomsel_chain, NULL);
   atomSelParser.add_keyword("\\(segname\\)\\|\\(segid\\)",
			     "segname", atomsel_segname, NULL);

   atomSelParser.add_singleword("all", "all", atomsel_all, NULL);
   atomSelParser.add_singleword("none", "none", 
				atomsel_none, NULL);

   atomSelParser.add_keyword("fragment",
			     "fragment", atomsel_fragment, NULL);
   atomSelParser.add_keyword("pfrag",
			     "pfrag", atomsel_pfrag, NULL);
   atomSelParser.add_keyword("nfrag",
			     "nfrag", atomsel_nfrag, NULL);
   atomSelParser.add_keyword("numbonds",
			     "numbonds", atomsel_numbonds, NULL);

   atomSelParser.add_singleword("backbone",
			    "backbone", atomsel_backbone, NULL);
   atomSelParser.add_singleword("sidechain",
			    "sidechain", atomsel_sidechain, NULL);
   atomSelParser.add_singleword("protein",
			    "protein", atomsel_protein, NULL);
   atomSelParser.add_singleword("nucleic",
			    "nucleic", atomsel_nucleic, NULL);
   atomSelParser.add_singleword("waters?",
			    "water", atomsel_water, NULL);

   ///// rasmol commands  -- note they are all lowercase!
   // the "at" command may change in the future; "at" is too english
   atomSelParser.add_singleword("at", "at", atomsel_rasmol_at, NULL);
   atomSelParser.add_singleword("acidic", "acidic", 
				atomsel_rasmol_acidic, NULL);
   atomSelParser.add_singleword("acyclic", "acyclic", 
				atomsel_rasmol_acyclic, NULL);
   atomSelParser.add_singleword("aliphatic", "aliphatic", 
				atomsel_rasmol_aliphatic, NULL);
   atomSelParser.add_singleword("alpha", "alpha", atomsel_rasmol_alpha,
				NULL);
   atomSelParser.add_singleword("amino", "amino", atomsel_rasmol_amino,
				NULL);
   atomSelParser.add_singleword("aromatic", "aromatic", 
				atomsel_rasmol_aromatic, NULL);
   atomSelParser.add_singleword("basic", "basic", atomsel_rasmol_basic,
				NULL);
   atomSelParser.add_singleword("bonded", "bonded", atomsel_rasmol_bonded,
				NULL);
   atomSelParser.add_singleword("buried", "buried", atomsel_rasmol_buried,
				NULL);
   atomSelParser.add_singleword("cg", "cg", atomsel_rasmol_cg, NULL);
   atomSelParser.add_singleword("charged", "charged", atomsel_rasmol_charged,
				NULL);
   atomSelParser.add_singleword("cyclic", "cyclic", atomsel_rasmol_cyclic,
				NULL);
   atomSelParser.add_singleword("hetero", "hetero", atomsel_rasmol_hetero,
				NULL);
   atomSelParser.add_singleword("hydrogen", "hydrogen", 
				atomsel_rasmol_hydrogen, NULL);
   atomSelParser.add_singleword("hydrophobic", "hydrophobic", 
				atomsel_rasmol_hydrophobic, NULL);
   atomSelParser.add_singleword("large", "large", atomsel_rasmol_large, NULL);
   atomSelParser.add_singleword("medium", "medium", atomsel_rasmol_medium,
				NULL);
   atomSelParser.add_singleword("neutral", "neutral", atomsel_rasmol_neutral,
				NULL);
   atomSelParser.add_singleword("polar", "polar", atomsel_rasmol_polar,
				NULL);
   atomSelParser.add_singleword("purine", "purine", atomsel_rasmol_purine,
				NULL);
   atomSelParser.add_singleword("pyrimidine", "pyrimidine", 
				atomsel_rasmol_pyrimidine, NULL);
   atomSelParser.add_singleword("small", "small", atomsel_rasmol_small, NULL);
   atomSelParser.add_singleword("surface", "surface", atomsel_rasmol_surface,
				NULL);

   // secondary structure functions
   atomSelParser.add_singleword("helix", "helix", 
				atomsel_helix, atomsel_set_helix);
   atomSelParser.add_singleword("alpha_helix", "alpha_helix", 
				atomsel_alpha_helix, atomsel_set_helix);
   atomSelParser.add_singleword("helix_3_10", "helix_3_10", 
				atomsel_3_10_helix, atomsel_set_3_10_helix);
   atomSelParser.add_singleword("pi_helix", "pi_helix", 
				atomsel_pi_helix, atomsel_set_pi_helix);
   atomSelParser.add_singleword("\\(sheet\\)\\|\\(beta_?sheet\\)", "sheet", 
				atomsel_sheet, atomsel_set_sheet);
   atomSelParser.add_singleword("extended_beta", "extended_beta", 
				atomsel_extended_sheet, 
				atomsel_set_sheet);
   atomSelParser.add_singleword("bridge_beta", "bridge_beta", 
				atomsel_bridge_sheet, 
				atomsel_set_bridge_sheet);
   atomSelParser.add_singleword("turn", "turn", 
				atomsel_turn, atomsel_set_turn);
   atomSelParser.add_singleword("coil", "coil", 
				atomsel_coil, atomsel_set_coil);
   atomSelParser.add_keyword("structure", "structure",
			     atomsel_sstruct, atomsel_set_sstruct);


   atomSelParser.add_keyword("x", "x", atomsel_xpos, 
			     atomsel_set_xpos);
   atomSelParser.add_keyword("y", "y", atomsel_ypos,
			     atomsel_set_ypos);
   atomSelParser.add_keyword("z", "z", atomsel_zpos,
			     atomsel_set_zpos);
   atomSelParser.add_keyword("radius", "radius", atomsel_radius, 
			     atomsel_set_radius);
   atomSelParser.add_keyword("mass", "mass", atomsel_mass, 
			     atomsel_set_mass);
   atomSelParser.add_keyword("charge", "charge", atomsel_charge, 
			     atomsel_set_charge);
   atomSelParser.add_keyword("beta", "beta", atomsel_beta, 
			     atomsel_set_beta);
   atomSelParser.add_keyword("occupancy", "occupancy", 
			     atomsel_occupancy, atomsel_set_occupancy);


   atomSelParser.add_stringfctn("sequence", "sequence", 
				atomsel_sequence, NULL);
   atomSelParser.add_stringfctn("annmm", "annmm", 
				atomsel_annmm, NULL);
   atomSelParser.add_stringfctn("rasmol", "rasmol", 
				atomsel_rasmol_primitive, NULL);

   atomSelParser.add_keyword("ufx", "ufx", atomsel_user_force_x, 
			     atomsel_set_user_force_x);
   atomSelParser.add_keyword("ufy", "ufy", atomsel_user_force_y, 
			     atomsel_set_user_force_y);
   atomSelParser.add_keyword("ufz", "ufz", atomsel_user_force_z, 
			     atomsel_set_user_force_z);
   atomSelParser.add_keyword("changed_userforces", "changed_userforces",
			     atomsel_user_forces_changed, 
			     atomsel_modify_user_forces);

   // three letters for resname, 1 or more letters for resid
   ////  DOESN'T WORK WITH PARSER -- breaks 'segname PRO1'
   //   atomSelParser.add_singleword("[a-zA-Z][a-zA-Z][a-zA-Z][0-9]+",
   //				"resnameID", atomsel_resname_resid,
   //				NULL);
   atomSelParser.add_keyword("\\$.*", "$variables",
			     atomsel_tcl_variables, NULL);
   atomSelParser.add_singleword("\\@.*", "@variables",
			     atomsel_tcl_bool_variables, NULL);
   atomSelParser.add_keyword("bondlist", "bondlist",
			     atomsel_bondlist, NULL);

   // and a few functions for good measure
   atomSelParser.add_function("sqr", "sqr", atomsel_square);
   atomSelParser.add_function("sqrt", "sqrt", sqrt);
   atomSelParser.add_function("abs", "abs", fabs);
   atomSelParser.add_function("floor", "floor", floor);
   atomSelParser.add_function("ceil", "ceil", ceil);
//   atomSelParser.add_function("trunc", "trunc", trunc);
   atomSelParser.add_function("rint", "rint", rint);
   
			      
   atomSelParser.add_function("sin", "sin", sin);
   atomSelParser.add_function("cos", "cos", cos);
   atomSelParser.add_function("tan", "tan", tan);
   atomSelParser.add_function("atan", "atan", atan);
   atomSelParser.add_function("asin", "asin", asin);
   atomSelParser.add_function("acos", "acos", acos);

   atomSelParser.add_function("sinh", "sinh", sinh);
   atomSelParser.add_function("cosh", "cosh", cosh);
   atomSelParser.add_function("tanh", "tanh", tanh);

   atomSelParser.add_function("exp", "exp", exp);
   atomSelParser.add_function("log", "log", log);
   atomSelParser.add_function("log10", "log10", log10);
   atomSelParser.add_function("erf", "erf", erf);
   atomSelParser.add_function("erfc", "erfc", erfc);

}

//////////////////////////  constructor and destructor
// constructor; parse string and see if OK
AtomSel::AtomSel(MoleculeList *mlist) {
  
  // initialize variables
  molList = mlist;
  mol = NULL;
  molid = -1;
  selected = NO_PARSE;
  on = NULL;
  cmdStr = NULL;
  tree = NULL;
  num_atoms = 0;
  change(DEFAULT_ATOMSEL);
  which_frame = TS_NOW;
}


// copy constructor

AtomSel::AtomSel(AtomSel& as) {
   cmdStr = NULL;
   tree = NULL;
   on = NULL;
   selected = NO_PARSE;
   num_atoms = 0;
   mol = as.mol;
   molid = as.molid;
   molList = as.molList;
   if (as.cmdStr) {       // now make the parse tree
      change(as.cmdStr);
   } else {
      change(DEFAULT_ATOMSEL);
   }
   which_frame = as.which_frame;
   find(mol);       // and find the data
  
}


// destructor; free up space
AtomSel::~AtomSel(void) {
   if(on) {
      delete [] on;
   }
   if (tree) {
      delete tree;
   }
   if (cmdStr) {
      delete [] cmdStr;
   }
}

// assignment operator, to change the current settings.
// The BaseMolecule gets the AtomSel for the whole system.
// That means it does NOT change the current molecule.
AtomSel& AtomSel::operator=(const AtomSel &as) {
  if (tree) {
     delete tree;
     tree = NULL;
  }
  if (on) {
     delete [] on;
     on = NULL;
  }
  change(as.cmdStr);
  return *this;
}


// provide new settings; does a 'find' at the end if a mol has
// been previously provided.  Return the number of atoms found
// or ATOMSEL_NO_PARSE if bad text
// or ATOMSEL_NO_MOL if no molecule
int AtomSel::change(char *newcmd)
{
   ParseTree *newtree = atomSelParser.parse(newcmd);
   if (!newtree) {
      return NO_PARSE;
   }
   if (cmdStr) delete [] cmdStr;
   cmdStr = stringdup(newcmd);
   if (tree) delete tree;
   tree = newtree;
   // and evaluate
   int i = find(mol);
   return i;
}

// find the selection for a molecule
// return the number of atoms found
// or NO_MOL or NO_PARSE if no good
int AtomSel::find(BaseMolecule *m)
{
   if (!tree) {
      return NO_PARSE;
   }
   if (!m) {
      return NO_MOL;
   }
   mol = m;
   atom_sel_mol = (DrawMolecule *) mol;
   atom_sel = this;
   molid = m -> id();
   if (on) delete [] on;
   on = new int[mol->nAtoms];
   num_atoms = mol->nAtoms;
   tree->evaluate(mol->nAtoms, on);
   // count up the number of 1s (there are only 0s and 1s)
   // 'selected' is a class variable
   selected = 0;
   for (int i=mol->nAtoms-1; i>=0; i--) {
      selected += on[i];
   }
   return selected;
}

// return NULL if the molecule was deleted
BaseMolecule *AtomSel::molecule(void) {
  // check that the molecule exists
  if (moleculeList -> mol_index_from_id(mol -> id()) < 0) {
    return NULL;
  }
  return mol;
}

// use this molecule for doing atom selections
void AtomSel::use(void) {
  atom_sel_mol =  (DrawMolecule *) mol;
  atom_sel = this;
}

// return the current coordinates (or NULL if error)
float *AtomSel::coordinates(void)
{
  BaseMolecule *mol = molecule();
  if (!mol) {
    msgErr << "No molecule" << sendmsg;
    return NULL;                    // no molecules
  }
  if (! mol -> is_current()) {
    msgErr << "No coordinates" << sendmsg;
    return NULL;  // no data
  }
  switch (which_frame) {
  case TS_LAST : return mol -> item( mol -> Animation::num() - 1) -> pos;
  case TS_NOW : return mol -> current() -> pos;
  default:
    if (!mol -> item(which_frame)) {   // if past end of coords
                                      // return last coord
      return mol -> item( mol -> Animation::num() - 1) -> pos; 

    }
  }
  return mol -> item(which_frame) -> pos;
}

