/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Buttons.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 1995/12/19 17:01:56 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Provides an interface to request a set of buttons of various types to
 * have a common method of query.  A Button object is created by providing
 * a DisplayDevice which knows how to read the state of specific buttons.
 * Each Button object is told which 'bank' of buttons to use, which then
 * appear as buttons 0, 1, and 2 for the user.  There are several different
 * button 'banks':
 *      POINTER_BANK_0 : the mouse buttons (0=left,1=middle,2=right)
 *      POINTER_BANK_1 : an alternate bank of mouse buttons; if the device
 *                       only has one set of mouse buttons, this is the
 *                       same as POINTER_BANK_0.
 *      KEY_BANK_0 : functions keys set #1 (0=F1, 1=F2, 2=F3)
 *      KEY_BANK_1 : functions keys set #2 (0=F5, 1=F6, 2=F7)
 *      KEY_BANK_2 : functions keys set #3 (0=F9, 1=F10, 2=F11)
 *      ESC_BANK : the escape key ... all three buttons are the same (ESC)
 ***************************************************************************/

#include <string.h>
#include "Buttons.h"
#include "DisplayDevice.h"
#include "utilities.h"
#include "ToolControl.h"


/////////////////////////  constructor: number of buttons this object contains
Buttons::Buttons(DisplayDevice *d, int b) {

  // default values
  has_changed = FALSE;
  myBank = POINTER_BANK_0;
  bankStart = DisplayDevice::B_LEFT;
  for(int i=0; i < NUM_BUTTONS; i++)
    state[i] = changes[i] = FALSE;

  // set values from arguments
  dispDev = d;
  bank(b);
}


// change which bank to use
void Buttons::bank(int b) {

  switch(b) {
    case POINTER_BANK_0:        bankStart = DisplayDevice::B_LEFT; break;
    case POINTER_BANK_1:        bankStart = DisplayDevice::B2_LEFT; break;
    case KEY_BANK_0:            bankStart = DisplayDevice::B_F1; break;
    case KEY_BANK_1:            bankStart = DisplayDevice::B_F5; break;
    case KEY_BANK_2:            bankStart = DisplayDevice::B_F9; break;
    case ESC_BANK:              bankStart = DisplayDevice::B_ESC; break;
    default: return;
  }

  // if here, b was a legal value
  myBank = b;
}


// force a button to be down (or up).  
void Buttons::press_button(ToolControl *tool, int buttonnum, int go_down) {
  if (!tool || buttonnum < 0 || buttonnum >= num()) return;

  for (int i=0; i<NUM_BUTTONS; i++) {
    changes[i] = 0;
  }

  if (go_down) {
    // this is a special state, indicated by the "3".  A normal
    // down state has a value of "1"
    if (!state[buttonnum]) { // button just pressed, so do the call
      state[buttonnum] = 3;
      has_changed = TRUE;
      changes[buttonnum] = 1;
      tool -> top_tool() -> button_event(tool, &(tool->my_info));
    } else {
      state[buttonnum] = 3;
    }
  } else {
    if (state[buttonnum]) {
      state[buttonnum] = 0; // there is no special state for being forced up
      changes[buttonnum] = -1;
      has_changed = TRUE;
      tool -> top_tool() -> button_event(tool, &(tool->my_info));
    } else {
      state[buttonnum] = 0;
    }
  }
}


// check the current state of all the buttons, and save them (TRUE=pressed)
void Buttons::check_event(void) {

  int dispButtons = (myBank == ESC_BANK ? 1 : NUM_BUTTONS);

  // get the state of each button and store it, and check if it has changed
  has_changed = FALSE;
  for(int i=0; i < dispButtons; i++) {
    int new_state = dispDev->button_down(bankStart + i);
    //    msgInfo << i << " state " << state[i] << " new_state " << new_state
    //	    << sendmsg;
    if (state[i] == 3) {      // I'm in the special down state
      if (new_state == 1) {   //  and only if the button is down do I
	state[i] = new_state; //  revert to the normal state
	changes[i] = 0;
      } else {
	changes[i] = 0;
      }
    } else {
      changes[i] = new_state - state[i];
      state[i] = new_state;
    }
    has_changed = has_changed || (changes[i] != 0);
  }
}


