/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Buttons.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1995/12/19 17:01:56 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Provides an interface to request a set of buttons of various types to
 * have a common method of query.  A Button object is created by providing
 * a DisplayDevice which knows how to read the state of specific buttons.
 * Each Button object is told which 'bank' of buttons to use, which then
 * appear as buttons 0, 1, and 2 for the user.  There are several different
 * button 'banks':
 *      POINTER_BANK_0 : the mouse buttons (0=left,1=middle,2=right)
 *      POINTER_BANK_1 : an alternate bank of mouse buttons; if the device
 *                       only has one set of mouse buttons, this is the
 *                       same as POINTER_BANK_0.
 *      KEY_BANK_0 : functions keys set #1 (0=F1, 1=F2, 2=F3)
 *      KEY_BANK_1 : functions keys set #2 (0=F5, 1=F6, 2=F7)
 *      KEY_BANK_2 : functions keys set #3 (0=F9, 1=F10, 2=F11)
 *      ESC_BANK : the escape key ... all three buttons are the same (ESC)
 ***************************************************************************/
#ifndef BUTTONS_H
#define BUTTONS_H

#include "utilities.h"
class DisplayDevice;

#define NUM_BUTTONS     3
class ToolControl;

// class for the different types of button inputs
class Buttons {

public:
  // different button banks we can use
  enum ButtonBanks { POINTER_BANK_0, POINTER_BANK_1, KEY_BANK_0, KEY_BANK_1,
			KEY_BANK_2, ESC_BANK, TOTAL_BUTTON_BANKS };

private:
  // device to ask about button states
  DisplayDevice *dispDev;

  // which bank of buttons to use, (F1, F2, F3) , (F5, ...)
  int myBank;  

  // the first 'real' button code for the current bank
  int bankStart;

  // place to store the button info so I don't constantly repoll the same data
  int state[NUM_BUTTONS];      

  // not FALSE indicated that that button changed since the last polling
  int changes[NUM_BUTTONS];    

  // has anything changed since the last time?  Here for speed
  int has_changed;

public:
  // constructor: device to check for events, which bank of Buttons to use
  Buttons(DisplayDevice *, int = POINTER_BANK_0);

  // number of buttons
  int num(void) { return NUM_BUTTONS; }
  
  // return which bank is being used, or change which bank
  int bank(void) { return myBank; }
  void bank(int);

  // return state of a button -- TRUE is on, FALSE is off
  int button(int buttonnum) { 
    return 0<= buttonnum && buttonnum < num() ?
             state[buttonnum] : FALSE;
  }

  // force a button to change mode.  If down, the only way to get
  // back up is for an explicit up or if the physical button comes up.
  // This is designed for the text control of the button state where I
  // want to override the input button.
  //   go_down == TRUE is "down"
  void press_button(ToolControl *tool, int buttonnum, int go_down);
  
  // return the value by which a button has changed
  int change(int buttonnum) {
     return 0<= buttonnum && buttonnum < num() ?
              changes[buttonnum] : 0;
  }

  // for speed, I want to know if anything has changed since the last
  // polling w/o actually checking all the state variables
  int changed(void) { return has_changed; }
  
  // get the current button state.  If it changed, generate a Command
  void check_event(void);
};

#endif

