/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdDisplay.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.12 $	$Date: 1997/03/24 17:21:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   These are the Commands that control the various aspects
 * of the display, like, the clipping planes, eye separation, etc.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdDisplay.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "ColorList.h"
#include "LightList.h"
#include "Axes.h"
#include "Stage.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// These are the Command:: enums used by this file:
//  DISP_RESHAPE, DISP_STEREO, DISP_PROJ, DISP_EYESEP, DISP_FOCALLEN, 
//  DISP_LIGHT_ON, DISP_LIGHT_HL, DISP_LIGHT_ROT, DISP_MATERIALS_CHANGE,
//  DISP_CLIP, DISP_DEPTHCUE, DISP_ANTIALIAS, DISP_SCRHEIGHT, DISP_SCRDIST,
//  DISP_DETAIL, DISP_ALTDETAIL, CMD_AXES, CMD_STAGE


////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'display'; return TRUE if an error occurs.
#define TCL_RET(fmt, val)      \
sprintf(s, fmt, val);          \
Tcl_AppendElement(interp, s);  \
return FALSE;

int text_cmd_display(int argc, char **argv, CommandQueue *cmdQueue, 
		     int id, Tcl_Interp *interp) {

  // not much help, but anything is nice
  if (argc <= 1) {
    interp -> result = 
"display get <eyesep | focallength | height | distance | antialias | depthcueue |\n"
"             detail | altdetail | details | depthsort | stereo | stereomodes |\n"
"             projection | projections | nearclip | farclip>\n"
"display <reshape | resetview>\n"
"display <eyesep | focallength | height | distance | antialias | depthcue |\n"
"         stereo | stereo | detail | altdetail | details | depthsort> newvalue\n"
"display <nearclip | farclip> <set | add> newvalue\n"
"display update [on | off | status | ui]";
    return TCL_ERROR;
  }

  // the new 'get' commands
  if (argc == 3 && !strupncmp(argv[1], "get", CMDLEN)) {
    char s[30];
    if        (!strupncmp(argv[2], "eyesep", CMDLEN)) {
      TCL_RET("%f", display -> eyesep());
    } else if (!strupncmp(argv[2], "focallength", CMDLEN)) {
      TCL_RET("%f", display -> eye_dist());
    } else if (!strupncmp(argv[2], "height", CMDLEN)) {
      TCL_RET("%f", display -> screen_height());
    } else if (!strupncmp(argv[2], "distance", CMDLEN)) {
      TCL_RET("%f", display -> distance_to_screen());
    } else if (!strupncmp(argv[2], "antialias", CMDLEN)) {
      Tcl_AppendElement(interp, display -> aa_enabled() ? "on" : "off");
      return FALSE;
    } else if (!strupncmp(argv[2], "depthcue", CMDLEN)) {
      Tcl_AppendElement(interp, display -> cueing_enabled() ? "on" : "off");
      return FALSE;
    } else if (!strupncmp(argv[2], "detail", CMDLEN)) {
      Tcl_AppendElement(interp, (char *) (display -> detail_name(
				        display -> detail_level()))
			);
      return FALSE;
    } else if (!strupncmp(argv[2], "altdetail", CMDLEN)) {
      Tcl_AppendElement(interp, (char *) (display -> detail_name(
				         display -> alt_detail_level()))
			);
      return FALSE;
    } else if (!strupncmp(argv[2], "details", CMDLEN)) {
      int i;
      for (i = display -> min_detail_level(); 
	   i<= display -> max_detail_level();
	   i++) {
	Tcl_AppendElement(interp, (char *) display -> detail_name(i));
      }
      return FALSE;
    } else if (!strupncmp(argv[2], "depthsort", CMDLEN)) {
      Tcl_AppendElement(interp, display -> depth_sort() ? "on" : "off");
      return FALSE;
    } else if (!strupncmp(argv[2], "stereo", CMDLEN)) {
      Tcl_AppendElement(interp, 
			display -> stereo_name(display -> stereo_mode()));
      return FALSE;
    } else if (!strupncmp(argv[2], "stereomodes", CMDLEN)) {
      int i;
      for (i=0; i<display->num_stereo_modes(); i++) {
	Tcl_AppendElement(interp, display -> stereo_name(i));
      }
      return FALSE;
    } else if (!strupncmp(argv[2], "projection", CMDLEN)) {
      // blantantly hardcoded
      if (display -> projection() == DisplayDevice::PERSPECTIVE) {
	interp -> result = "perspective";
      } else {
	interp -> result = "orthographic";
      }
      return FALSE;
    } else if (!strupncmp(argv[2], "projections", CMDLEN)) {
      interp -> result = "perspective orthographic";
      return FALSE;
    } else if (!strupncmp(argv[2], "nearclip", CMDLEN)) {
      TCL_RET("%f", display -> near_clip());
    } else if (!strupncmp(argv[2], "farclip", CMDLEN)) {
      TCL_RET("%f", display -> far_clip());
    } else {
      interp -> result = "possible parameters to 'display get' are:\n"
	"eyesep focallength height distance antialias depthcue detail\n"
	"altdetail details depthsort stereo stereomodes nearclip farclip\n";
	return TRUE;
    }
    /// return FALSE;  // never reached
  }

  if(argc == 2) {
    if(!strupncmp(argv[1],"reshape",CMDLEN)) {
      cmdQueue->append(new CmdDisplayReshape(id));
    }  else if(!strupncmp(argv[1],"resetview",CMDLEN)) {
      cmdQueue->append(new CmdResetView(id));
    } else if(!strupncmp(argv[1],"update",CMDLEN)) {
      cmdQueue->append(new CmdDisplayUpdate(NULL, id));
    } else
      return TRUE;

  } else if(argc == 3) {
    if(!strupncmp(argv[1],"eyesep",CMDLEN)) {
      cmdQueue->append(new CmdDisplayEyesep(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"focallength",CMDLEN)) {
      cmdQueue->append(new CmdDisplayFocallen(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"height",CMDLEN)) {
      cmdQueue->append(new CmdDisplayScreenHeight(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"distance",CMDLEN)) {
      cmdQueue->append(new CmdDisplayScreenDistance(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"antialias",CMDLEN)) {
      cmdQueue->append(new CmdDisplayAAOn(!strupcmp(argv[2],"on"), id));
    } else if(!strupncmp(argv[1],"depthcue",CMDLEN)) {
      cmdQueue->append(new CmdDisplayDepthcueOn(!strupcmp(argv[2],"on"), id));
    } else if(!strupncmp(argv[1],"stereo",CMDLEN)) {
      int i, modes = display->num_stereo_modes();
      for(i=0; i < modes; i++)
        if(!strupncmp(argv[2],display->stereo_name(i), CMDLEN))
	  break;
      cmdQueue->append(new CmdDisplayStereo(i, id));
    } else if (!strupncmp(argv[1], "projection", CMDLEN) ||
	       !strupncmp(argv[1], "proj", CMDLEN)) {
      if (!strupncmp(argv[2], "perspective", CMDLEN) ||
	  !strupncmp(argv[2], "persp", CMDLEN)) {
	cmdQueue -> append(new CmdDisplayProj(DisplayDevice::PERSPECTIVE));
      } else if (!strupncmp(argv[2], "orthographic", CMDLEN) ||
		 !strupncmp(argv[2], "ortho", CMDLEN)) {
	cmdQueue -> append(new CmdDisplayProj(DisplayDevice::ORTHOGRAPHIC));
      } else {
	return TRUE;
      }
    } else if (!strupncmp(argv[1], "detail", CMDLEN)) {
      // check if it is one of the strings
      int i;
      for (i=display->min_detail_level(); i<=display->max_detail_level();
	   i++) {
	if (!strupncmp(argv[2], display->detail_name(i), CMDLEN)) {
	  cmdQueue -> append(new CmdDisplayDetail(i, id));
	  return FALSE;
	}
      }
      // otherwise, check if it is a number
      if (isdigit(argv[2][0])) {
	cmdQueue -> append(new CmdDisplayDetail(atoi(argv[2]), id));
	return FALSE;
      }
      return TRUE;
    } else if (!strupncmp(argv[1], "altdetail", CMDLEN)) {
      // check if it is one of the strings
      int i;
      for (i=display->min_detail_level(); i<=display->max_detail_level();
	   i++) {
	if (!strupncmp(argv[2], display->detail_name(i), CMDLEN)) {
	  cmdQueue -> append(new CmdDisplayAltDetail(i, id));
	  return FALSE;
	}
      }
      // otherwise, check if it is a number
      if (isdigit(argv[2][0])) {
	cmdQueue -> append(new CmdDisplayAltDetail(atoi(argv[2]), id));
	return FALSE;
      }
      return TRUE;
    } else if (!strupncmp(argv[1], "depthsort", CMDLEN)) {
      // turn the depthsort method on or off
      int i = str2bool(argv[2]);
      if (i == -1) {
	// there was a problem
      } else {
	cmdQueue -> append(new CmdDisplayDepthSort(i, id));
	return FALSE;
      }
    } else if(!strupncmp(argv[1],"update",CMDLEN)) {
      if (!strcmp(argv[2], "status")) {
	char s[20];
	TCL_RET("%d", vmdUpdateDisplay);
      }
      cmdQueue->append(new CmdDisplayUpdate(argv[2], id));
    } else
      return TRUE;

  } else if(argc == 4) {
    if(!strupncmp(argv[1],"nearclip",CMDLEN)) {
      if(!strupncmp(argv[2],"set",CMDLEN))
	cmdQueue->append(new CmdDisplayClipNear(atof(argv[3]), id));
      else if(!strupncmp(argv[2],"add",CMDLEN))
	cmdQueue->append(new CmdDisplayClipNearRel(atof(argv[3]), id));
      else
        return TRUE;
    } else if(!strupncmp(argv[1],"farclip",CMDLEN)) {
      if(!strupncmp(argv[2],"set",CMDLEN))
	cmdQueue->append(new CmdDisplayClipFar(atof(argv[3]), id));
      else if(!strupncmp(argv[2],"add",CMDLEN))
	cmdQueue->append(new CmdDisplayClipFarRel(atof(argv[3]), id));
      else
        return TRUE;
    } else
      return TRUE;
  }

  // if here, completed successfully
  return FALSE;
}


// text callback routine for 'light'; return TRUE if an error occurs.
int text_cmd_light(int argc, char **argv, CommandQueue *cmdQueue, 
		   int id, Tcl_Interp *interp) {

  if (argc <= 1) {
      interp -> result = "light <number> [on|off|highlight|unhighlight|"
	"status]\nlight <number> rot <axis> <deg>\nlight num";
      return TRUE;
  }
  if(argc > 1) {
    if (argc == 2 && !strupncmp(argv[1], "num", CMDLEN)) {
      // return the number of lights
      sprintf(interp->result, "%d", lights -> num());
      return FALSE;
    }
    int n;
    if (Tcl_GetInt(interp, argv[1], &n) != TCL_OK) {
      Tcl_AppendResult(interp, " -- light <number> ...", NULL);
      return TRUE;
    }
    if(argc == 3) {
      if(!strupncmp(argv[2],"on",CMDLEN))
	cmdQueue->append(new CmdDisplayLightOn(n, TRUE, id));
      else if(!strupncmp(argv[2],"off",CMDLEN))
	cmdQueue->append(new CmdDisplayLightOn(n, FALSE, id));
      else if(!strupncmp(argv[2],"highlight",CMDLEN))
	cmdQueue->append(new CmdDisplayLightHL(n, TRUE, id));
      else if(!strupncmp(argv[2],"unhighlight",CMDLEN))
	cmdQueue->append(new CmdDisplayLightHL(n, FALSE, id));
      else if(!strupncmp(argv[2],"status",CMDLEN)) {
	// return the pair { is on , is highlight} as
	// eg: {on unhighlight}
	Light *light = lights -> item(n);
	if (!light) {
	  sprintf(interp -> result, "light value %d out of range", n);
	  return TRUE;
	}
	sprintf(interp -> result, "%s %s", 
		light -> light_displayed() ? "on" : "off",
		light -> light_highlighted() ? "highlight" : "unhighlight");
	return FALSE;
      } else 
	return TRUE;

    } else if(argc == 5 && !strupncmp(argv[2],"rot",CMDLEN)) {
      char axis = (char)(tolower(argv[3][0]));
      float deg = atof(argv[4]);
      cmdQueue->append(new CmdDisplayLightRot(n, deg, axis, id));

    } else
      return TRUE;

  } else
    return TRUE;

  // if here, completed successfully
  return FALSE;
}


// text callback routine for 'axes'; return TRUE if an error occurs.
int text_cmd_axes(int argc, char **argv, CommandQueue *cmdQueue, 
		  int id, Tcl_Interp *interp) {
  int i;
  if (axes && argc == 2) {
    if(!strupncmp(argv[1],"location",CMDLEN)) {
      // return the current location
      interp -> result = axes -> loc_description(axes->location());
      return FALSE;
    } else if(!strupncmp(argv[1],"locations",CMDLEN)) {
      // return all the possible locations
      for (int i=0; i<axes->locations(); i++) {
	Tcl_AppendElement(interp, axes->loc_description(i));
      }
      return FALSE;
    }
    // else we are at an error, so return a short list
    Tcl_AppendResult(interp, 
		     "axes [location|locations]\n",
		     "axes location [off|origin|lowerleft|lowerright|"
		                          "upperleft|upperright]",
		     (char *) NULL
		     );
    return TRUE;
		     
  }
  if(axes && argc == 3) {
    if(!strupncmp(argv[1],"location",CMDLEN)) {
      for(i=0; i < axes->locations(); i++)
	if(!strupncmp(argv[2],axes->loc_description(i),CMDLEN))
	  break;
      cmdQueue->append(new CmdDisplayAxes(i, id));
    } else
      return TRUE;
  } else
    return TRUE;
 
  // if here, completed successfully
  return FALSE;
}


// text callback routine for 'stage'; return TRUE if an error occurs.
int text_cmd_stage(int argc, char **argv, CommandQueue *cmdQueue, 
		   int id, Tcl_Interp *interp) {
  if (stage && argc == 2) {
    if (!strupncmp(argv[1], "location", CMDLEN)) {
      Tcl_AppendElement(interp, stage->loc_description(stage->location()));
      return FALSE;
    } else if (!strupncmp(argv[1], "locations", CMDLEN)) {
      int i;
      for (i=0; i < stage->locations(); i++) {
	Tcl_AppendElement(interp, stage->loc_description(i));
      }
      return FALSE;
    } else if (!strupncmp(argv[1], "panels", CMDLEN)) {
      char s[20];
      sprintf(s, "%d", stage->panels());
      Tcl_AppendElement(interp, s);
      return FALSE;
    } else {
      Tcl_AppendResult(interp, "possible commands are: location, locations, "
		       "panels [value]", (char *) NULL);
      msgErr << interp -> result << sendmsg;
      Tcl_ResetResult(interp);
      return TRUE;
    }
    // doesn't get here
  }
  if(stage && argc == 3) {
    int i;
    if(!strupncmp(argv[1],"location",CMDLEN)) {
      for(i=0; i < stage->locations(); i++) {
	if(!strupncmp(argv[2],stage->loc_description(i),CMDLEN)) {
	  cmdQueue->append(new CmdDisplayStage(CmdDisplayStage::LOCATION,
					       i, id));
	  return FALSE;
	}
      }
      // didn't find anything, so print the list out
      Tcl_AppendResult(interp, "Possible locations are ", (char *) NULL);
      char *s[3] = {"stage", "locations", NULL};
      text_cmd_stage(2, s, cmdQueue, id, interp);
      
      return TRUE;
    } else if(!strupncmp(argv[1],"panels",CMDLEN)) {
      cmdQueue->append(new CmdDisplayStage(CmdDisplayStage::PANELS,
				atoi(argv[2]), id));
    } else
      return TRUE;

  } else
    return TRUE;
 
  // if here, completed successfully
  return FALSE;
}


//////////////////// call for or toggle the display of the scene
// this will either set the flag for whether to update the display,
// or else it will force a redraw (regardless of the current flag setting).
// newUpdateValue has the following meanings:
//    >= 0 : new value for flag (==0 -> false,  > 0 -> false)
//    -1   : error
//    -2   : force a redraw
//    -123 : force a redraw and get more UI input
int CmdDisplayUpdate::do_execute(void) {
  int retval;

  if(retval = (display != NULL)) {
    if(newUpdateValue >= 0) {  // display update on | display update off
      // set the update flag
      vmdUpdateDisplay = (newUpdateValue != 0);
    } else if(newUpdateValue < (-1)) {  // display update | display update ui
      // force a redraw
      int prevUpdateFlag = vmdUpdateDisplay;
      vmdUpdateDisplay = TRUE;
      VMDupdate(newUpdateValue == -123 ? TRUE : FALSE);
      vmdUpdateDisplay = prevUpdateFlag;
    } else {
      // error
      msgErr << "Unknown option specified for 'display update' command.";
      msgErr << sendmsg;
      retval = FALSE;
    }
  }

  return retval;
}

void CmdDisplayUpdate::create_text(void) {
  if(newUpdateValue != (-1))  {
    *cmdText << "display update";
    if(newUpdateValue >= 0)
      *cmdText << (newUpdateValue != 0 ? " on" : " off");
    else if (newUpdateValue == -123) {
      *cmdText << " ui";
    }
    *cmdText << ends;
  } else {
    Command::create_text();
  }
}

CmdDisplayUpdate::CmdDisplayUpdate(char *newval, int newUIid)
  : Command(Command::DISP_UPDATE, newUIid) {
    if(newval) {
      if (!strcasecmp(newval, "ui")) {
	newUpdateValue = -123;
      } else {
	newUpdateValue = str2bool(newval);  // -1 (bad) 0 (false) 1 (true)
      }
    }
    else
      newUpdateValue = (-2);
}


//////////////////// reshape the display
int CmdDisplayReshape::do_execute(void) {
  int retval;
  if(retval = (display != NULL))
    display->reshape();
  return retval;
}

CmdDisplayReshape::CmdDisplayReshape(int newUIid)
  : Command(Command::DISP_RESHAPE, newUIid) { }


///////////////// Change the display rendering detail
int CmdDisplayDetail::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->detail_level(my_level);
  }
  return retval;
}

void CmdDisplayDetail::create_text(void) {
  *cmdText << "display detail " << display->detail_level() << ends;
}

CmdDisplayDetail::CmdDisplayDetail(int new_level, int newUIid) 
: Command(Command::DISP_DETAIL, newUIid) {
  my_level = new_level;
}
  

///////////////// Change the display alt detail rendering setting
int CmdDisplayAltDetail::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
   display->alt_detail_level(my_level);
  }
  return retval;
}

void CmdDisplayAltDetail::create_text(void) {
  *cmdText << "display altdetail " << display->alt_detail_level() << ends;
}

CmdDisplayAltDetail::CmdDisplayAltDetail(int new_level, int newUIid) 
: Command(Command::DISP_ALTDETAIL, newUIid) {
  my_level = new_level;
}
  
///////////////// Change the display depthsort method
int CmdDisplayDepthSort::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->depth_sort(my_sort);
  }
  return retval;
}

void CmdDisplayDepthSort::create_text(void) {
  *cmdText << "display depthsort " << (display->depth_sort()?"on":"off") << ends;
}

CmdDisplayDepthSort::CmdDisplayDepthSort(int new_sort, int newUIid) 
: Command(Command::DISP_DEPTHSORT, newUIid) {
  my_sort = new_sort;
}
  

///////////////// reset the current view for the current scene
int CmdResetView::do_execute(void) {
  int retval;
  if(retval = (scene != NULL))
    scene->reset_transformation();
  return retval;
}

void CmdResetView::create_text(void) {
  *cmdText << "display resetview" << ends;
}

CmdResetView::CmdResetView(int newUIid)
  : Command(Command::DISP_RESETVIEW, newUIid) {
}

//////////////////// set stereo mode of display
int CmdDisplayStereo::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    if(retval = (stmode >= 0 && stmode < display->num_stereo_modes()))
      display->set_stereo_mode(stmode);
    else
      msgErr << "Cannot set stereo: unknown stereo mode specified." <<sendmsg;
  }
  return retval;
}

void CmdDisplayStereo::create_text(void) {
  if(display && stmode >= 0 && stmode < display->num_stereo_modes())
    *cmdText << "display stereo " << display->stereo_name(stmode);
  else
    *cmdText << "display stereo " << stmode;
  *cmdText << ends;
}

CmdDisplayStereo::CmdDisplayStereo(int newstmode, int newUIid)
  : Command(Command::DISP_STEREO, newUIid) {
  stmode = newstmode;
}

//////////////////// set perspective projection mode of display
int CmdDisplayProj::do_execute(void) {
  if (display == NULL) return FALSE;
  display->projection(proj);
  return TRUE;
}

void CmdDisplayProj::create_text(void) {
  if(proj == DisplayDevice::PERSPECTIVE)
    *cmdText << "display projection perspective";
  else
    *cmdText << "display projection orthographic";
  *cmdText << ends;
}

CmdDisplayProj::CmdDisplayProj(DisplayDevice::Projection newproj, int newUIid)
  : Command(Command::DISP_PROJ, newUIid) {
  proj = newproj;
}


//////////////////// set eye separation of display
int CmdDisplayEyesep::do_execute(void) {
  int retval;
  if(retval = (display != NULL))
    display->set_eyesep(sep);
  return retval;
}

void CmdDisplayEyesep::create_text(void) {
  *cmdText << "display eyesep " << sep << ends;
}

CmdDisplayEyesep::CmdDisplayEyesep(float newsep, int newUIid)
  : Command(Command::DISP_EYESEP, newUIid) {
  sep = newsep;
}


//////////////////// set focal length of display
int CmdDisplayFocallen::do_execute(void) {
  int retval;
  if(retval = (display != NULL
      && flen != 0.0 && display->eye_dist() != 0.0)) {
    float lx, ly, lz, fl;
    display->where_looking(lx, ly, lz);
    fl = flen / (display->eye_dist());
    lx *= fl;  ly *= fl;  lz *= fl;
    display->change_look(lx, ly, lz);
  }
  return retval;
}

void CmdDisplayFocallen::create_text(void) {
  *cmdText << "display focallength " << flen << ends;
}

CmdDisplayFocallen::CmdDisplayFocallen(float newlen, int newUIid)
  : Command(Command::DISP_FOCALLEN, newUIid) {
  flen = newlen;
}


//////////////////// set screen height value
int CmdDisplayScreenHeight::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->screen_height(val);
  }
  return retval;
}

void CmdDisplayScreenHeight::create_text(void) {
  *cmdText << "display height " << val << ends;
}

CmdDisplayScreenHeight::CmdDisplayScreenHeight(float newval, int newUIid)
  : Command(Command::DISP_SCRHEIGHT, newUIid) {
  val = newval;
}


//////////////////// set distance to screen from origin
int CmdDisplayScreenDistance::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->distance_to_screen(val);
  }
  return retval;
}

void CmdDisplayScreenDistance::create_text(void) {
  *cmdText << "display distance " << val << ends;
}

CmdDisplayScreenDistance::CmdDisplayScreenDistance(float newval, int newUIid)
  : Command(Command::DISP_SCRDIST, newUIid) {
  val = newval;
}


//////////////////// turn on/off antialiasing
int CmdDisplayAAOn::do_execute(void) {
  int retval;
  if(retval = (display != NULL && display->aa_available())) {
    if(onoff)
      display->aa_on();
    else
      display->aa_off();
  }
  return retval;
}

void CmdDisplayAAOn::create_text(void) {
  *cmdText << "display antialias " << (onoff ? "on" : "off") << ends;
}

CmdDisplayAAOn::CmdDisplayAAOn(int turnon, int newUIid)
  : Command(Command::DISP_ANTIALIAS, newUIid) {
  onoff = turnon;
}


//////////////////// turn on/off depth cueing
int CmdDisplayDepthcueOn::do_execute(void) {
  int retval;
  if(retval = (display != NULL && display->cueing_available())) {
    if(onoff)
      display->cueing_on();
    else
      display->cueing_off();
  }
  return retval;
}

void CmdDisplayDepthcueOn::create_text(void) {
  *cmdText << "display depthcue " << (onoff ? "on" : "off") << ends;
}

CmdDisplayDepthcueOn::CmdDisplayDepthcueOn(int turnon, int newUIid)
  : Command(Command::DISP_DEPTHCUE, newUIid) {
  onoff = turnon;
}


/////////////////////  clipping plane controls
// This handles the whole range of clipping plane options
//  There are derived classes so you won't have to have the funky flags
// to change {near,fixed} clipping plane {to a fixed,by a relative} amount
// or not
int CmdDisplayClip::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    if (changenear) {
      if (setval)
        display -> set_near_clip(amount);
      else
        display->addto_near_clip(amount);
    } else {
      if (setval)
        display -> set_far_clip(amount);
      else
        display->addto_far_clip(amount);
    }
  }
  return retval;
}

void CmdDisplayClip::create_text(void) {
  *cmdText << "display " << (changenear ? "near" : "far");
  *cmdText << "clip " << (setval ? "set " : "add ");
  *cmdText << amount << ends;
}

CmdDisplayClip::CmdDisplayClip(int ischangenear, int issetval, 
                                float newamt, int UIid)
  : Command(Command::DISP_CLIP, UIid) {
  changenear = ischangenear;
  setval = issetval;
  amount = newamt;
}  

 
////////////////////  change the axes location
int CmdDisplayAxes::do_execute(void) {
  int retval;
  if(retval=(axes != NULL)) {
    if(retval = (newpos >= 0 && newpos < axes->locations()))
      retval = axes->location(newpos);
    else
      msgErr << "Cannot change axes location: unknown location." << sendmsg;
  }
  return retval;
}

void CmdDisplayAxes::create_text(void) {
  if((axes != NULL) && newpos >= 0 && newpos < axes->locations())
    *cmdText << "axes location " << axes->loc_description(newpos);
  *cmdText << ends;
}

CmdDisplayAxes::CmdDisplayAxes(int npos, int newUIid)
  : Command(Command::CMD_AXES, newUIid) {
  newpos = npos;
}
    

////////////////////  change the stage location
int CmdDisplayStage::do_execute(void) {
  int retval;
  if(settingToChange == LOCATION) {
    if(retval=(stage !=NULL)) {
      if(retval = (newpos >= 0 && newpos < stage->locations()))
        retval = stage->location(newpos);
      else
        msgErr << "Cannot change stage location: unknown location." <<sendmsg;
    }
  } else if(settingToChange == PANELS) {
    if(retval=(stage !=NULL))
      retval = stage->panels(newpos);
  } else
    retval = FALSE;
  return retval;
}

void CmdDisplayStage::create_text(void) {
  if((stage != NULL)) {
    *cmdText << "stage ";
    if(settingToChange == LOCATION && newpos >= 0 && 
        newpos < stage->locations()) {
      *cmdText << "location " << stage->loc_description(newpos);
    } else if(settingToChange == PANELS) {
      *cmdText << "panels " << newpos;
    }
  }
  *cmdText << ends;
}

CmdDisplayStage::CmdDisplayStage(Settings toset, int npos, int newUIid)
  : Command(Command::CMD_STAGE, newUIid) {
  newpos = npos;
  settingToChange = toset;
}


//////////////////// turn on/off the Nth light
int CmdDisplayLightOn::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    if(onoff)
      (lights->current())->on();
    else
      (lights->current())->off();
  }
  return retval;
}

void CmdDisplayLightOn::create_text(void) {
  *cmdText << "light " << n << (onoff ? " on" : " off") << ends;
}

CmdDisplayLightOn::CmdDisplayLightOn(int ln, int turnon, int newUIid)
  : Command(Command::DISP_LIGHT_ON, newUIid) {
  n = ln;  onoff = turnon;;
}


//////////////////// highlight the Nth light
int CmdDisplayLightHL::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    (lights->current())->highlight(hl);
  }
  return retval;
}

void CmdDisplayLightHL::create_text(void) {
  *cmdText << "light " << n << (hl ? " highlight" : " unhighlight") << ends;
}

CmdDisplayLightHL::CmdDisplayLightHL(int ln, int highlt, int newUIid)
  : Command(Command::DISP_LIGHT_HL, newUIid) {
  n = ln;  hl = highlt;
}


//////////////////// rotate the position of the Nth light
int CmdDisplayLightRot::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    (lights->current())->add_rotation(theta, axis);
  }
  return retval;
}

void CmdDisplayLightRot::create_text(void) {
  *cmdText << "light " << n << " rot " << axis << " " << theta << ends;
}

CmdDisplayLightRot::CmdDisplayLightRot(int ln, float th, char ax,
  int newUIid): Command(Command::DISP_LIGHT_ROT, newUIid) {
  n = ln;  theta = th;  axis = ax;
}


