/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdMol.h,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.16 $      $Date: 1996/12/12 21:32:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for affecting molecules.
 *
 ***************************************************************************/
#ifndef CMDMOL_H
#define CMDMOL_H

#include "Command.h"
#include "Global.h"
#include "MoleculeFile.h"
#include "MoleculeList.h"
#include "DrawMolItem.h"
#include "utilities.h"


// the following uses the Cmdtypes MOL_NEW, MOL_LIST, MOL_DEL, MOL_ACTIVE,
// MOL_FIX, MOL_ON, MOL_TOP, MOL_SELECT, MOL_REP, MOL_COLOR, MOL_ADDREP,
// MOL_MODREP, MOL_DELREP, MOL_MODREPITEM


/*
 * NOTES:
 *
 *	1) When referring to a molecule in a command by a number, the
 * unique molecule ID is used, NOT the relative index of the molecule into
 * the molecule list.  This is because the relative index changes as molecules
 * are added/removed, but the unique ID stays the same until the molecule is
 * deleted.
 *
 *	2) All commands which have a particular molecule being referred to
 * take a string argument, which can be "all", "top", "active", "displayed",
 * "none", or a number which is a molecule ID as explained in note #1.
 *
 *	3) Almost all Molecule commands are derived from the general base
 * class "CmdMolecule" (except for commands which do not do any action to an
 * existing molecule, such as CmdMolNew)
 * which stores and frees the string used to refer to the molecule.
 *
 */

// text callback routine for 'mol'
typedef struct Tcl_Interp;
extern int text_cmd_mol(int, char **, CommandQueue *, int, Tcl_Interp *);


////////////////////////////////  CmdMolecule: base class  ////////////////
class CmdMolecule : public Command {

protected:
  // string specifying which molecule to use.  Deleted in destructor.
  char *whichMol;

  // array of id's for which molecule to apply action to.  If NULL, none
  // specified.
  int *idList;
  int numMol;

  // calculate which molecules to operate upon from the given molecule name;
  // put the data in idList, and return the pointer to the integer list.
  // If NULL is returned, there was an error or no molecules were specified.
  // If an error occurs, this prints out the error message as well.
  int *find_mol_list(char *);

  // display an error message that the molecule specification is in error.
  // arguments are total name, and part that had problems
  void molname_error(char *, char *);

public:
  // constructor: molecule specification, command code, id number
  CmdMolecule(char *, Cmdtype, int newUIid=(-1));

  // destructor
  virtual ~CmdMolecule(void);
};


///////// load a new molecule, either from a file, or from a remote connection
// NOTE: this is NOT derived from CmdMolecule, because it does not need to
// have the user specify a molecule on which to operate.
class CmdMolNew : public Command {
public:
  // id of new molecule (-1 if problem)
  int newID;

  // method to read structure
  enum MolSource { MOLFILE, MOLURL, MOLREMOTE, MOLSIGMA, MOLGRAPHICS };

private:
  // where is the molecule coming from?
  MolSource molSource;

  // structure file
  char *molStrFile;
  
  // what kind of structure file is this?
  int molStrFileType;

  // initial coordinate file ... if NULL, no coordinates loaded at start.
  char *molCoorFile;
  
  // type of coordinate file
  int molCoorFileType;

  void file_init(char *fname, int sftype, char *cfile, int cftype);

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor A: read struct from file
  CmdMolNew(char *fname, int sftype, char *cfile, int cftype,
		int newUIid=(-1));
  
  // constructor B: read struct from remote.  Don't specify anything.
  CmdMolNew(int newUIid=(-1));
  
  // constructor C: read struct from file, establish link to Sigma
  CmdMolNew(char *fname, int sftype, char *cfile, int cftype,
		MolSource sigma, int newUIid=(-1));
  
  // constructure D: read from a URL, given format type
  CmdMolNew(char *url, int sftype, int newUIid=(-1));

  // destructor
  virtual ~CmdMolNew(void);
};

///////////////  list the current set of loaded molecules
class CmdMolList : public CmdMolecule {

private:
  // print out molecule summary line
  void print_mol_summary(int i) ;
  
  // print out atom rep summary line
  void print_arep_summary(Molecule *mol, int i) ;
  
protected:
  virtual int do_execute(void);
  virtual void create_text(void);

public:
  // constructor: which molecule, user ID
  CmdMolList(char *, int newUIid=(-1)) ;
};


///////////////  delete the Nth molecule in the molecule List
class CmdMolDelete : public CmdMolecule {

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which molecule, user ID
  CmdMolDelete(char *, int newUIid=(-1));

  // peform actual action for molecule i.  return success.
  int do_action(int) ;
};


///////////////  make the Nth molecule 'active' or 'inactive'
class CmdMolActive : public CmdMolecule {

protected:
  int yn;		// new state for characteristic

  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which molecule, new setting, user ID
  CmdMolActive(char *, int, int newUIid=(-1));

  // peform actual action for molecule i.  return success.
  int do_action(int) ;
};


///////////////  make the Nth molecule 'fixed' or 'free'
class CmdMolFix : public CmdMolecule {

protected:
  int yn;		// new state for characteristic

  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which molecule, new setting, user ID
  CmdMolFix(char *, int, int newUIid=(-1));

  // peform actual action for molecule i.  return success.
  int do_action(int) ;
};



///////////////  make the Nth molecule 'on' or 'off'
class CmdMolOn : public CmdMolecule {

protected:
  int yn;		// new state for characteristic

  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which molecule, new setting, user ID
  CmdMolOn(char *, int, int newUIid=(-1));

  // peform actual action for molecule i.  return success.
  int do_action(int) ;
};



///////////////  make the Nth molecule 'top'.  If more than one molecule
// is specified, make the the first one in the list 'top'.
class CmdMolTop : public CmdMolecule {

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which molecule, new setting, user ID
  CmdMolTop(char *, int newUIid=(-1));
};


///////////// set the current atom selection in moleculeList
class CmdMolSelect : public Command {
public:
  // new selection command (if NULL, print out current settings)
  char *sel;
  
protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  CmdMolSelect(char *newsel, int newUIid=(-1));
  
  virtual ~CmdMolSelect(void) ;
};


///////////// set the current atom representation in moleculeList
class CmdMolRep : public Command {
public:
  // new representation command (if NULL, print out current settings)
  char *sel;
  
protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  CmdMolRep(char *newsel, int newUIid=(-1));
  
  virtual ~CmdMolRep(void) ;
};


///////////// set the current atom coloring method in moleculeList
class CmdMolColor : public Command {
public:
  // new color command (if NULL, print out current settings)
  char *sel;
  
protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  CmdMolColor(char *newsel, int newUIid=(-1));
  
  virtual ~CmdMolColor(void) ;
};


///////////// add a new representation to the specified molecules
class CmdMolAddRep : public CmdMolecule {

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which molecule, user ID
  CmdMolAddRep(char *, int newUIid=(-1));
};


///////////// change a representation for the specified molecule
// NOTE: this can ONLY work on one molecule; if more than one is given, an
// error is printed.
class CmdMolChangeRep : public CmdMolecule {

public:
  // which rep to change
  int repn;

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which rep, which mol, UI id
  CmdMolChangeRep(int rpos, char *, int newUIid=(-1));
};


///////////// change 1 representation characteristic for the specified mol
// NOTE: this can ONLY work on one molecule; if more than one is given, an
// error is printed.
class CmdMolChangeRepItem : public CmdMolecule {

public:
  // characteristics that can be modified
  enum RepData { COLOR, REP, SEL };

  // which rep to change
  int repn;

  // type of item to change
  RepData repData;
  
  // new value
  char *str;

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  // constructor: which rep, which mol, rep data, UI id
  CmdMolChangeRepItem(int rpos, char *, RepData rd, char *s,
  	int newUIid=(-1));

  virtual ~CmdMolChangeRepItem(void);
};


///////////// delete a representation for the specified molecule
// NOTE: this can ONLY work on one molecule; if more than one is given, an
// error is printed.
class CmdMolDeleteRep : public CmdMolecule {

public:
  // which rep to delete
  int repn;

protected:
  virtual int do_execute(void) ;
  virtual void create_text(void);

public:
  CmdMolDeleteRep(int rpos, char *, int newUIid=(-1));
};

#endif

