/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdRemote.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.10 $      $Date: 1996/12/12 21:32:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for affecting remote connections.
 *
 ***************************************************************************/

#include <stdlib.h>
#include "CmdRemote.h"
#include "CmdMol.h"
#include "MoleculeRemote.h"
#include "DrawPatch.h"
#include "RemoteList.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'remote'; return TRUE if an error occurs.
int text_cmd_remote(int argc, char **argv, CommandQueue *cmdQueue, 
		    int id, Tcl_Interp *interp) {

  // The only thing that refers to Tcl is the "list" command
  // However, someday everything should be directly connected to
  // Tcl so users can get feedback for events like "initialize"
  // As it is, now scripts have to query/guess through "list" :(
  if (argc < 2) {
    interp -> result = 
      "remote [cancel|list|run]\n"
      "remote initialize <remote machine> [|<user name>]\n"
      "remote list [all|jobs|apps|parameters|connection]\n"
      "remote [new|attach] <integer>\n"
      "remote set <option> <new value>\n"
      "remote [readoptions|writeoptions] <filename>\n"
      ;
    return TRUE;
  }
  if (!strcmp(argv[1], "list")) {
    if (argc == 2 || (argc == 3 && !strcmp(argv[2], "all"))) {
      // list everything -- calls a Tcl function to do this
      if (Tcl_Eval(interp, "vmd_remote_list_all") != TCL_OK) {
	return TRUE;
      }
      return FALSE;
    }
    // check that we actually have a remote connection
    if (!remote) {
      // only thing allowed is "remote list conection"
      if (argc == 3 && !strcmp(argv[2], "connection")) {
	Tcl_AppendElement(interp, "");
	Tcl_AppendElement(interp, "");
	Tcl_AppendElement(interp, "");
	Tcl_AppendElement(interp, "No Connection");
	return FALSE;
      }
      Tcl_AppendResult(interp, "must initialize remote before listing",
		       NULL);
      return TRUE;
    }
    if (argc == 3) {
      if (!strcmp(argv[2], "jobs")) {
	// return the available jobs
	char s[20];
	int pid, uid;
	for (int i=0; i < remote->running_jobs(); i++) {
	  Tcl_AppendResult(interp, i==0?"":" ", "{", NULL);
	  Tcl_AppendElement(interp, remote->running_job_name(i));
	  remote -> running_job_ids(i, pid, uid);
	  sprintf(s, " %d %d}", pid, uid);
	  Tcl_AppendResult(interp, s, NULL);
	}
	return FALSE;

      } else if (!strcmp(argv[2], "apps")) {
	// return the available apps
	for (int i=0; i<remote->avail_apps(); i++) {
	  Tcl_AppendElement(interp, remote->avail_app_name(i));
	}
	return FALSE;

      } else if (!strcmp(argv[2], "parameters")) {
	// return the available parameters and their values
	for (int i=0; i<remote->options(); i++) {
	  Tcl_AppendElement(interp, remote->option_string(i));
	}
	return FALSE;

      } else if (!strcmp(argv[2], "connection")) {
	// return the host name, user name, application, status
	Tcl_AppendElement(interp, remote->host());
	Tcl_AppendElement(interp, remote->user());
	Tcl_AppendElement(interp, remote->app());
	Tcl_AppendElement(interp, remote->status());
	return FALSE;

      } else {
	Tcl_AppendResult(interp, "unknown 'remote list' parameter \"",
			 argv[2], "\"", NULL);
	return TRUE;
      }
    }
    return TRUE;
  }

  if(argc == 2) {
    if(!strupncmp(argv[1], "cancel", CMDLEN))
      cmdQueue->append(new CmdRemoteCancel(id));
    else if(!strupncmp(argv[1], "list", CMDLEN))
      cmdQueue->append(new CmdRemoteList(CmdRemoteList::ALL, id));
    else if(!strupncmp(argv[1], "run", CMDLEN))
      cmdQueue->append(new CmdRemoteRun(id));
    else
      return TRUE;
  } else if(argc == 3) {
    if(!strupncmp(argv[1], "list", CMDLEN)) {
      if(!strupncmp(argv[2], "all", CMDLEN))
        cmdQueue->append(new CmdRemoteList(CmdRemoteList::ALL, id));
      else if(!strupncmp(argv[2], "jobs", CMDLEN))
        cmdQueue->append(new CmdRemoteList(CmdRemoteList::JOBS, id));
      else if(!strupncmp(argv[2], "apps", CMDLEN))
        cmdQueue->append(new CmdRemoteList(CmdRemoteList::APPS, id));
      else if(!strupncmp(argv[2], "parameters", CMDLEN))
        cmdQueue->append(new CmdRemoteList(CmdRemoteList::PARAMS, id));
      else
        return TRUE;
    } else if(!strupncmp(argv[1], "initialize", CMDLEN)) {
      cmdQueue->append(new CmdRemoteInit(argv[2], NULL, id));
    } else if(!strupncmp(argv[1], "new", CMDLEN)) {
      cmdQueue->append(new CmdRemoteNew(atoi(argv[2]), id));
    } else if(!strupncmp(argv[1], "attach", CMDLEN)) {
      cmdQueue->append(new CmdRemoteAttach(atoi(argv[2]), id));
    } else if(!strupncmp(argv[1], "readoptions", CMDLEN)) {
      cmdQueue->append(new CmdRemoteReadopt(argv[2], id));
    } else if(!strupncmp(argv[1], "writeoptions", CMDLEN)) {
      cmdQueue->append(new CmdRemoteWriteopt(argv[2], id));
    } else
      return TRUE;
  } else if(argc == 4) {
    if(!strupncmp(argv[1], "initialize", CMDLEN)) {
      cmdQueue->append(new CmdRemoteInit(argv[2], argv[3], id));
    } else if(!strupncmp(argv[1], "set", CMDLEN) ||
	      // backward compatability -- remove sometime
	      !strupncmp(argv[1], "setoption", CMDLEN)) {  ///
	      
      cmdQueue->append(new CmdRemoteSetopt(argv[2], argv[3], id));
    } else
      return TRUE;
  } else
    return TRUE;
    
  // if here, everything worked out ok
  return FALSE;
}

// text callback routine for 'sim'; return TRUE if an error occurs.
int text_cmd_sim(int argc, char **argv, CommandQueue *cmdQueue, 
		 int id, Tcl_Interp *interp) {
  if (argc < 2) {
    interp -> result =
      "sim list\n"
      "sim patch [off|byatom|bynode|byload]\n"
      "sim modify [rate|keep] <value>\n"
      "sim [top|detach|kill] <molid>"
      ;
    return TRUE;
  }
  if (argc == 2) {
    if (!strupncmp(argv[1], "list", CMDLEN)) {
      // return a list of the running simulations
      // The format is: {molid remote_machine status}
      // The status comes from statusStrings in Remote.C
      for (int i=0; i<remoteList->num(); i++) {
	Tcl_AppendResult(interp, i==0 ? "" : " ", "{", NULL);
	MoleculeRemote *rem = remoteList -> remote(i);
	char s[20];
	sprintf(s, "%d", rem -> id());
	Tcl_AppendElement(interp, s);
	Tcl_AppendElement(interp, (rem->rem)->host());
	Tcl_AppendElement(interp, (rem->rem)->status());
	Tcl_AppendResult(interp, "}", NULL);
      }
      return FALSE;
    } else if (!strupncmp(argv[1], "top", CMDLEN)) {
      MoleculeRemote *tmp = remoteList -> top();
      sprintf(interp -> result, "%d", tmp ? tmp -> id() : -1);
      return FALSE;
    } else {
      // bad parameter
      return TRUE;
    }
  }

  if(argc == 4 && !strupncmp(argv[1], "modify", CMDLEN)) {
    cmdQueue->append(new CmdRemoteChangeParam(argv[2], argv[3], id));
  } else if(argc == 3 && !strupncmp(argv[1], "patch", CMDLEN)) {
    cmdQueue->append(new CmdRemotePatchDisplay(argv[2], id));
  } else if(argc == 3 && !strupncmp(argv[1], "top", CMDLEN)) {
    cmdQueue->append(new CmdRemoteTop(atoi(argv[2]), id));
  } else if(argc == 3 && !strupncmp(argv[1], "detach", CMDLEN)) {
    cmdQueue->append(new CmdRemoteDetach(atoi(argv[2]), id));
  } else if(argc == 3 && !strupncmp(argv[1], "kill", CMDLEN)) {
    cmdQueue->append(new CmdRemoteKill(atoi(argv[1]), id));
    //  } else if(argc == 2 && !strupncmp(argv[1], "list", CMDLEN)) {
    //    cmdQueue->append(new CmdRemoteSimList(id));
  } else
    return TRUE;

  // if here, everything worked out ok
  return FALSE;
}


//////////////////////  general commands used by these classes

  // print error message about no remote pointer
  void no_remote_error(void) {
    msgErr << "You must first initialize a new remote connection." << sendmsg;
  }
  
  void no_running_sim_error(void) {
    msgErr << "You must first have an active, running remote simulation.";
    msgErr << sendmsg;
  }

  // print out available application for current remote object.
  void print_avail_apps(void) {
    msgInfo << "Available applications to run on " << remote->host();
    msgInfo << ": " << remote->avail_apps() << sendmsg;
    for(int i=0; i < remote->avail_apps(); i++)
      msgInfo << i << ":  " << remote->avail_app_name(i) << sendmsg;
  }
  
  // print out available jobs on remote computer
  void print_avail_jobs(void) {
    int pid, uid;
    msgInfo << "Jobs currently running on " << remote->host();
    msgInfo << ": " << remote->running_jobs() << sendmsg;
    for(int i=0; i < remote->running_jobs(); i++) {
      remote->running_job_ids(i, pid, uid);
      msgInfo << i << ":  " << remote->running_job_name(i);
      msgInfo << " (pid=" << pid << ", uid=" << uid << ")" << sendmsg;
    }
  }
  
  // print out current parameter data for remote connection setup
  void print_parameters(void) {
    msgInfo << "Current parameters for remote application " << remote->app();
    msgInfo << sendmsg;
    for(int i=0; i < remote->options(); i++)
      msgInfo << i << ": " << remote->option_string(i) << sendmsg;
  }

  // print out all data in a summary
  void print_remote_status(CmdRemoteList::RemoteList whatToList) {
    msgInfo << "Current remote connection setup:" << sendmsg;
    msgInfo << "--------------------------------" << sendmsg;
    msgInfo << "Status: " << remote->status() << sendmsg;
    if(remote->error_status() != RAPP_EOK)
      msgInfo << "Last error: " << remote->error_status_desc() << sendmsg;
    if(whatToList == CmdRemoteList::APPS || whatToList == CmdRemoteList::ALL)
      print_avail_apps();
    if(whatToList == CmdRemoteList::JOBS || whatToList == CmdRemoteList::ALL)
      print_avail_jobs();
    if(whatToList == CmdRemoteList::PARAMS||whatToList == CmdRemoteList::ALL){
      if(remote->options() > 0)
        print_parameters();
      else
        msgInfo << "No parameters are yet available for editing." << sendmsg;
    }
  }
  
  // print out list of currently/previously running simulations
  void print_simulation_summary(void) {
    msgInfo << "Current and previous remote simulations:" << sendmsg;
    msgInfo << "----------------------------------------" << sendmsg;
    if(remoteList->num() > 0) {
      msgInfo << "ID    Connected to             Status" << sendmsg;
      for(int i=0; i < remoteList->num(); i++) {
        MoleculeRemote *rem = remoteList->remote(i);
	msgInfo << rem->id() << "   " << (rem->rem)->host() << "       ";
	msgInfo << (rem->rem)->status() << sendmsg;
      }
    } else {
      msgInfo << "No current or previous connections." << sendmsg;
    }
  }

///////////////////////////  CmdRemoteInit
///////////// initialize remote with a new connection
int CmdRemoteInit::do_execute(void) {
  if(remote == NULL) {
    // no remote, can initialize
    remote = new Remote(username, computer);
    if(remote->error_status() != RAPP_EOK) {
      msgErr << "Cannot initialize remote connection." << sendmsg;
      delete remote;
      remote = NULL;
    } else {
      msgInfo << "Remote connection established." << sendmsg;
      //      print_remote_status(CmdRemoteList::ALL);
      return TRUE;
    }
  } else {
    msgErr << "There is already an existing remote connection in progress.";
    msgErr << sendmsg;
  }
  
  // if here, there was an error
  return FALSE;
}

void CmdRemoteInit::create_text(void) {
  *cmdText << "remote initialize " << computer << " " << username << ends;
}

CmdRemoteInit::CmdRemoteInit(char *cmptr, char *un, int newUIid)
	: Command(Command::REMOTE_INIT, newUIid) {
  computer = stringdup(cmptr);
  if(un)
    username = stringdup(un);
  else
    username = ::username();
}

CmdRemoteInit::~CmdRemoteInit(void) {
  delete [] computer;
  delete [] username;
}


///////////////////////////  CmdRemoteCancel  
//////////// cancel the current connection
int CmdRemoteCancel::do_execute(void) {
  if(remote != NULL) {
    delete remote;
    remote = NULL;
  } else {
    no_remote_error();
    return FALSE;
  }
  return TRUE;
}

void CmdRemoteCancel::create_text(void) {
  *cmdText << "remote cancel" << ends;
}

CmdRemoteCancel::CmdRemoteCancel(int newUIid)
	: Command(Command::REMOTE_CANCEL, newUIid) { }


///////////////////////////  CmdRemoteList  
/////////////// list the current remote objects settings
int CmdRemoteList::do_execute(void) {
  if(remote != NULL) {
    print_remote_status(whatToList);
  } else {
    no_remote_error();
    return FALSE;
  }
  return TRUE;
}

void CmdRemoteList::create_text(void) {
  *cmdText << "remote list ";
  if(whatToList == JOBS)
    *cmdText << "jobs";
  else if(whatToList == APPS)
    *cmdText << "apps";
  else if(whatToList == PARAMS)
    *cmdText << "parameters";
  *cmdText << ends;
}

CmdRemoteList::CmdRemoteList(RemoteList whattodo, int newUIid)
	: Command(Command::REMOTE_LIST, newUIid) {
  whatToList = whattodo;
}


///////////////////////////  CmdRemoteNew  
/////////////// set the remote connection setup to get the parameters
/////////////// to start a new job on the remote computer, using the selected
/////////////// app.  App is an index into the list of possible applications
/////////////// that can be started.
int CmdRemoteNew::do_execute(void) {
  if(remote != NULL) {
    if(remote->get_parameters(whichApp)) {
      // add command to print out listing
      CmdRemoteList listParams(CmdRemoteList::PARAMS);
      return listParams.execute();
    } else {
      // error getting parameters
      msgErr << "Cannot get parameters for the selected application.";
      msgErr << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

void CmdRemoteNew::create_text(void) {
  *cmdText << "remote new " << whichApp << ends;
}

CmdRemoteNew::CmdRemoteNew(int newapp, int newUIid)
	: Command(Command::REMOTE_NEW, newUIid) {
  whichApp = newapp;
}


///////////////////////////  CmdRemoteAttach  
/////////////// have the remote simulation attach to the specified currently-
/////////////// running job ... a new molecule will be created if this is
/////////////// successful
int CmdRemoteAttach::do_execute(void) {
  if(remote != NULL) {
    if(remote->attach_to_job(whichJob)) {
      // attach was successful ... now create a new molecule
      CmdMolNew newMol((-1));		// constructor for creating remote mol
      return newMol.execute();		// return success of execution
    } else {
      // error attaching to job
      msgErr << "Cannot attach to the specified job.  Clearing remote setup.";
      msgErr << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

void CmdRemoteAttach::create_text(void) {
  *cmdText << "remote attach " << whichJob << ends;
}

CmdRemoteAttach::CmdRemoteAttach(int newjob, int newUIid)
	: Command(Command::REMOTE_ATTACH, newUIid) {
  whichJob = newjob;
}


///////////////////////////  CmdRemoteRun  
/////////////// creates a new simulation using previously established
/////////////// parameters.  Only works if a CmdRemoteNew command has been
/////////////// executed previously
int CmdRemoteRun::do_execute(void) {
  if(remote != NULL) {
    if(remote->start_new_simulation()) {
      // started successfully ... now create a new molecule
      CmdMolNew newMol((-1));		// constructor for creating remote mol
      return newMol.execute();		// return success of execution
    } else {
      // error starting new job
      msgErr << "Cannot start a new simulation.  Most recent error:\n";
      msgErr << "  " << remote->error_status_desc() << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

void CmdRemoteRun::create_text(void) {
  *cmdText << "remote run" << ends;
}

CmdRemoteRun::CmdRemoteRun(int newUIid)
	: Command(Command::REMOTE_RUN, newUIid) { }


///////////////////////////  CmdRemoteSetopt  
/////////////// sets the value for the Nth parameter in the current
/////////////// remote simulation being set up.  Only works if parameters
/////////////// are being edited.
int CmdRemoteSetopt::do_execute(void) {
  if(remote != NULL) {
    if(remote->set_option(keyword, paramVal)) {
      msgInfo << "Parameter " << keyword << " changed." << sendmsg;
      return TRUE;
    } else {
      msgErr << "Unable to change parameter value." << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

void CmdRemoteSetopt::create_text(void) {
  *cmdText << "remote set " << keyword << " " << paramVal << ends;
}

CmdRemoteSetopt::CmdRemoteSetopt(char *kw, char *pv, int newUIid)
	: Command(Command::REMOTE_SETOPT, newUIid) {
  keyword = stringdup(kw);
  paramVal = stringdup(pv);
}

CmdRemoteSetopt::~CmdRemoteSetopt(void) {
  delete [] paramVal;
  delete [] keyword;
}

///////////////////////////  CmdRemoteReadopt  
/////////////// read in parameter values from the given file.
/////////////// Only works if a CmdRemoteNew command has been
/////////////// executed previously.
int CmdRemoteReadopt::do_execute(void) {
  FILE *f;

  if(remote != NULL) {
    if(remote->editing_parameters()) {
      if(!(f = fopen(fname, "r"))) {
        msgErr << "Cannot open parameter file for reading." << sendmsg;
      } else {
        if(! remote->read_param_file(f)) {
	  msgErr << "Problem settings parameters from file " << fname;
	  msgErr << sendmsg;
	}
	fclose(f);
	return TRUE;
      }
    } else {
      msgErr << "You must first be editing a set of parameters." << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

void CmdRemoteReadopt::create_text(void) {
  *cmdText << "remote readoptions " << fname << ends;
}

CmdRemoteReadopt::CmdRemoteReadopt(char *fn, int newUIid)
	: Command(Command::REMOTE_READOPT, newUIid) {
  fname = stringdup(fn);
}

CmdRemoteReadopt::~CmdRemoteReadopt(void) {
  delete [] fname;
}


///////////////////////////  CmdRemoteReadopt  
/////////////// write current parameter values to the given file.
/////////////// This will work as long as the current remote object has
/////////////// any parameters to write.
int CmdRemoteWriteopt::do_execute(void) {
  FILE *f;

  if(remote != NULL) {
    if(remote->options() > 0) {
      if(!(f = fopen(fname, "w"))) {
        msgErr << "Cannot open parameter file for writing." << sendmsg;
      } else {
        if(! remote->write_param_file(f)) {
	  msgErr << "Problem writing parameters to file " << fname;
	  msgErr << sendmsg;
	}
	fclose(f);
	return TRUE;
      }
    } else {
      msgErr << "There are no available options to write." << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

void CmdRemoteWriteopt::create_text(void) {
  *cmdText << "remote writeoptions " << fname << ends;
}

CmdRemoteWriteopt::CmdRemoteWriteopt(char *fn, int newUIid)
	: Command(Command::REMOTE_WRITEOPT, newUIid) {
  fname = stringdup(fn);
}

CmdRemoteWriteopt::~CmdRemoteWriteopt(void) {
  delete [] fname;
}


///////////////////////////  CmdRemoteChangeParam  
/////////////// issue command to change parameter for top remote connection.
int CmdRemoteChangeParam::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->top()) != NULL && (rem->rem)->running_simulation()) {
    if((rem->rem)->change_setting(param, newval))
      return TRUE;
    else
      msgErr << "Bad parameter name '" << param << "'" << sendmsg;
  }

  // if here, did not work
  no_running_sim_error();
  return FALSE;
}

void CmdRemoteChangeParam::create_text(void) {
  *cmdText << "sim modify " << param << " " << newval << ends;
}

CmdRemoteChangeParam::CmdRemoteChangeParam(char *pm, char *nv, int newUIid)
	: Command(Command::SIM_SETPARAM, newUIid) {
  param = stringdup(pm);
  newval = stringdup(nv);
}

CmdRemoteChangeParam::~CmdRemoteChangeParam(void) {
  delete [] newval;
  delete [] param;
}


///////////////////////////  CmdRemotePatchDisplay  
/////////////// change how patches are displayed (if they are even available)
int CmdRemotePatchDisplay::do_execute(void) {
  MoleculeRemote *rem;

  if(newval >= 0) {
    if((rem = remoteList->top()) != NULL)
      return (rem->patch_display_method(newval));
    else
      return FALSE;
  } else
    msgErr << "Unknown patch display method specified." << sendmsg;

  // if here, did not work
  return FALSE;
}

void CmdRemotePatchDisplay::create_text(void) {
  if(newval >= 0)
    *cmdText << "sim patch " << patchDisplayMethodName[newval];
  *cmdText << ends;
}

CmdRemotePatchDisplay::CmdRemotePatchDisplay(char *nv, int newUIid)
	: Command(Command::SIM_PATCHDISP, newUIid) {
  int n = 0;
  newval = (-1);
  while(n < DrawPatch::TOTAL_PATCH_DISPLAY) {
    if(!strupncmp(nv, patchDisplayMethodName[n], CMDLEN)) {
      newval = n;
      break;
    }
    n++;
  }
}

CmdRemotePatchDisplay::CmdRemotePatchDisplay(int nv, int newUIid)
	: Command(Command::SIM_PATCHDISP, newUIid) {
  newval = nv;
  *cmdText << "sim patch " << patchDisplayMethodName[newval] << ends;
}


///////////////////////////  CmdRemoteTop  
/////////////// set the specified molecule as the top remote connection
int CmdRemoteTop::do_execute(void) {
  if(remoteList)
    return remoteList->make_top(remoteList->rem_index_from_id(id));
  
  // if here, did not work
  msgErr << "Bad molecule ID " << id << " specified" << sendmsg;
  return FALSE;
}

void CmdRemoteTop::create_text(void) {
  *cmdText << "sim top " << id << ends;
}

CmdRemoteTop::CmdRemoteTop(int newid, int newUIid)
	: Command(Command::SIM_TOP, newUIid) {
  id = newid;
}



///////////////////////////  CmdRemoteDetach  
/////////////// detach from a running simulation, but don't kill it
int CmdRemoteDetach::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->remote(remoteList->rem_index_from_id(id))) != NULL) {
    msgWarn << "Detaching from remote application - note that it is still";
    msgWarn << " running." << sendmsg;
    (rem->rem)->remote_close(TRUE);
    return TRUE;
  }
  
  // if here, did not work
  msgErr << "Bad molecule ID " << id << " specified" << sendmsg;
  return FALSE;
}

void CmdRemoteDetach::create_text(void) {
  *cmdText << "sim detach " << id << ends;
}

CmdRemoteDetach::CmdRemoteDetach(int newid, int newUIid)
	: Command(Command::SIM_DETACH, newUIid) {
  id = newid;
}


///////////////////////////  CmdRemoteKill  
/////////////// kill a running simulation
int CmdRemoteKill::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->remote(remoteList->rem_index_from_id(id))) != NULL) {
    msgWarn << "Killing remote application - note that it will no longer be";
    msgWarn << " running." << sendmsg;
    (rem->rem)->remote_close(FALSE);
    return TRUE;
  }
  
  // if here, did not work
  msgErr << "Bad molecule ID " << id << " specified" << sendmsg;
  return FALSE;
}

void CmdRemoteKill::create_text(void) {
  *cmdText << "sim kill " << id << ends;
}

CmdRemoteKill::CmdRemoteKill(int newid, int newUIid)
	: Command(Command::SIM_KILL, newUIid) {
  id = newid;
}


///////////////////////////  CmdRemoteSimList  
/////////////// list all current and previous simulations
int CmdRemoteSimList::do_execute(void) {
  if(remoteList)
    print_simulation_summary();
  else
    return FALSE;
    
  return TRUE;
}

void CmdRemoteSimList::create_text(void) {
  *cmdText << "sim list" << ends;
}

CmdRemoteSimList::CmdRemoteSimList(int newUIid)
	: Command(Command::SIM_LIST, newUIid) { }


