/***************************************************************************
 *cr
 *cr		(C) Copyright 1995 The Board of Trustees of the
 *cr			    University of Illinois
 *cr			     All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdSigma.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 96/12/12 21:32:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Command objects for affecting Sigma interface
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdSigma.h"
#include "CmdLabel.h"
#include "CommandQueue.h"
#include "Global.h"
#include "MoleculeSigma.h"
#include "utilities.h"

#include <mdio/global.h>

// the following defines commands for the Cmdtypes:
// SIGMA_EVENT

// The simulation we're controlling, if any
MoleculeSigma *CmdSigmaEvent::sigma = NULL;

void CmdSigmaEvent::register_simulation(MoleculeSigma *_sigma) {
    sigma = _sigma;
}

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'sigma'; return TRUE if an error occurs.
// sigma commands are:
//	sigma ev	 - send event 'ev'
//	sigma ev param	 - send event 'ev' with numeric argument 'param'
//	sigma ev 'id' object elem - send event 'ev' with Id arg (object,elem)
//	sigma ev 'pt' x y z - send event 'ev' with Point3 argument (x,y,z)
// TBD	sigma ev 'tug' tug group
//
//	sigma restraint clear [id] - clear existing restraint from MD and display
//	sigma md idle|stop	- don't do MD until mode changed
//	sigma md mouse		- only do MD when mouse moves a restraint
//	sigma md synchronous	- do MD synchronously with display updates
//	sigma md continuous	- do MD continuously
//	sigma update protein	- only update protein atoms
//	sigma update all	- update all atoms
//	sigma snapshot file	- snapshot current simulation state to file
//
// more commands, not yet accessible via menu:
//	sigma force #		- set force constant for restraints (default 10)
//	sigma update # [#]	- set update frequency for protein and solvent (# = timesteps/update)
//	sigma retain #		- set retention frequency for animation (# = received steps/snapshot)
//	sigma connect		- connect/disconnect to SigmaX
//	sigma disconnect
//
// returns the value for texterr -- FALSE means all okay
int text_cmd_sigma(int argc, char **argv, CommandQueue *cmdQueue, 
		   int id, Tcl_Interp *) {

    // textual commands
    // These all have a required first argument. The second argument
    //	will be "\0" if not specified, so a check on argc >= 2 is sufficient
    //	for now.
    if (argc >= 2 && isalpha(argv[1][0])) {
	if (!strncmp(argv[1], "restraint", 3)) {
	    if (!strncmp(argv[2], "clear", 3)) {
		TugInfo tug;

		if (argc > 3) {
		    // Optional argument is the tug ID to clear
		    tug.tug_id = atoi(argv[3]);
		} else {
		    tug.tug_id = -1;	// Clear all restraints
		}
		tug.group_id = -1;	// Unused at present

		// Tell MD to stop using restraints and remove them from
		//  VMD display
		cmdQueue->append(new CmdSigmaEvent(MD_REMOVE_TUG, tug, id));
		return FALSE;
	    } else if (!strncmp(argv[2], "list", 3)) {
		cmdQueue->append(new CmdSigmaEvent(SMD_CMD_LIST_TUGS, id));
		return FALSE;
	    }
	} else if (!strncmp(argv[1], "force", 3)) {
	    // force constant is sent as X vector component, for now
	    float k = atof(argv[2]);
	    cmdQueue->append(new CmdSigmaEvent(SMD_CMD_FORCE_CONSTANT, k, id));
	    return FALSE;
	} else if (!strncmp(argv[1], "md", 2)) {
	    if (!strncmp(argv[2], "idle", 2) || !strncmp(argv[2], "stop", 2)) {
		cmdQueue->append(new CmdSigmaEvent(MD_COMPUTE_MODE, SmdIdle, id));
		return FALSE;
	    } else if (!strncmp(argv[2], "mouse", 3)) {
		cmdQueue->append(new CmdSigmaEvent(MD_COMPUTE_MODE, SmdMouse, id));
		return FALSE;
	    } else if (!strncmp(argv[2], "synchronous", 3)) {
		cmdQueue->append(new CmdSigmaEvent(MD_COMPUTE_MODE, SmdSynchronous, id));
		return FALSE;
	    } else if (!strncmp(argv[2], "continuous", 3)) {
		cmdQueue->append(new CmdSigmaEvent(MD_COMPUTE_MODE, SmdContinuous, id));
		return FALSE;
	    }
	} else if (!strncmp(argv[1], "update", 3)) {
	    if (!strncmp(argv[2], "protein", 3)) {
		cmdQueue->append(new CmdSigmaEvent(MD_UPDATE_PROTEIN, id));
		return FALSE;
	    } else if (!strncmp(argv[2], "all", 3)) {
		cmdQueue->append(new CmdSigmaEvent(MD_UPDATE_ALL, id));
		return FALSE;
	    } else if (isdigit(argv[2][0])) {
		int f = atoi(argv[2]);
		cmdQueue->append(new CmdSigmaEvent(MD_UPDATE_PROTEIN_FREQUENCY, f, id));

		if (argc > 3 && isdigit(argv[3][0])) {
		    f = atoi(argv[3]);
		    cmdQueue->append(new CmdSigmaEvent(MD_UPDATE_SOLVENT_FREQUENCY, f, id));
		}
	    }
	} else if (!strncmp(argv[1], "retain", 3)) {
	    int f = atoi(argv[2]);
msgInfo << "*** Sigma retention frequency = " << f << " (not yet implemented)" << sendmsg;

	} else if (!strncmp(argv[1], "snapshot", 4)) {
	    // should use filename; later
	    cmdQueue->append(new CmdSigmaEvent(MD_WRITE_SNAPSHOT, id));
	    return FALSE;
	} else if (!strncmp(argv[1], "connect", 3)) {
	    cmdQueue->append(new CmdSigmaEvent(SMD_CMD_CONNECT, id));
	    return FALSE;
	} else if (!strncmp(argv[1], "disconnect", 3)) {
	    cmdQueue->append(new CmdSigmaEvent(SMD_CMD_DISCONNECT, id));
	    return FALSE;
	} else {
	    msgErr << "Unrecognized string argument '" << argv[2]
		   << "' to 'sigma' command" << sendmsg;
	    return TRUE;
	}
    }

    int event;
    if (argc < 2 || argc > 6 || sscanf(argv[1], "%d", &event) != 1) {
	msgErr << "Expecting 'sigma event-num [param-num|id # #|pt x y z]'";
	msgErr << sendmsg;
	return TRUE;
    }

    if (argc == 2) {
	cmdQueue->append(new CmdSigmaEvent(event, id));
	return FALSE;
    }

    if (argc == 3) {
	int param = atoi(argv[2]);
	cmdQueue->append(new CmdSigmaEvent(event, param, id));
	return FALSE;
    }

    if (argc == 6 && !strupncmp(argv[2], "pt", CMDLEN)) {
	float x, y, z;

	x = atof(argv[3]);
	y = atof(argv[4]);
	z = atof(argv[5]);
	cmdQueue->append(new CmdSigmaEvent(event, x, y, z, id));
	return FALSE;
    }

    msgErr << "Unrecognized arguments to 'sigma' command" << sendmsg;
    msgErr << "Expecting args: event-num [param-num|id # #|pt x y z|tug id group]";
    msgErr << sendmsg;
    return TRUE;
}


///////////////////// Send an event to Sigma
int CmdSigmaEvent::do_execute(void) {
    int retval = 1; // success

    MSGDEBUG(2, "CmdSigma::do_execute " << sendmsg);

    if (sigma == NULL) {
	MSGDEBUG(2, "event = " << event << sendmsg);
	msgErr << "CmdSigma::do_execute: no simulation to send to!" << sendmsg;
	return 0;
    }

    switch (paramType) {
	case EV_NO_PARAM:
	    MSGDEBUG(2, "event = " << event << sendmsg);
	    sigma->send_event(event);
	    break;
	case EV_INT_PARAM:
	    MSGDEBUG(2, "event = " << event << " param = " << param << sendmsg);
	    sigma->send_event(event, param);
	    break;
	case EV_FLOAT_PARAM:
	    MSGDEBUG(2, "event = " << event << " param = " << fparam << sendmsg);
	    sigma->send_event(event, fparam);
	    break;
	case EV_POINT3_PARAM:
	    MSGDEBUG(2, "event = " << event << " point = " << x << " "
		    << y << " " << z << sendmsg);
	    sigma->send_event(event, x, y, z);
	    break;
	case EV_TUGINFO_PARAM:
	    MSGDEBUG(2, "event = " << event << " tug_id = " << tparam.tug_id
		<< " group_id = " << tparam.group_id << sendmsg);
	    sigma->send_event(event, &tparam);
	    break;
	default:
	    msgErr << "CmdSigma::do_execute: unknown paramType " << paramType << ". No message sent!"
		   << sendmsg;
	    retval = 0;
	    break;
    }
    return retval;
}

void CmdSigmaEvent::create_text(void) {
    *cmdText << "sigma " << event;
    switch (paramType) {
	case EV_INT_PARAM:
	    *cmdText << " " << param;
	    break;
	case EV_FLOAT_PARAM:
	    *cmdText << " " << fparam;
	    break;
	case EV_POINT3_PARAM:
	    *cmdText << " pt " << x << " " << y << " " << z;
	    break;
	case EV_TUGINFO_PARAM:
	    *cmdText << " tug " << tparam.tug_id << " " << tparam.group_id;
	    break;
	case EV_NO_PARAM:
	    break;
    }
    *cmdText << ends;
}


// constructor: event code
CmdSigmaEvent::CmdSigmaEvent(int ev, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
  paramType = EV_NO_PARAM;
  event = ev;
}

// constructor: event code, numeric parameter
CmdSigmaEvent::CmdSigmaEvent(int ev, int p, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
    paramType = EV_INT_PARAM;
    event = ev;
    param = p;
}

// constructor: event code, numeric parameter
CmdSigmaEvent::CmdSigmaEvent(int ev, float p, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
    paramType = EV_FLOAT_PARAM;
    event = ev;
    fparam = p;
}

// constructor: event code, coordinate x,y,z
CmdSigmaEvent::CmdSigmaEvent(int ev, float xc, float yc, float zc, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
    paramType = EV_POINT3_PARAM;
    event = ev;
    x = xc;
    y = yc;
    z = zc;
}

// constructor: event code, tug description
CmdSigmaEvent::CmdSigmaEvent(int ev, TugInfo &t, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
    paramType = EV_TUGINFO_PARAM;
    event = ev;
    tparam = t;
}
