/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdTracker.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1996/12/12 21:32:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Code for the various tracker commands.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include "CmdTracker.h"
#include "TrackerList.h"
#include "Tracker.h"
#include "UIVR.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'tracker'; return TRUE if an error occurs.
int text_cmd_tracker(int argc, char **argv, CommandQueue *cmdQueue, 
		     int id, Tcl_Interp *interp) {
  if (argc <= 1) {
    interp -> result = 
      "tracker list [|avail]\n"
      "tracker [start|pause] <number>\n"
      "tracker location <number> <sensor>\n"
      "tracker [move|offset] <number> <sensor>\n"
      "tracker position <number> <sensor> <x0> <y0> <z0> <x1> <y1> <z1>\n"
      "tracker [orient|rotate] <number> <sensor> {9 values of a 3x3 "
      "rotation matrix}";
    return TRUE;
  }
  // so the first word [0] is "tracker".  Here are the options
  // tracker list         -- list running trackers
  // tracker list avail   -- list available trackers
  // tracker start number [on remotemachine] -- start avail tracker "number"
  //                            possible remotely
  // tracker pause number [on remotemachine] -- pause tracker "number"
  // tracker location number sensor -- print the current location of 
  //                            a sensor on the tracker "number"
  // tracker move number sensor x y z   -- move the sensor to a given position
  // tracker offset number sensor x y z -- move the sensor by a given offset
  // tracker orient number sensor a11 a12 a13 ... a32 a33 -- set the rotation
  //                         matrix for the given sensor
  // tracker rotate number sensor a11 a12 a13 ... a32 a33 -- apply the rotation
  //                         to the current rotation matrix
  // tracker position number sensor x1 y1 z1 x2 y2 z2 -- set the position and
  //                         so the offset is at x1, y1, z2 and the tip is at
  //                         towards x2, y2, z2

  // "tracker list" and "tracker list avail"
  if (!strupncmp(argv[1], "list", CMDLEN)) {  // ############
    ///    if (argc == 2) { // tracker list
    //      cmdQueue->append(new CmdTrackerList(id));
    ///    } else if ( !strupncmp(argv[2], "avail", CMDLEN)) {
    ///      cmdQueue->append(new CmdTrackerListAvail(id));
    ///    } else
    ///      return TRUE;
    if (argc == 2) { // tracker list
      // list the running trackers
      for (int i = 0; i<trackerList -> num(); i++) {
	Tcl_AppendResult(interp, i==0?"":" ", "{", NULL);
	Tcl_AppendElement(interp, trackerList -> name(i));
	Tcl_AppendElement(interp, trackerList -> tracker(i)->paused() ?
			  "paused" : "running");
	Tcl_AppendResult(interp, "}", NULL);
      }
      return FALSE;
    } else if ( !strupncmp(argv[2], "avail", CMDLEN)) {
      // list the available trackers
      for (int i=0; i<trackerList->num_avail(); i++) {
	Tcl_AppendElement(interp, (char *) trackerList -> avail_name(i));
      }
      return FALSE;
    } else {
      interp -> result = "tracker list [|avail]";
      return TRUE;
    }

  } else if (!strupncmp(argv[1], "start", CMDLEN)) {
    // tracker start number [on remotemachine]  #################
    if (argc !=3 && !(argc == 5 && strupncmp(argv[3], "on", CMDLEN)))
      return TRUE;

    // send a start message to the appropriate tracker
    if (argc == 3)
      cmdQueue->append(new CmdTrackerStart(atoi(argv[2]), id));
    else
      cmdQueue->append(new CmdTrackerStart(atoi(argv[2]), argv[4], id));

  } else if (!strupncmp(argv[1], "location", CMDLEN)) {
    // tracker location number sensor  ######################
    if (argc != 4)
      return TRUE;

    cmdQueue->append(new CmdTrackerLoc(atoi(argv[2]), atoi(argv[3]), id));

 } else if (!strupncmp(argv[1], "move", CMDLEN)) {
    // tracker move number sensor x y z
    if (argc != 7) {
       return TRUE;
    }
    cmdQueue->append(new CmdTrackerSetPos(atoi(argv[2]), atoi(argv[3]),
					  atof(argv[4]), atof(argv[5]),
					  atof(argv[6]), id));
 } else if (!strupncmp(argv[1], "offset", CMDLEN)) {
    // tracker offset number sensor x y z
    if (argc != 7) {
       return TRUE;
    }
    cmdQueue->append(new CmdTrackerModPos(atoi(argv[2]), atoi(argv[3]),
					  atof(argv[4]), atof(argv[5]),
					  atof(argv[6]), id));
 } else if (!strupncmp(argv[1], "orient", CMDLEN)) {
    // tracker orient number sensor a11 a12 a13 a21 a22 a23 a31 a32 a33
    if (argc != 13) {
       return TRUE;
    }
    float tmp[9];
    for (int i=0; i<9; i++) {
       tmp[i] = atof(argv[4+i]);
    }
    cmdQueue->append(new CmdTrackerSetOrient(atoi(argv[2]), atoi(argv[3]),
					     tmp, id));
 } else if (!strupncmp(argv[1], "rotate", CMDLEN)) {
    // tracker rotate number sensor a11 a12 a13 a21 a22 a23 a31 a32 a33
    if (argc != 13) {
       return TRUE;
    }
    float tmp[9];
    for (int i=0; i<9; i++) {
       tmp[i] = atof(argv[4+i]);
    }
    cmdQueue->append(new CmdTrackerModOrient(atoi(argv[2]), atoi(argv[3]),
					     tmp, id));
 } else if (!strupncmp(argv[1], "position", CMDLEN)) {
    // tracker position number sensor x1 y1 z1 x2 y2 z2
    if (argc != 10) {
       return TRUE;
    }
    // this is actually implemented as two commands, a set position
    // and a set rotation
    cmdQueue->append(new CmdTrackerSetPos(atoi(argv[2]), atoi(argv[3]),
					  atof(argv[4]), atof(argv[5]),
					  atof(argv[6]), id));
    // the rotation matrix is made of a rotation about z and another
    // about y.  The length is ignored.
    float tmp[3];
    tmp[0] = atof(argv[7]); tmp[1] = atof(argv[8]); tmp[2] = atof(argv[9]);
    tmp[0] -= atof(argv[4]); tmp[1] -= atof(argv[5]); tmp[2] -= atof(argv[6]);
    tmp[2] = -tmp[2];
    normalize(tmp);
    {
       float mtx[9];
       float len = sqrt(tmp[1]*tmp[1] + tmp[0]*tmp[0]);
       if (len > 0.00001) {
	  float cosphi = tmp[0] / len;
	  float sinphi = tmp[1] / len;
	  float costheta = len;
	  float sintheta = tmp[2];
	  mtx[0] = costheta * cosphi;
	  mtx[1] = costheta * sinphi;
	  mtx[2] = -sintheta;
	  mtx[3] = -sinphi;
	  mtx[4] = cosphi;
	  mtx[5] = 0;
	  mtx[6] = sintheta * cosphi;
	  mtx[7] = sintheta * sinphi;
	  mtx[8] = costheta;
	  
//	  msgInfo << "sin1 " << sin1 << " cos1 " << cos1 << sendmsg;
//	  msgInfo << "sin2 " << sin2 << " cos2 " << cos2 << sendmsg;
       } else {  // the degenerate case (along the z axis)
	  mtx[0] = mtx[1] = mtx[3] = mtx[5] = mtx[7] = mtx[8] = 0;
	  int q = (tmp[2] < 0 ? -1 : 1);
	  mtx[2] = mtx[6] = -(tmp[2] * q);
	  mtx[4] = +(1.0 * q);
//	  msgInfo << "near z axis" << sendmsg;
       }
//       msgInfo << mtx[0] << " " << mtx[1] << " " << mtx[2] << sendmsg;
//       msgInfo << mtx[3] << " " << mtx[4] << " " << mtx[5] << sendmsg;
//       msgInfo << mtx[6] << " " << mtx[7] << " " << mtx[8] << sendmsg;
       cmdQueue->append(new CmdTrackerSetOrient(atoi(argv[2]), atoi(argv[3]),
						mtx, id));
    }
  } else if (!strupncmp(argv[1], "pause", CMDLEN)) {
    // tracker pause 'tracker num'  ###########
    if (argc != 3)
      return TRUE;

    cmdQueue->append(new CmdTrackerPause(atoi(argv[2]), id));

  } else if (!strupncmp(argv[1], "unpause", CMDLEN)) {
    // tracker unpause 'tracker num'  ###########
    if (argc != 3)
      return TRUE;

    cmdQueue->append(new CmdTrackerUnpause(atoi(argv[2]), id));

  } else
    return TRUE;
    
  // if here, everything worked out ok
  return FALSE;
}

/////////////////////////////  utility functions  //////////////////////////

// prints an error message and returns FALSE when the tracker is
// outside the allowed range of running ones.  Otherwise, it returns TRUE
int valid_running_tracker_range(int tracker) {
  int maxtracker;
  maxtracker = (trackerList ? trackerList -> num() - 1 : (-1));
  if (tracker < 0 || tracker > maxtracker) {
    if (maxtracker <= 0)
      msgErr << "There are no trackers running." << sendmsg;
    else {
      msgErr << "Illegal tracker number '" << tracker << "'." << sendmsg;
      msgErr << "The running trackers number from 0 to " << maxtracker;
      msgErr << "." << sendmsg;
    }
    return FALSE;
  }
  return TRUE;  // all a-okay
}


// prints an error message and returns FALSE when the tracker is
// outside the allowed range of available ones.  Otherwise, it returns TRUE
int valid_avail_tracker_range(int tracker) {
  int maxtracker;
  maxtracker = (trackerList ? trackerList -> num_avail() - 1 : (-1));
  if (tracker < 0 || tracker > maxtracker) {
    if (maxtracker <= 0)
      msgErr << "There are no trackers available." << sendmsg;
    else {
      msgErr << "Illegal tracker number '" << tracker << "'." << sendmsg;
      msgErr << "The available trackers number from 0 to " << maxtracker;
      msgErr << "." << sendmsg;
    }
    return FALSE;
  }
  return TRUE;  // all a-okay
}


// prints an error message and returns FALSE when the sensor is
// outside the allowed range.  Otherwise, it returns TRUE
int valid_sensor_range(int tracker, int sensor) {
  int maxsensor;

  // first make sure the tracker is OK
  if(valid_running_tracker_range(tracker)) {
    maxsensor = (trackerList -> tracker(tracker)) -> numSensors() - 1;
    if (sensor < 0 || sensor > maxsensor) {
      if (maxsensor <= 0) {
        msgErr << "There are no sensors available for tracker " << tracker;
        msgErr << "." << sendmsg;
      } else {
        msgErr << "Illegal sensor number '" << sensor << "'." << sendmsg;
        msgErr << "The sensors number from 0 to " << maxsensor;
        msgErr << "." << sendmsg;
      }
      return FALSE;
    }
    return TRUE;  // all a-okay
  }
  return FALSE;
}


/////////////////////////// Start a tracker
CmdTrackerStart::CmdTrackerStart(int newavailnum, char *newhostname, int id)
	: Command(Command::TRACKER_START, id) {
	if (newhostname == NULL) newhostname = "";
  hostname = stringdup(newhostname);
  availnum = newavailnum;
}

CmdTrackerStart::CmdTrackerStart(int newavailnum, int id)
	: Command(Command::TRACKER_START, id) {
  hostname = stringdup("");
  availnum = newavailnum;
}

CmdTrackerStart::~CmdTrackerStart(void) {
  delete [] hostname;
}

void CmdTrackerStart::create_text(void) {
  *cmdText << "tracker start " << availnum;
  if(strlen(hostname) > 0)
    *cmdText << " on " << hostname;
  *cmdText << ends;
}

int CmdTrackerStart::do_execute(void) {
  if(valid_avail_tracker_range(availnum))
    return (trackerList -> start_avail(availnum, hostname)) == TRACKER_OK;
  else
    return FALSE;    
}


/////////////////////////// Pause a tracker, given its number on the list
CmdTrackerPause::CmdTrackerPause(int tmptrackernum, int fromUIid)
	: Command(Command::TRACKER_PAUSE, fromUIid) {
  trackernum = tmptrackernum;
}

void CmdTrackerPause::create_text(void) {
  *cmdText << "tracker pause " << trackernum << ends;
}

int CmdTrackerPause::do_execute(void) {
  if(valid_running_tracker_range(trackernum)) {
    trackerList -> tracker(trackernum) -> pause();
    return TRUE;
  } else {
    return FALSE;
  }
}


/////////////////////////// Pause a tracker, given its number on the list
CmdTrackerUnpause::CmdTrackerUnpause(int tmptrackernum, int fromUIid)
	: Command(Command::TRACKER_UNPAUSE, fromUIid) {
  trackernum = tmptrackernum;
}

void CmdTrackerUnpause::create_text(void) {
  *cmdText << "tracker unpause " << trackernum << ends;
}

int CmdTrackerUnpause::do_execute(void) {
  if(valid_running_tracker_range(trackernum)) {
    trackerList -> tracker(trackernum) -> unpause();
    return TRUE;
  } else {
    return FALSE;
  }
}


/////////////////////////// list all the available trackers
CmdTrackerListAvail::CmdTrackerListAvail(int fromUIid)
	: Command(Command::TRACKER_LIST_AVAIL, fromUIid) { }

void CmdTrackerListAvail::create_text(void) {
  *cmdText << "tracker list avail" << ends;
}

int CmdTrackerListAvail::do_execute(void) {
  int i = trackerList -> num_avail();
  if (i==0) {
    msgInfo << "  No trackers are available." << sendmsg;
  } else {
    msgInfo << "Available trackers:\n" << sendmsg;
    for (int j=0; j<i; j++)
      msgInfo << " " << j << "   " << trackerList->avail_name(j) << sendmsg;
  }
  return TRUE;
}


/////////////////////////// list all running trackers
CmdTrackerList::CmdTrackerList(int fromUIid)
	: Command(Command::TRACKER_LIST, fromUIid) { }

void CmdTrackerList::create_text(void) {
  *cmdText << "tracker list" << ends;
}

int CmdTrackerList::do_execute(void) {
  int i = trackerList -> num();
  if (i == 0 ) {
    msgInfo << "  No trackers are running.\n" << sendmsg;
  } else {
    msgInfo << "Running trackers:\n" << sendmsg;
    for (int j=0; j<i; j++) {
      msgInfo << " " << j;
      if (trackerList->tracker(j)->paused())
        msgInfo << " (paused) ";
      else
        msgInfo << "          ";
      msgInfo  << trackerList -> name(j) << '\n' << sendmsg;
    }
  }
  return TRUE;
}


////////////////////// print the current location of a tracker/ sensor pair
CmdTrackerLoc::CmdTrackerLoc(int tracker, int sensor, int fromUIid)
	: Command(Command::TRACKER_LOC, fromUIid) {
  trackernum = tracker;
  sensornum = sensor;
}

void CmdTrackerLoc::create_text(void) {
  *cmdText << "tracker location " << trackernum << " " << sensornum << ends;
}

// return the current location (x,y,z) of a tracker/sensor pair
int CmdTrackerLoc::do_execute(void) {
  float x,y,z;
  if(valid_sensor_range(trackernum, sensornum)) {
    (trackerList -> tracker(trackernum)) -> get_position(sensornum, x, y, z);
    msgInfo << "Tracker: " << trackernum << "  Sensor: " << sensornum;
    msgInfo << "   \nx= " << x << " y= " << y << " z= "<< z << sendmsg;
    return TRUE;
  }
  return FALSE;
}
//////////////////////////////  set the current tracker/ sensor location
CmdTrackerSetPos::CmdTrackerSetPos(int tracker, int sensor, float newx,
				   float newy, float newz, int fromUIid)
: Command(Command::TRACKER_SETPOS, fromUIid)
{
   trackernum = tracker; sensornum = sensor;
   x = newx; y = newy; z=newz;
}
void CmdTrackerSetPos::create_text(void)
{
   *cmdText << "tracker move " << trackernum << " " << sensornum << " "
	    << x << " " << y << " " << z << ends;
}
int CmdTrackerSetPos::do_execute(void)
{
   if (valid_sensor_range(trackernum, sensornum)) {
      (trackerList -> tracker(trackernum)) -> set_position(sensornum, x, y, z);
      return TRUE;
   }
   return FALSE;
}
//////////////////////////////  offset the current tracker/ sensor location
CmdTrackerModPos::CmdTrackerModPos(int tracker, int sensor, float newx,
				   float newy, float newz, int fromUIid)
: Command(Command::TRACKER_MODPOS, fromUIid)
{
   trackernum = tracker; sensornum = sensor;
   x = newx; y = newy; z=newz;
}
void CmdTrackerModPos::create_text(void)
{
   *cmdText << "tracker offset " << trackernum << " " << sensornum << " "
	    << x << " " << y << " " << z << ends;
}
int CmdTrackerModPos::do_execute(void)
{
   if (valid_sensor_range(trackernum, sensornum)) {
      (trackerList -> tracker(trackernum)) -> mod_position(sensornum, x, y, z);
      return TRUE;
   }
   return FALSE;
}
//////////////////////////////  set the current tracker/ sensor orientation
CmdTrackerSetOrient::CmdTrackerSetOrient(int tracker, int sensor,
					 float *neworient, int fromUIid)
: Command(Command::TRACKER_SETORIENT, fromUIid)
{
   trackernum = tracker; sensornum = sensor;
   memcpy(orient, neworient, 9*sizeof(float));
}
void CmdTrackerSetOrient::create_text(void)
{
   *cmdText << "tracker orient " << trackernum << " " << sensornum << " "
	    << orient[0] << " "	    << orient[1] << " "
	    << orient[2] << " "	    << orient[3] << " "
	    << orient[4] << " "	    << orient[5] << " "
	    << orient[6] << " "	    << orient[7] << " "
	    << orient[8] << " "	    << ends;
}
int CmdTrackerSetOrient::do_execute(void)
{
   if (valid_sensor_range(trackernum, sensornum)) {
      (trackerList -> tracker(trackernum)) -> set_orientation(sensornum,
							      orient);
      return TRUE;
   }
   return FALSE;
}
//////////////////////////////  set the current tracker/ sensor orientation
CmdTrackerModOrient::CmdTrackerModOrient(int tracker, int sensor,
					 float *neworient, int fromUIid)
: Command(Command::TRACKER_MODORIENT, fromUIid)
{
   trackernum = tracker; sensornum = sensor;
   memcpy(orient, neworient, 9*sizeof(float));
}
void CmdTrackerModOrient::create_text(void)
{
   *cmdText << "tracker rotate " << trackernum << " " << sensornum << " "
	    << orient[0] << " "	    << orient[1] << " "
	    << orient[2] << " "	    << orient[3] << " "
	    << orient[4] << " "	    << orient[5] << " "
	    << orient[6] << " "	    << orient[7] << " "
	    << orient[8] << " "	    << ends;
}
int CmdTrackerModOrient::do_execute(void)
{
   if (valid_sensor_range(trackernum, sensornum)) {
      (trackerList -> tracker(trackernum)) -> mod_orientation(sensornum,
							      orient);
      return TRUE;
   }
   return FALSE;
}


