/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdTrans.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1996/12/12 21:32:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Command objects for transforming the current scene.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include "CmdTrans.h"
#include "Scene.h"
#include "Mouse.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// The following uses the Cmdtypes ROTATE, TRANSLATE, SCALE, ROCKON, ROCKOFF,
// STOPROT from the Command class

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'rotmat'; return TRUE if error occurs.
int text_cmd_rotmat(int argc, char **argv, CommandQueue *cmdQueue, 
		    int id, Tcl_Interp *) {
   if (argc != 11) {
      return TRUE;
   }
   int by_or_to;
   if (!strcasecmp(argv[1], "by")) {
      by_or_to = CmdRotMat::BY;
   } else if (!strcasecmp(argv[1], "to")) {
      by_or_to = CmdRotMat::TO;
   } else {
      return TRUE;
   }
   
   float tmp[16];
   memset(tmp, 0, sizeof(tmp));
   for (int i=0; i<9; i++) {
      tmp[i+i/3] = atof(argv[i+2]);
   }
   tmp[15] = 1.0;
   Matrix4 rot(tmp);
   cmdQueue->append(new CmdRotMat(tmp, by_or_to, id));
   return FALSE;
}

// text callback routine for 'rotate'; return TRUE if an error occurs.
int text_cmd_rotate(int argc, char **argv, CommandQueue *cmdQueue, 
		    int id, Tcl_Interp *) {
  if(argc == 2) {
    if(!strupncmp(argv[1],"stop",CMDLEN))
      cmdQueue->append(new CmdStopRot(id));
    else
      return TRUE;

  } else if(argc >= 4 && argc <= 5) {
    char axis = (char)(tolower(*(argv[1])));
    int rotby = ( !strupcmp(argv[2],"by") ? CmdRotate::BY : CmdRotate::TO );
    float deg = atof(argv[3]);
    if(argc == 4)
      cmdQueue->append(new CmdRotate(deg, axis, rotby, id));
    else
      cmdQueue->append(new CmdRotate(deg, axis, rotby, atof(argv[4]), id));

  } else
    return TRUE;

  // if here, command was successfully parsed
  return FALSE;
}
      
// text callback routine for 'translate'; return TRUE if an error occurs.
int text_cmd_translate(int argc, char **argv, CommandQueue *cmdQueue,
		       int id, Tcl_Interp *) {
  if(argc == 5) {
    int trby=(!strupcmp(argv[1],"by") ? CmdTranslate::BY : CmdTranslate::TO);
    cmdQueue->append(new CmdTranslate(atof(argv[2]), atof(argv[3]),
					atof(argv[4]), trby, id));
  } else
    return TRUE;

  // if here, command was successfully parsed
  return FALSE;
}
      
// text callback routine for 'scale'; return TRUE if an error occurs.
int text_cmd_scale(int argc, char **argv, CommandQueue *cmdQueue,
		   int id, Tcl_Interp *) {
  if(argc == 3) {
    int scby = (!strupcmp(argv[1],"by") ? CmdScale::BY : CmdScale::TO );
    cmdQueue->append(new CmdScale(atof(argv[2]), scby, id));
  } else
    return TRUE;

  // if here, command was successfully parsed
  return FALSE;
}
      
// text callback routine for 'rock'; return TRUE if an error occurs.
int text_cmd_rock(int argc, char **argv, CommandQueue *cmdQueue,
		  int id, Tcl_Interp *) {
  if(argc == 2) {
    if(!strupncmp(argv[1],"off",CMDLEN))
      cmdQueue->append(new CmdRockOff(id));
    else
      return TRUE;

  } else if(argc >= 4 && argc <= 5) {
    char axis = (char)(tolower(argv[1][0]));
    float deg = atof(argv[3]);
    int steps = (-1);
    if(argc == 5)
      steps = atoi(argv[4]);
    cmdQueue->append(new CmdRockOn(deg, axis, steps, id));

  } else
    return TRUE;

  // if here, command was successfully parsed
  return FALSE;
}

///////////////// apply a generic transformation matrix to the scene
int CmdRotMat::do_execute(void) {
   int retval;
   if (retval = (scene != NULL)) {
      if (byOrTo == CmdRotMat::BY) {
	 scene -> add_rot(rotMat);
      } else {
	 scene -> set_rot(rotMat);
      }
   }
   return retval;
}

void CmdRotMat::create_text(void) {
   *cmdText << "rotmat " << (byOrTo == CmdRotMat::BY ? "by " : "to ")
	    << rotMat << ends;
}

CmdRotMat::CmdRotMat(const Matrix4& m, int by_or_to, int newUIid)
: Command(Command::ROTMAT, newUIid) {
   byOrTo = by_or_to;
   rotMat = m;
}


///////////////// rotate the current scene
int CmdRotate::do_execute(void) {
  int retval;
  if(retval = (scene != NULL)) {
    scene->stop_rocking();
    if(byOrTo == CmdRotate::BY) {
      if(steps < 0)
        scene->add_rot(deg, axis);		// rotate in one big step
      else
        scene->start_rocking(deg, axis, steps, TRUE);
    } else {			// first clear rotation, then set to value
      Matrix4 tempIdent;
      scene->set_rot(tempIdent);
      scene->add_rot(deg, axis);
    }
  }
  return retval;
}

void CmdRotate::create_text(void) {
  *cmdText << "rotate " << axis;
  *cmdText << ( byOrTo == CmdRotate::BY ? " by " : " to ");
  *cmdText << deg;
  if(steps > 0)
    *cmdText << " " << (deg / ((float)steps));
  *cmdText << ends;
}

// first constructor: a single rotation, no smooth transition
CmdRotate::CmdRotate(float a, char ax, int by_or_to, int newUIid)
  : Command(Command::ROTATE, newUIid) {

  steps = (-1);

  // make sure the axis specified is a legal one ...
  if(ax >= 'x' && ax <= 'z') {
    byOrTo = by_or_to;
    axis = ax;
    deg = a;
  } else {
    // if not legal, just do no rotation.
    byOrTo = CmdRotate::BY;
    axis = 'y';
    deg = 0.0;
  }
}

// second constructor: a smooth rotation in given increments ...
// only useful for "by" rotations.  If "to" is given to this constructor,
// a single-step rotation is done.
CmdRotate::CmdRotate(float a, char ax, int by_or_to, float inc, int newUIid)
  : Command(Command::ROTATE, newUIid) {

  // make sure the axis specified is a legal one ...
  if(ax >= 'x' && ax <= 'z' && inc != 0) {
    byOrTo = by_or_to;
    axis =  ax;
  
    // determine by how much to rotate, and number of steps to use.  If we
    // are doing 'to' rotation, just do it in one big step.
    if(byOrTo == CmdRotate::TO) {
      steps = (-1);
      deg = a;
    } else {
      steps = (int)(fabs(a / inc) + 0.5);

      // make sure there is at least one step
      if(steps < 1) {
        steps = (-1);
	deg = a;
      } else {
        deg = (a < 0.0 ? - fabs(inc) : fabs(inc));
      }
    }

  } else {
    // if not legal, just do no rotation.
    byOrTo = CmdRotate::BY;
    axis = 'y';
    deg = 0.0;
    steps = (-1);
  }
}


///////////////// translate the current scene
int CmdTranslate::do_execute(void) {
  int retval;
  if(retval = (scene != NULL)) {
    if(byOrTo == CmdTranslate::BY) {
      scene->add_glob_trans(x,y,z);
    } else {
      scene->set_glob_trans(x,y,z);
    }
  }
  return retval;
}

void CmdTranslate::create_text(void) {
  *cmdText << "translate ";
  *cmdText << (byOrTo == CmdTranslate::BY ? "by " : "to ");
  *cmdText << x << " " << y << " " << z << ends;
}

CmdTranslate::CmdTranslate(float nx, float ny, float nz, int by_or_to, 
                              int newUIid)
  : Command(Command::TRANSLATE, newUIid) {
  x = nx;  y = ny;  z = nz;
  byOrTo = by_or_to;
}


///////////////// scale the current scene
int CmdScale::do_execute(void) {
  int retval;
  if(retval = (scene != NULL)) {
    if(byOrTo == CmdScale::BY) {
      scene->mult_scale(s);
    } else {
      scene->set_scale(s);
    }
  }
  return retval;
}

void CmdScale::create_text(void) {
  *cmdText << "scale ";
  *cmdText << ( byOrTo == CmdScale::BY ? "by " : "to ");
  *cmdText << s;
  *cmdText << ends;
}

CmdScale::CmdScale(float ns, int by_or_to, int newUIid)
  : Command(Command::SCALE, newUIid) {
  s = ns;
  byOrTo = by_or_to;
}


///////////////// rock the current scene
int CmdRockOn::do_execute(void) {
  int retval;
  if(retval = (scene != NULL))
    scene->start_rocking(deg, axis, steps);
  return retval;
}

void CmdRockOn::create_text(void) {
  *cmdText << "rock " << axis << " by " << deg;
  if(steps >= 0)
    *cmdText << " " << steps;
  *cmdText << ends;
}

CmdRockOn::CmdRockOn(float a, char ax, int nsteps, int newUIid)
  : Command(Command::ROCKON, newUIid) {
  deg = a;
  axis = ((ax >= 'x' && ax <= 'z') ? ax : 'y');
  steps = nsteps;
}


///////////////// stop rocking the current scene
int CmdRockOff::do_execute(void) {
  int retval;
  if(retval = (scene != NULL))
    scene->stop_rocking();
  return retval;
}

void CmdRockOff::create_text(void) {
  *cmdText << "rock off" << ends;
}

CmdRockOff::CmdRockOff(int newUIid) : Command(Command::ROCKOFF, newUIid) {
}


///////////////// stop rotating (both rocking, and mouse movement) for
///////////////// the current scene
int CmdStopRot::do_execute(void) {
  int retval;
  if(retval = (scene != NULL)) {
    scene->stop_rocking();
    if(retval = (mouse != NULL))
      mouse->stop_rotation();
  }
  return retval;
}

void CmdStopRot::create_text(void) {
  *cmdText << "rotate stop" << ends;
}

CmdStopRot::CmdStopRot(int newUIid) : Command(Command::STOPROT, newUIid) {
}


