/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdUtil.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.12 $      $Date: 1997/03/22 06:01:58 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for doing various utilities, such as executing a shell,
 * displaying help, or quitting
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdUtil.h"
#include "UIList.h"
#include "UIText.h"
#include "DisplayDevice.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// The following uses the Cmdtypes RESET, DEBUG,
// QUIT, WAIT, ECHO, LOGWRITE, LOGREAD from the Command class

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'quit'; return TRUE if an error occurs.
int text_cmd_quit(int argc, char **argv, CommandQueue *cmdQueue, int id,
		  Tcl_Interp *) {
  if(argc == 1) {
    cmdQueue->append(new CmdQuit(FALSE,id));

  } else if(argc == 2) {
    if(!strupncmp(argv[1], "confirm", CMDLEN))
      cmdQueue->append(new CmdQuit(TRUE, id));
    else if(!strupncmp(argv[1], "now", CMDLEN))
      cmdQueue->append(new CmdQuit(FALSE, id));
    else
      return TRUE;

  } else
    return TRUE;

  // if here, no problem with command
  return FALSE;
}

// text callback routine for 'wait'; return TRUE if an error occurs.
int text_cmd_wait(int argc, char **argv, CommandQueue *cmdQueue, 
		  int id, Tcl_Interp *) {
  if(argc == 2) {
    cmdQueue->append(new CmdWait(atof(argv[1]), id));

  } else
    return TRUE;

  // if here, no problem with command
  return FALSE;
}

// text callback routine for 'echo'; return TRUE if an error occurs.
int text_cmd_echo(int argc, char **argv, CommandQueue *cmdQueue, 
		  int id, Tcl_Interp *) {
  if(argc == 2) {
    cmdQueue->append(new CmdEcho(argv[1], id));

  } else
    return TRUE;

  // if here, no problem with command
  return FALSE;
}

// text callback routine for 'debug'; return TRUE if an error occurs.
int text_cmd_debug(int argc, char **argv, CommandQueue *cmdQueue, 
		   int id, Tcl_Interp *) {
  if(argc == 2) {
    cmdQueue->append(new CmdDebug(CmdDebug::TOGGLE,
		!strupncmp(argv[1],"on",CMDLEN), id));

  } else if(argc == 3) {
    if(!strupncmp(argv[1], "level",CMDLEN))
      cmdQueue->append(new CmdDebug(CmdDebug::LEVEL, atoi(argv[2]), id));
    else
      return TRUE;

  } else
    return TRUE;

  // if here, no problem with command
  return FALSE;
}

// text callback routine for 'log'; return TRUE if an error occurs.
int text_cmd_log(int argc, char **argv, CommandQueue *cmdQueue, 
		 int id, Tcl_Interp *) {
  if(argc == 2) {
    char *logfile;
    logfile = ((!strupncmp(argv[1],"off",CMDLEN)) ? (char *) NULL : argv[1]);
    cmdQueue->append(new CmdLogWrite(logfile, id));

  } else
    return TRUE;

  // if here, no problem with command
  return FALSE;
}

// text callback routine for 'play'; return TRUE if an error occurs.
int text_cmd_play(int argc, char **argv, CommandQueue *cmdQueue, 
		  int id, Tcl_Interp *) {
  if(argc == 2) {
    cmdQueue->append(new CmdLogRead(argv[1], id));

  } else
    return TRUE;

  // if here, no problem with command
  return FALSE;
}

#ifdef VMDFORMS
#include "forms.h"
#else
#include <iostream.h>
#endif

//////////// exit the program, possibly asking for confirmation
int CmdQuit::do_execute(void) {
  int doexit = TRUE;
  if(askConfirm) {
    doexit = FALSE;
    // for now, does nothing
#ifdef VMDFORMS
#if FL_INCLUDE_VERSION > 81
    if(fl_show_question("Really Quit?", 0))
#else
    if(fl_show_question("Really Quit?", "",""))
#endif
      doexit = TRUE;
#else
    {
      char c;
      cout << "Really Quit? (Y/N) :";
      cin >> c;
      if (c == 'Y' || c == 'y') doexit = TRUE;
    }
#endif	
  }
  if(doexit)
    VMDexit("Exiting normally", 0);

  return FALSE;		// only get here if confirm = no
}

void CmdQuit::create_text(void) {
  *cmdText << "quit " << (askConfirm ? "confirm" : "now") << ends;
}

CmdQuit::CmdQuit(int confirm, int newUIid) : Command(Command::QUIT, newUIid) {
  askConfirm = confirm;
}


//////////// set the debug level for the program
int CmdDebug::do_execute(void) {
  int retval = TRUE;
  if(action == TOGGLE) {
    msgDebug.on(newVal);
  } else if(action == LEVEL) {
    msgDebug.output_level(newVal);
  } else {
    retval = FALSE;
  }
  return retval;
}

void CmdDebug::create_text(void) {
  *cmdText << "debug ";
  if(action == TOGGLE) {
    *cmdText << (newVal ? "on" : "off");
  } else if(action == LEVEL) {
    *cmdText << "level " << newVal;
  }
  *cmdText << ends;
}

CmdDebug::CmdDebug(DebugAction actn, int nval, int newUIid)
	: Command(Command::DEBUG, newUIid) {
  action = actn;
  newVal = nval;
}


//////////// tell the text processor to wait an interval between commands
int CmdWait::do_execute(void) {
  uiText->wait(delay);
  return TRUE;
}

void CmdWait::create_text(void) {
  *cmdText << "wait " << delay << ends;
}

CmdWait::CmdWait(float d, int newUIid)
	: Command(Command::WAIT, newUIid) {
  delay = d;
}


//////////// toggle on/off whether to echo items to the console
int CmdEcho::do_execute(void) {
  if(onoff >= 0)
    uiText->echo(onoff);
  else
    msgErr << "You must specify 'on' or 'off' with echo." << sendmsg;
  return (onoff >= 0);
}

void CmdEcho::create_text(void) {
  if(onoff == 1 || onoff == 0)
    *cmdText << "echo " << (onoff == 1 ? "on" : "off");
  *cmdText << ends;
}

CmdEcho::CmdEcho(char *t, int newUIid) : Command(Command::ECHO, newUIid) {
  if(!strupncmp(t, "on", CMDLEN))
    onoff = 1;
  else if(!strupncmp(t, "off", CMDLEN))
    onoff = 0;
  else
    onoff = (-1);
}


//////////// toggle on/off whether to echo items to the console
int CmdLogWrite::do_execute(void) {
  if(logfile)
    commandQueue->log_on(logfile);
  else
    commandQueue->log_off();
  return TRUE;
}

CmdLogWrite::CmdLogWrite(char *t, int newUIid)
	: Command(Command::LOGWRITE, newUIid) {
  logfile = (t ? stringdup(t) : (char *) NULL);
}

CmdLogWrite::~CmdLogWrite(void) {
  if(logfile)
    delete [] logfile;
}


//////////// start playing back a log file/script
int CmdLogRead::do_execute(void) {
  uiText->read_from_file(logfile);
  return TRUE;
}

CmdLogRead::CmdLogRead(char *t, int newUIid)
	: Command(Command::LOGREAD, newUIid) {
  logfile = stringdup(t);
}

CmdLogRead::~CmdLogRead(void) {
  delete [] logfile;
}


