/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ColorInfo.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1997/03/13 17:38:56 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Routines for Tcl to get the color and color category information
 *
 ***************************************************************************/


#include "tcl.h"
#include "TclCommands.h"
#include "ColorList.h"
extern ColorList *colors;

// They are:
// colorinfo categories
//   Display, Axes, ..., Structure, ...
// colorinfo category Display
//   Background
// colorinfo category Axes
//   X, Y, Z, Origin, Labels
// colorinfo num
//   value of REGCLRS
// colorinfo max
//   value of MAXCOLORS
// colorinfo colors
//   blue, red, ..., black
// colorinfo rgb blue
//   0.25 0.25 1.
// colorinfo alpha blue
//   1.
// colorinfo shininess blue
//   40.
// colorinfo ambient blue
//   0. 0. 0.
// colorinfo specular blue
//   1. 1. 1.
// colorinfo scale method
//   0
// colorinfo scale midpoint
//   0.5
// colorinfo scale min
//   0
// colorinfo scale max
//   1


// return a list of the top-level categories
int tcl_colorinfo_categories(Tcl_Interp *interp, int argc, char *[])
{
  if (argc != 0) {
    interp -> result = "colorinfo: categories takes no parameters";
    return TCL_ERROR;
  }
  int num = (colors->color_categories()).num();
  for (int i=0; i<num; i++) {
    Tcl_AppendElement(interp, (colors->color_categories()).name(i));
  }
  return TCL_OK;
}


// return either a list elements in the category or, given an element
// in the category, return the color associated with it
int tcl_colorinfo_category(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1 && argc !=2) {
    interp -> result = "colorinfo: category takes one parameter (for a list) "
      "or two for a mapping";
    return TCL_ERROR;
  }

  // find the match for a top-level category
  int num = (colors->color_categories()).num();
  int cat = -1;
  int i;
  for (i=0; i<num; i++) {
    if (!strcasecmp((colors->color_categories()).name(i), argv[0])) {
      cat = i;
      break;
    }
  }
  if (cat == -1) {
    interp -> result = "colorinfo: category: couldn't find category";
    return TCL_ERROR;
  }

  // Start the next section
  NameListIntPtr col_cat = colors->color_category(cat);
  int num_names = col_cat -> num();

  // One of two possitilities ....
  if (argc == 1) { // ... list the categories
    for (i=0; i<num_names; i++) {
      Tcl_AppendElement(interp, col_cat -> name(i));
    }
    return TCL_OK;
  }
  //  ....  or return a mapping
  int name_id = -1;
  for (i=0; i<num_names; i++) {
    if (!strcasecmp(col_cat->name(i), argv[1])) {
      name_id = i;
      break;
    }
  }
  if (name_id == -1) {
    interp -> result = "colorinfo: category: couldn't find category element";
    return TCL_ERROR;
  }
  // return the color mapping
  int the_color_index = col_cat -> data(i);
  Tcl_AppendElement(interp, colors->colorNames.name(the_color_index));
  return TCL_OK;
}

// the colors with a name (not those prefixed by a 'trans_')
int tcl_colorinfo_num(Tcl_Interp *interp, int argc, char *[])
{
  if (argc != 0) {
    interp -> result = "colorinfo: numcolors takes no parameters" ;
    return TCL_ERROR;
  }
  sprintf(interp -> result, "%d", REGCLRS);
  return TCL_OK;
}

// ALL of the colors
int tcl_colorinfo_max(Tcl_Interp *interp, int argc, char *[])
{
  if (argc != 0) {
    interp -> result = "colorinfo: maxcolor takes no parameters" ;
    return TCL_ERROR;
  }
  sprintf(interp -> result, "%d", MAXCOLORS);
  return TCL_OK;
}

// return a list of the regular colors
int tcl_colorinfo_colors(Tcl_Interp *interp, int argc, char *[])
{
  if (argc != 0) {
    interp -> result = "colorinfo: colors takes no parameters" ;
    return TCL_ERROR;
  }
  for (int i=0; i<REGCLRS; i++) {
    Tcl_AppendElement(interp, colors->colorNames.name(i));
  }
  return TCL_OK;
}

// takes:
//  a color index
//  a color name (like "red")
//  a color name prepended by "trans_" to get the translucent form
// (see TclGraphics and CmdColor for similar versions(!) of this function)
static int lookup_color(Tcl_Interp *interp, char *s)
{
  // is it a valid number?
  int id;
  if (Tcl_GetInt(interp, s, &id) == TCL_OK) {
    // is it valid?
    if (id >= 0 && id < MAXCOLORS) {
      return id;
    }
  }
  id = -1;
  Tcl_ResetResult(interp);
  // now see if it is a name
  id = colors->colorNames.typecode(s);
  if (id >= 0) {
    return id;
  }
  // finally, is this a transparent color name?
  if (!strncmp(s, "trans_", 6)) {
    id = colors->colorNames.typecode(s+6);
    if (id >= 0) {
      id += REGCLRS;
    }
  }
  return id;  
}

//////////////// get the RGB value of the given color
int tcl_colorinfo_rgb(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1) {
    interp -> result = "colorinfo: color takes one color name or index";
    return TCL_ERROR;
  }
  int id = lookup_color(interp, argv[0]);
  if (id < 0) {
    interp -> result = "colorinfo: color: couldn't find color name or index";
    return TCL_ERROR;
  }
  char *s = interp -> result;
  s = tcl_append_double(s, colors->color(id)[DIFFUSE_INDEX+0], 0);
  s = tcl_append_double(s, colors->color(id)[DIFFUSE_INDEX+1], 1);
      tcl_append_double(s, colors->color(id)[DIFFUSE_INDEX+2], 1);
  return TCL_OK;
}

////////////////////// get the alpha value of the given color
int tcl_colorinfo_alpha(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1) {
    interp -> result = "colorinfo: alpha takes one color name or index";
    return TCL_ERROR;
  }
  int id = lookup_color(interp, argv[0]);
  if (id < 0) {
    interp -> result = "colorinfo: alpha: couldn't find color name or index";
    return TCL_ERROR;
  }
  char *s = interp -> result;
  tcl_append_double(s, colors->color(id)[ALPHA_INDEX], 0);
  return TCL_OK;
}

////////////////////////// the shininess
int tcl_colorinfo_shininess(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1) {
    interp -> result = "colorinfo: shininess takes one color name or index";
    return TCL_ERROR;
  }
  int id = lookup_color(interp, argv[0]);
  if (id < 0) {
    interp -> result = "colorinfo: shininess: couldn't find color name or index";
    return TCL_ERROR;
  }
  char *s = interp -> result;
  tcl_append_double(s, colors->color(id)[SHININESS_INDEX], 0);
  return TCL_OK;
}

////////////////////////// the ambient value
int tcl_colorinfo_ambient(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1) {
    interp -> result = "colorinfo: ambient takes one color name or index";
    return TCL_ERROR;
  }
  int id = lookup_color(interp, argv[0]);
  if (id < 0) {
    interp -> result = "colorinfo: ambient: couldn't find color name or index";
    return TCL_ERROR;
  }
  char *s = interp -> result;
  s = tcl_append_double(s, colors->color(id)[AMBIENT_INDEX+0], 0);
  s = tcl_append_double(s, colors->color(id)[AMBIENT_INDEX+1], 1);
      tcl_append_double(s, colors->color(id)[AMBIENT_INDEX+2], 1);
  return TCL_OK;
}

////////////////////////////// and the specular value
int tcl_colorinfo_specular(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1) {
    interp -> result = "colorinfo: specular takes one color name or index";
    return TCL_ERROR;
  }
  int id = lookup_color(interp, argv[0]);
  if (id < 0) {
    interp -> result = "colorinfo: specular: couldn't find color name or index";
    return TCL_ERROR;
  }
  char *s = interp -> result;
  s = tcl_append_double(s, colors->color(id)[SPECULAR_INDEX+0], 0);
  s = tcl_append_double(s, colors->color(id)[SPECULAR_INDEX+1], 1);
      tcl_append_double(s, colors->color(id)[SPECULAR_INDEX+2], 1);
  return TCL_OK;
}
////////////////////////////// info about the color scale
int tcl_colorinfo_scale(Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc != 1) {
    interp -> result = "colorinfo: scale takes "
      "method|methods|midpoint|min|max";
    return TCL_ERROR;
  }
  if (!strcmp(argv[0], "method")) {
    interp -> result = colorScaleMethod[colors->scale_method()];
    return TCL_OK;
  }
  if (!strcmp(argv[0], "methods")) {
    for (int i=0; i<ColorList::SCALE_TOTAL; i++) {
      Tcl_AppendElement(interp, colorScaleMethod[i]);
    }
    return TCL_OK;
  }
  if (!strcmp(argv[0], "midpoint")) {
    tcl_append_double(interp -> result, colors -> scale_midpoint());
    return TCL_OK;
  }
  if (!strcmp(argv[0], "min")) {
    tcl_append_double(interp -> result, colors -> scale_min());
    return TCL_OK;
  }
  if (!strcmp(argv[0], "max")) {
    tcl_append_double(interp -> result, colors -> scale_max());
    return TCL_OK;
  }
  Tcl_AppendResult(interp, "colorinfo: scale called with incorrect ",
		   "parameter '", argv[0], "'", NULL);
  return TCL_ERROR;
}

////////////////////////////////////////////////////////////////////////
int tcl_colorinfo(ClientData, Tcl_Interp *interp, int argc, char *argv[])
{
  if (argc < 2) {
    //    interp -> result = "colorinfo: needs a parameter";
    interp -> result =
      "colorinfo categories\n"
      "colorinfo category <category>\n"
      "colorinfo [num|max|colors]\n"
      "colorinfo [rgb|alpha|shininess|ambient|specular] <name|value>\n"
      "colorinfo scale [method|methods|midpoint|min|max]";
    return TCL_ERROR;
  }
  if (!strcmp(argv[1], "categories")) {
    return tcl_colorinfo_categories(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "category")) {
    return tcl_colorinfo_category(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "num")) {
    return tcl_colorinfo_num(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "max")) {
    return tcl_colorinfo_max(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "colors")) {
    return tcl_colorinfo_colors(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "rgb")) {
    return tcl_colorinfo_rgb(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "alpha")) {
    return tcl_colorinfo_alpha(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "shininess")) {
    return tcl_colorinfo_shininess(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "ambient")) {
    return tcl_colorinfo_ambient(interp, argc-2, argv+2);
  }
  if (!strcmp(argv[1], "specular")) {
    return tcl_colorinfo_specular(interp, argc-2, argv+2);
  }
  // color scale info
  if (!strcmp(argv[1], "scale")) {
    return tcl_colorinfo_scale(interp, argc-2, argv+2);
  }

  Tcl_AppendResult(interp, "colorinfo: couldn't understand first parameter: ",
		   argv[1], NULL);
  return TCL_ERROR;
}

