/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CommandQueue.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1996/12/12 23:10:36 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * This stores all the Commands to be run in a queue.  The idea is that
 * the various events add Commands to the command queue then they
 * are read off the queue and the UIs are notified.
 *
 * Commands may be logged to a file, if desired.
 *
 ***************************************************************************/

#include "CommandQueue.h"
#include "Command.h"
#include "UIList.h"
#include "utilities.h"
#include "config.h"

#ifdef VMDTCL
#include "UIText.h"
extern UIText *uiText;
#include "tcl.h"
#endif

///////////////////////////  constructor
CommandQueue::CommandQueue(UIList *u) : cmdlist(64) {
  uil = u;
  loggingCmds = FALSE;
  logfile = NULL;
}
    

///////////////////////////  destructor
// we must remove all commands, and delete them as well.
// if logging, must close file
CommandQueue::~CommandQueue(void) {
  while(cmdlist.num() > 0)
    delete_current();
  if(logging())
    log_off();
}


////////////////////////////  private routines  ////////////////////////

// execute the given command.  Return success of command
int CommandQueue::do_execute(Command *cmd) {

  MSGDEBUG(3,"CQ: Executing '" << *cmd << "' ..." << sendmsg);
	
  // if logging, write to log file
  if(logging() && cmd->has_text())
    fprintf(logfile,"%s\n", cmd->text());

#ifdef VMDTCL
  if (uiText && cmd->has_text()) {
    Tcl_CmdInfo info;
    //      Tcl_ResetResult(uiText->tclInterp);
    char *s[2];
    // call the function 'vmdlog' with the results (should this be an
    // event, instead?)
    s[0] = "vmdlog";
    s[1] = cmd->text();
    if (Tcl_GetCommandInfo(uiText->tclInterp, "vmdlog", &info)) {
      (*info.proc)(info.clientData, uiText->tclInterp, 2, s);
    }
  }
#endif

  // do the command ...
  int retval = cmd->execute();
  
  // ... and report to UIList the action has been done
  if(uil) {
    Command::Cmdtype cmdtype = cmd -> gettype();
#ifdef VMDTCL
    // inform the Tcl variable for this object that an event has occured
    // in which it might be interested (used to get the Tk menus
    // to update their variables)
    Tcl_SetVar2(uiText -> tclInterp, "vmd_event", 
		commandEventName[cmdtype], 
		commandName[cmdtype],
		TCL_GLOBAL_ONLY);
#endif

    uil->act_on_command_UI(cmdtype, cmd, retval); 
  }

  // report the success of the commmand
  return retval;
}


// delete the current element in the linked list.  Needed since we
// must delete the Command object stored as well as the list element.
void CommandQueue::delete_current(void) {
  if(cmdlist.num() > 0) {
    delete cmdlist[0];	// delete the Command object store
    cmdlist.remove(0);
  }
}


////////////////////////////  public routines  ////////////////////////

// turn on logging ... open file, and write header comments
void CommandQueue::log_on(char *fname) {
  if(!logging()) {
    if((logfile = fopen(fname, "w")) != NULL) {
      msgInfo << "Logging commands to '" << fname << "'." << sendmsg;
      loggingCmds = TRUE;
      fprintf(logfile,"# %s\n", VERSION_MSG);
      fprintf(logfile,"# Log file '%s', created by user %s\n", fname,
		  username());
    } else {
      msgErr << "Cannot open log file '" << fname << "' for writing."
	     << sendmsg;
    }
  } else {
    msgErr << "Cannot open log file '" << fname << "' ... already "
	   << "logging commands to a file." << sendmsg;
  }
}


// turn off logging ... close the file
void CommandQueue::log_off(void) {
  if(logging()) {
    fprintf(logfile,"# %s\n", VERSION_MSG);
    fprintf(logfile,"# end of log file.\n");
    fclose(logfile);
    loggingCmds = FALSE;
    msgInfo << "Log file closed." << sendmsg;
  } else {
    msgErr << "Not currently logging commands." << sendmsg;
  }
}

// print a message to the log file
void CommandQueue::print_log(char *s) {
  if (logging()) {
    fputs(s, logfile);
    fputc('\n', logfile);
  }
}


// add a new command to the list ... if we have a UIList already, execute it,
// otherwise queue it.  Return success if run, or (-1) if queued.
int CommandQueue::runcommand(Command *cmd) {
  register int retval = (-1);

  if(!cmd)
    return FALSE;

  if(uil) {
    // execute the command, and then delete it
    retval = do_execute(cmd);
    delete cmd;
  } else {
    cmdlist.append(cmd);
  }

  return retval;
}


// add a new command to the list ... always adds to queue, does not
// execute.  Return whether it could be added to queue.
int CommandQueue::append(Command *cmd) {

  if(cmd)
    cmdlist.append(cmd);
  else
    return FALSE;

  // if here, things went OK
  return TRUE;
}


// execute the first command in the queue ... return success of command, or
// FALSE if no commands available
int CommandQueue::execute(UIList *UIL) {
  int retval = FALSE;
  UIList *olduil = uil;

  // execute the command, and then delete it
  if(cmdlist.num() > 0) {
    if(UIL)
      uil = UIL;			// set UIList to use if necessary

    // get command to execute
    Command *execmd = cmdlist[0];

    // remove the command from the list
    cmdlist.remove(0);

    // actually run the command now
    retval = do_execute(execmd);

    // delete the used command
    delete execmd;

    // restore proper UIList
    uil = olduil;
  }
  
  return retval;
}


// execute ALL the commands in the queue
void CommandQueue::execute_all(UIList *UIL) {
  while (cmdlist.num() > 0)
    execute(UIL);
}


