/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CoorCRD.C,v $
 *      $Author: ulrich $        $Locker:  $                $State: Exp $
 *      $Revision: 1.3 $      $Date: 1997/03/24 17:40:02 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 ***************************************************************************/

#include <stdio.h>
#include "CoorCRD.h"
#include "BaseMolecule.h"
#include "Timestep.h"
#include "Inform.h"
#include "Atom.h"

/****************************  ICoorCRD routines  **************************/

// constructor
ICoorCRD::ICoorCRD(BaseMolecule *newmol) {
  // save the molecule pointer
  mol = newmol;

  // save number of atoms expected in file
  nAtoms = mol->nAtoms;
  boxInfo = mol->amberBoxInfo;
  
  // a CRD file can store many sets, but we don't know how many yet.
  
  fileType = ASCII;
  fileNum = (-1);
  frameStart = frameSkip = 0;
  namnf = 0;
  
  X = Y = Z = NULL;
  freeatoms = NULL;
}


// close the input file; return success
ICoorCRD::~ICoorCRD(void) {
  if(opened()) 
    if(file_type() == ASCII)
      fclose(cf);
  if(X) delete [] X;
  if(Y) delete [] Y;
  if(Z) delete [] Z;
}


// initializer; reads header info and inits all variables.  Returns if file was
// initialized.
int ICoorCRD::init(char *fn) {
  
  // if file is already open, we can continue processing it.
  if(opened())
    return TRUE;

  // initially we assume init failed; also, must do base-class initialization
  Initialized = FALSE;
  ICoorFile::init(fn);
  if(strlen(fn) == 0) {
    file = stdin;
  } else if (!(file = fopen (filename(), "r"))) {
       msgErr << "Error opening input CRD file\n";
       msgErr << sendmsg;
       return FALSE;
  }

  // First line is title, so skip past it
  while (getc(file) != 10);
  // read past header here 
  X = new float[nAtoms];
  Y = new float[nAtoms];
  Z = new float[nAtoms];
  return (Initialized = TRUE);
}


// read the next set of coordinates; return list of positions, or NULL if
// error
Timestep* ICoorCRD::read() {
  int i,j,k;
  Timestep *pos;
  float *myx = X, *myy = Y, *myz = Z;

  if(!opened() || (nFrames && currFrames == nFrames))
    return NULL;

  for (i = 0; i < nAtoms; i++) {
     j = fscanf (file,"%f %f %f", &X[i], &Y[i], &Z[i]);
     if (j == EOF) {return NULL;}
     else if (j <= 0) {
	msgErr << "Problem Reading CRD File.";
	msgErr << sendmsg;
	return NULL;
     }
  }
  if ((boxInfo == 1) || (boxInfo == 2)) {
     // right now we just scan past any box info without using it.
     j = fscanf (file, "%f %f %f", &mol->a_length, &mol->b_length, &mol->c_length);
     mol->alpha = 90.0; mol->beta = 90.0; mol->gamma = 90.0;
     //while (getc(file) != 10);
     //while (getc(file) != 10);
     if (j == EOF) { return NULL; }
  }


    
  // create a new position list to store the data
  if(mol->num() > 0) {
    // a frame exists to get extra data from
    pos = new Timestep(nAtoms, timeStep, (mol->item(0))->data);
  } else {
    // this is the first frame; must have Timestep allocate space for data
    pos = new Timestep(nAtoms, timeStep);
  }

  
  // put data in Timestep
  for(i=0, j=0; i < nAtoms; i++) {
    pos->pos[j++] = *(myx++);
    pos->pos[j++] = *(myy++);
    pos->pos[j++] = *(myz++);
  };
  
  // possibly put in extra data
  if(mol->num() == 0) {
    for(i=0, j=0; i < nAtoms; i++) {
      MolAtom *atom = mol->atom(i);
      for(k=0; k < ATOMEXTRA_DYNAMIC; k++)
        pos->data[j++] = atom->extra[k];
    }
  }
  
  currFrames++;
  return pos;
}




/****************************  OCoorCRD routines  **************************/

// constructor
OCoorCRD::OCoorCRD(BaseMolecule *nm, CRDTYPE typ, int nframes) : OCoorFile() {
  // save the molecule pointer
  mol = nm;

  // save number of atoms expected in file
  nAtoms = mol->nAtoms;

  // Should we be writing out a box?
  boxInfo = mol->amberBoxInfo;

  fileType = typ;
  nFrames = (nframes > 0 ? nframes : 0);
     
  // currently, only binary files can be written or read
}

// destructor
// close the output file; return success
OCoorCRD::~OCoorCRD(void) {
  if (opened()) {
    if(file_type() == ASCII)
      fclose(cf);
  }

  if(X) delete[] X;
  if(Y) delete[] Y;
  if(Z) delete[] Z;
}

//// initializer; writes header info and inits all variables.  Returns TRUE
// if file was initialized.
int OCoorCRD::init(char *fn) {
  
  // if file is already open, we can continue processing it.
  if(opened())
    return TRUE;

  Initialized = FALSE;

  // must do base-class initialization
  OCoorFile::init(fn);

  // open the file


  if(strlen(fn) == 0) {
    file = stdout;
  } else  if (!(file = fopen (filename(), "w"))) {
    msgErr << "Error opening output CRD file\n.";
    msgErr << sendmsg;
    return FALSE;
  }

  fprintf (file, "TITLE\n");

  fileType = ASCII;

  X = new float[nAtoms + 1];
  Y = new float[nAtoms + 1];
  Z = new float[nAtoms + 1];

  return (Initialized = TRUE);
}

// write the coordinates in the given coordinate set to the file.  Return
// total frames written so far, or (-1) if error.
int OCoorCRD::write(Timestep *pos) {
  int i, j;
  float *myx = X, *myy = Y, *myz = Z;

  // check for errors
  if (!opened() || !pos)
    return(-1);

  if (pos->num != nAtoms) {
      return (-1);
   }

  // write the coordinate data
    for(i=0, j=0; i < nAtoms; i++) {
    *(myx++) = pos->pos[j++];
    *(myy++) = pos->pos[j++];
    *(myz++) = pos->pos[j++];
  }

  j = 0;
  for (i = 0; i < nAtoms; i++) {
   fprintf (file,"%8.3f", X[i]);
   j++; if (j % 10 == 0) fprintf (file, "\n");
   fprintf (file,"%8.3f", Y[i]);
   j++; if (j % 10 == 0) fprintf (file, "\n");
   fprintf (file,"%8.3f", Z[i]);
   j++; if (j % 10 == 0) fprintf (file, "\n");
  }

  if ((boxInfo == 1) || (boxInfo == 2)) {
   // right now we just right down a zero for the box size
   // VMD doesn't maintain this quantity
   fprintf (file, "\n0.000 0.000 0.000\n");
  }

return ++currFrames;
}

