/***************************************************************************
 *cr
 *cr		(C) Copyright 1995 The Board of Trustees of the
 *cr			    University of Illinois
 *cr			     All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile$
 *	$Author: leech $			$Locker:  $		$State: Exp $
 *	$Revision: 1.1 $			$Date: 96/11/22 02:54:53 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A child Displayable component for a MoleculeSigma. DrawTugs contains
 *  a list of tugs (SIgMA postition well restraints) applied to the
 *  molecule by SIgMA. It manages creating, destroying, modifying,
 *  displaying tugs as well as sending them to SIgMA to maintain a
 *  consistent model by both programs. DrawTugs is loosely based on
 *  Andrew's DrawForce class which serves a similar function for NAMD.
 *
 ***************************************************************************/

#include "DrawTugs.h"
#include "DrawMolecule.h"
#include "ColorList.h"
#include "PickModeQuery.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "Atom.h"
#include "Timestep.h"
#include "Inform.h"
#include "utilities.h"

// Utility function to map motion of a pick from a current object
//  position to a new object position. Parameters:
//
//	d	    DisplayDevice on which the pick happened
//	tm	    Transform matrix from world to object space
//	dim	    Dimension of the pick (2 or 3)
//	pick_pos    Pick position, in mouse or world coordinates
//	object_pos  Old object position
//	new_object_pos New object position (modified)
void map_pick_move(DisplayDevice *d, Matrix4 &tm, int dim, float *pick_pos,
		   float *object_pos, float *new_object_pos) {
    // If a 2D pick, map mouse position to 3D and then into
    //	object space; otherwise, just map into object space.
    float new_world_pos[3];
    if (dim == 2) {
	float object_world_pos[3];

	// Transform object to world coordinates
	tm.multpoint3d(object_pos, object_world_pos);

	SIGMADEBUG(2, "\told world space pos   = " << object_world_pos[0] << " "
		<< object_world_pos[1] << " " << object_world_pos[2] << sendmsg);

	// Map mouse position to same depth as old object position
	d->find_3D_from_2D(object_world_pos, pick_pos, new_world_pos);
	SIGMADEBUG(2, "\tnew world space pos   = " << new_world_pos[0] << " "
		<< new_world_pos[1] << " " << new_world_pos[2] << sendmsg);

	// Transform mapped mouse position back to object space,
	//  giving new object coordinate.
	Matrix4 tminv(tm);

	tminv.inverse();
	tminv.multpoint3d(new_world_pos, new_object_pos);
    } else {
	// This probably needs to be mapped into object space
	for (int i = 0; i < 3; i++)
	    new_object_pos[i] = pick_pos[i];
    }
}

////////////////////////////  constructor

DrawTugs::DrawTugs(DrawMolecule *mr, MoleculeSigma *sig, int _pickmode)
	: Displayable3D(MULT, mr->name, mr, mr->nAtoms/8) {

    SIGMADEBUG(1, "Creating new force representation, for parent molecule:\n");
    SIGMADEBUG(1, " ==> '" << mr->name << "'" << sendmsg);

    // Save molecule and its Sigma component
    mol = mr;
    sigma = sig;

    // initialize variables
    needRegenerate = TRUE;
    colorCat = (-1);
    tugs = new ResizeArray<TugInfo>;

    // register as a pickable item
    register_with_picklist(this->origScene);

    // This "Tugs" pick mode we're interested in. Could just look
    //	for it by name as MoleculeSigma constructor does.
    tugPickMode = _pickmode;

    // No tug being picked now
    cur_tug_id = -1;

    // create new pick modes to allow pointers to add forces
    // atomForcePickMode = add_pick_mode("ForceAtom", 0); [etc]
}

DrawTugs::~DrawTugs() {
    delete tugs;
}

///////////////////////////  protected virtual routines

// do action when a new color list is provided
// This creates a color category for use for drawing forces.
// These colors can then be edited by the user.
void DrawTugs::do_use_colors(void) {
    // add new category (or just get it's index if it exists)
//  colorCat = colorList->add_color_category("Forces");

    // add components, and their default colors
//  (colorList->color_category(colorCat))->add_name("Constant", REGORANGE);
}

// do action due to the fact that a color for the given ColorList for the
// specified category has changed.
void DrawTugs::do_color_changed(ColorList *, int ccat) {
    // right now this does nothing, since we always redraw the list.  But
    // the general thing it would do is set the flag that a redraw is needed,
    // so looking ahead I'll do this now.
    if (ccat = colorCat) {
	needRegenerate = TRUE;
    }
}

// create a new pick mode object for use as a separate pick mode to add
// forces via the mouse.
PickMode *DrawTugs::create_pick_mode(int m) {
    // m is just a code used by this object to tell which one to create.
    if (m >= 0 && m < 3) {
	msgInfo << "Returning PickModeQuery for mode " << m
		<< " in DrawTugs::create_pick_mode" << sendmsg;
	return new PickModeQuery;
    } else {
	msgErr << "Unknown pick mode " << m << " in DrawTugs::create_pick_mode";
	msgErr << sendmsg;
    }

    // if here, error
    return NULL;
}

//////////////////////////////// private routines

// regenerate the command list
void DrawTugs::create_cmdlist(void) {
    // do we need to recreate everything?
    if (!needRegenerate)
	return;

    MSGDEBUG(2, "Regenerating display list for forces '" << name << "' ...");
    MSGDEBUG(2, sendmsg);
//msgInfo << "DrawTugs::create_cmdlist: # tugs = " << tugs->num() << sendmsg;

    // regenerate both data block and display commands
    needRegenerate = FALSE;
    reset_disp_list();

    // only put in commands if there is a current frame
    if (mol->frame() < 0)
	return;

    Timestep *ts = mol->current();

    // display each tug
    for (int i = 0; i < tugs->num(); i++) {
	TugInfo &tug = tugs->item(i);
	MolAtom *atm = mol->atom(tug.atom_id);
	float *tugpos = tug.tug_pos;

	float *atompos = ts->pos + 3 * tug.atom_id;

	// put in commands to set line style, etc. if this is the first
	// item drawn
	if (i == 0) {
	    // cmdMaterials.putdata(FALSE,this);
	    // cmdLineType.putdata(DASHEDLINE, this);
	    // cmdLineWidth.putdata(3, this);
	    cmdMaterials.putdata(TRUE,this);

	    cmdLineType.putdata(SOLIDLINE, this);

	    // int c = REGORANGE;
	    // if (colorCat >= 0)
	    //	  c = (colorList->color_category(colorCat))->data("Constant");
	    // cmdColorIndex.putdata(REGCOLOR(c), this);
	}

	// set line color
	// For want of a better one - could base on tug distance
	int sc = REGORANGE;
	//if (sc >= MAPCLRS)
	//    sc = MAPCLRS - 1;
	cmdColorIndex.putdata(MAPCOLOR(sc), this);

	// Draw the tug as a line. Put a pick point at the
	//  end whose tag is the tug id.
	cmdLine.putdata(atompos, tugpos, this);
	pickPoint.putdata(tugpos, tug.tug_id, this);
    }
}

// prepare for drawing ... do any updates needed right before draw.
void DrawTugs::prepare(DisplayDevice *) {
    // see if the frame has changed ... right now, always update
    needRegenerate = TRUE;
    create_cmdlist();
}

// return if we are interested in the given pick mode or not ... here, we
// are interested in the tug picking mode
int DrawTugs::want_pick_mode(int mode) {
    SIGMADEBUG(1, "DrawTugs::want_pick_mode mode = " << mode << sendmsg);
    return TRUE;
}

// return whether the pickable object is being displayed
int DrawTugs::pickable_on(void) {
    return (tugs->num() > 0 && Displayable3D::pickable_on());
}

// Find the tug with a tug_id corresponding to the given ID.
// Return a pointer to the tug, or NULL if not found.
// Return the index of the tug in *index.
TugInfo *DrawTugs::find_tug_from_id(int id, int *index) {
    for (int i = 0; i < tugs->num(); i++) {
	TugInfo &tug = tugs->item(i);
	if (tug.tug_id == id) {
	    if (index != NULL)
		*index = i;
	    return &tug;
	}
    }
    return NULL;
}

// called when a pick is begun
//    args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void DrawTugs::pick_start(DisplayDevice *, Pickable *p,
			    int btn, int mode, int tag, int dim, float *pos) {
    // There are no child Pickables that might be passed.
    // This will also gets picks on the atoms, which are handled by
    //	MoleculeSigma and should be ignored here.
    if (p != this)
	return;
    if (mode != tugPickMode) {
	SIGMADEBUG(1, "DrawTugs::pick_start: unknown pick mode " << mode << sendmsg);
	return;
    }

    TugInfo *t = find_tug_from_id(tag);
    if (t == NULL) {
	SIGMADEBUG(1, "DrawTugs::pick_start: bad tug ID " << tag << sendmsg);
	return;
    }
    cur_tug_id = tag;

    SIGMADEBUG(1, "DrawTugs::pick_start tag = " << tag << " dim = " << dim
	    << " button = " << btn << " mode = " << mode << sendmsg);
    SIGMADEBUG(1, "\tpos = " << pos[0] << "\t" << pos[1]
	    << "\t" << pos[2] << sendmsg);

    return;
}

// called when a pick moves
void DrawTugs::pick_move(DisplayDevice *d, Pickable *p,
			    int btn, int mode, int tag, int dim, float *pos) {
    if (p != this)
	return;
    if (mode != tugPickMode)
	return;
    TugInfo *t;
    if (cur_tug_id == -1 ||
	(t = find_tug_from_id(cur_tug_id)) == NULL) {
	msgErr << "DrawTugs::pick_move: no current tug to adjust!" << sendmsg;
	return;
    }

    // Could do sanity check that cur_tug_id == tag
    // Could check for shift state or middle button and do nothing

    SIGMADEBUG(1, "DrawTugs: pick_move tag = " << tag << " dim = " << dim
	    << " button = " << btn << " mode = " << mode << sendmsg);
    SIGMADEBUG(1, "\tpos = " << pos[0] << "\t" << pos[1]
	    << "\t" << pos[2] << sendmsg);

    float *tug_pos = t->tug_pos;

    // Do mapping math
    float new_tug_pos[3];
    map_pick_move(d, mol->tm, dim, pos, tug_pos, new_tug_pos);

    SIGMADEBUG(1, "\ttug position moved to " << new_tug_pos[0] << " "
	    << new_tug_pos[1] << " " << new_tug_pos[2] << sendmsg);

    move_tug(cur_tug_id, new_tug_pos);
}

// called when a pick ends
void DrawTugs::pick_end(DisplayDevice *d, Pickable *p,
			    int btn, int mode, int tag, int dim, float *pos) {
    (void)btn; (void)tag; (void)dim; (void)pos;

    if (p != this)
	return;
    if (mode != tugPickMode)
	return;
    if (cur_tug_id == -1) {
	msgErr << "DrawTugs::pick_move: no current tug to adjust!" << sendmsg;
	return;
    }

    // Pick with left button moves the tug
    // Pick with shift-left prints info on the tug
    // Pick with middle button moves the tug (for now)
    // Pick with shift-middle deletes the tug
    if (btn == MouseEvent::B_LEFT) {
	if (d->shift_state() & DisplayDevice::SHIFT)
	    list_tugs(cur_tug_id);
	else
	    finalize_tug(cur_tug_id);
    } else if (btn == MouseEvent::B_MIDDLE) {
	if (d->shift_state() & DisplayDevice::SHIFT)
	    remove_tug(cur_tug_id);
	else
	    finalize_tug(cur_tug_id);
    }

    // Indicate there's no current tug
    cur_tug_id = -1;
}

// Send a tug event to Sigma. This offsets the atom index
//  before sending to account for the 0/1 index basis differences.
void send_tug_event(int event, TugInfo *t) {
    t->atom_id++;
    buffer_send_event(event, t);
    t->atom_id--;
}

// Add a new tug at specified position and force constant; return its ID
// Tell Sigma about the tug
int DrawTugs::add_tug(int atomid, const float *pos, float force) {
    int id, group;
    // temporary hacks; should have separate namespaces for
    //	VMD and Sigma tug IDs and group IDs generated by a static
    //	flag and function in class TugInfo.
    static int next_id = 1000000;
    id = next_id++;
    group = 42;

    TugInfo t(atomid, pos, force, id, group);
    tugs->append(t);
    SIGMADEBUG(1, "DrawTugs::add_tug atom " << atomid << " -> tug ID "
	    << id << sendmsg);

    // Adjust atom # from 0-based VMD indices to 1-based Sculpt protocol
    //	indices.
    send_tug_event(MD_CREATE_TUG, &t);

    return id;
}

// Change position of an existing tug, given its ID
void DrawTugs::move_tug(int tug_id, const float *pos) {
    TugInfo *t = find_tug_from_id(tug_id);
    if (t == NULL) {
	msgErr << "DrawTugs::move_tug: bad tug ID " << tug_id << sendmsg;
	return;
    }

    t->position(pos);
    SIGMADEBUG(1, "DrawTugs::move_tug ID " << tug_id << " pos " <<
	pos[0] << " " << pos[1] << " " << pos[2] << sendmsg);

    send_tug_event(MD_MOVE_TUG, t);  // Offsets atom indices
}

// Finalize movement of a tug, given its ID
// Eventually this could use an alternate representation; right
//  now it's a no-op
void DrawTugs::finalize_tug(int tug_id) {
    TugInfo *t = find_tug_from_id(tug_id);

    if (t == NULL) {
	msgErr << "DrawTugs::move_tug: bad tug ID " << tug_id << sendmsg;
	return;
    }

    SIGMADEBUG(1, "DrawTugs::finalize_tug ID " << tug_id << sendmsg);
}

// Remove tugs (tug id == -1 => all tugs, otherwise specified tug)
void DrawTugs::remove_tug(int tug_id) {
    if (tug_id == -1) {
	tugs->remove(-1, -1);
	TugInfo t;
	t.tug_id = t.group_id = -1;
	send_tug_event(MD_REMOVE_TUG, &t);
    } else {
	int index;
	TugInfo *t = find_tug_from_id(tug_id, &index);

	if (t != NULL) {
	    tugs->remove(index, -1);
	    send_tug_event(MD_REMOVE_TUG, t);
	}
    }

    SIGMADEBUG(1, "DrawTugs::clear_tugs: cleared tug(s) ID " << tug_id
	    << ", " << tugs->num() << " remaining" << sendmsg);
}

static void print_header() {
    msgInfo << "Atom\tForce\tTug ID\tGroup\tPosition" << sendmsg;
    msgInfo << "----\t-----\t------\t-----\t--------------------------------"
	    << sendmsg;
}

static void print_tug(TugInfo &tug) {
    msgInfo << tug.atom_id << "\t" << tug.tug_force << "\t"
	<< tug.tug_id << "\t" << tug.group_id << "\t"
	<< tug.tug_pos[0] << "\t" << tug.tug_pos[1] << "\t"
	<< tug.tug_pos[2] << sendmsg;
}

// List tug(s) known to VMD
// If id == -1, list all tugs, otherwise list specified tug
void DrawTugs::list_tugs(int tug_id) {
    if (tug_id != -1) {
	TugInfo *t = find_tug_from_id(tug_id, NULL);
	if (t == NULL) {
	    msgInfo << "No tug with ID " << tug_id << " exists" << sendmsg;
	} else {
	    print_header();
	    print_tug(*t);
	}
    } else if (tugs->num() == 0) {
	msgInfo << "No tugs are specified" << sendmsg;
    } else {
	print_header();
	for (int i = 0; i < tugs->num(); i++)
	    print_tug(tugs->item(i));
    }
}
