/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: FileRenderer.C,v $
 *	$Author: ulrich $	$Locker:  $		$State: Exp $
 *	$Revision: 1.15 $	$Date: 1997/03/18 20:29:14 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * The FileRenderer class implements the data and functions needed to 
 * render a scene to a file in some format (postscript, raster3d, etc.)
 *
 ***************************************************************************/

#include <stdio.h>
#include "utilities.h"
#include "DispCmds.h"
#include "FileRenderer.h"

// constructor
FileRenderer::FileRenderer(char *public_name, char *default_file_name,
			   char *default_command_line) : 
  transMat(10), DisplayDevice(public_name)
{
  // save the various names
  publicName = stringdup(public_name);
  defaultFilename = stringdup(default_file_name);
  defaultCommandLine = stringdup(default_command_line);

  // init some state variables
  outfile = NULL;
  isOpened = FALSE;
  my_filename = NULL;
}

// close (to be on the safe side) and delete
FileRenderer::~FileRenderer(void)
{
  close_file();
  delete [] my_filename;
  delete [] publicName;
  delete [] defaultFilename;
  delete [] defaultCommandLine;
}

// open file, closing the previous file if it was still open
int FileRenderer::open_file(char *filename)
{
  if (isOpened) {
    close_file();
  }
  if ( (outfile = fopen(filename, "wa")) == NULL) {
    return FALSE;
  }
  my_filename = stringdup(filename);
  isOpened = TRUE;
  reset_state();
  return TRUE;
}

void FileRenderer::reset_state(void)
{
  // empty out the viewing stack
  while (transMat.num()) {
    transMat.pop();
  }
  // reset everything else
  colorIndex = -1;
  dataBlock = NULL;
  lineWidth = 1;
  lineStyle = 1;
  sphereResolution = 4;
  sphereStyle = 1;
  materials_on = TRUE;
}

// close the file.  This could be called by open_file if the previous
// file wasn't closed, so don't put too much here
void FileRenderer::close_file(void)
{
  if (outfile) {
    fclose(outfile);
    outfile = NULL;
  }
  delete [] my_filename;
  my_filename = NULL;
  isOpened = FALSE;
}

///////////////////////////// header and trailer information

void FileRenderer::write_header(void)
{
  if (outfile) {
    fprintf(outfile, "Render not implemented for %s\n", visible_name());
  }
}

void FileRenderer::write_trailer(void)
{
  if (outfile) {
    fprintf(outfile, "# Thank you for using VMD\n");
  }
}


/////////////////////////////////// render the display lists

void FileRenderer::render(void *display_list)
{
  if (!display_list) return;
  char *cmdptr = (char *) display_list;


  // scan through the list and do the action based on the token type
  // if the command relates to the viewing state, keep track of it
  // for those renderers that don't store state
  
  // init some values
  dataBlock = NULL;
  int tok, cmdsize;

  while (transMat.num()) {    // clear the stack
    transMat.pop();
  }
  Matrix4 m;
  transMat.push(m);           // push on the identity matrix

  colorIndex = 0;
  lineWidth = 1;
  lineStyle = 1;
  sphereResolution = 4;
  sphereStyle = 1;

  while ( (tok = ((int *) cmdptr) [0])  != DLASTCOMMAND) {
    if (tok == DLINKLIST) {
       cmdptr += sizeof(int);       // step forward one (no length here)
       cmdptr = *((char **) cmdptr);// and go to the next link
       tok = ((int *)cmdptr)[0];    // this is guarenteed to be neither
                                    // a DLINKLIST nor DLASTCOMMAND
    }

    cmdsize = ((int *) cmdptr)[1];  // how much data is present?
    cmdptr += 2*sizeof(int);        // go to the start of the data

    switch (tok) {   // plot a point
    case DPOINT:
      point( (float *) cmdptr);
      break;
    case DPOINT_I:   // draw a point from the dataBlock
      point( dataBlock + ((int *) cmdptr) [0]);
      break;
    case DSPHERE:    // draw a sphere
      sphere( (float *) cmdptr);
      break;
    case DSPHERE_I: {// plot a sphere using indices into dataBlock
      int n1 = (int)(((float *)cmdptr)[0]);
      float spheredata[4];
      spheredata[0] = dataBlock[n1++];
      spheredata[1] = dataBlock[n1++];
      spheredata[2] = dataBlock[n1];
      spheredata[3] = ((float *)cmdptr)[1];
      sphere(spheredata);
      break;
    }
    case DLINE:    // plot a line
      // don't draw lines of zero length
      if (memcmp(cmdptr, cmdptr+3*sizeof(float), 3*sizeof(float))) {
	line((float *)cmdptr, ((float *)cmdptr) + 3);
      }
      break;
    case DLINE_I:  // three guesses and the first two don't count
      if (memcmp( dataBlock + ((int *)cmdptr)[0], 
		  dataBlock + ((int *)cmdptr)[1], 
		  3*sizeof(float))) {
	line(dataBlock + ((int *)cmdptr)[0], dataBlock + ((int *)cmdptr)[1]);
      }
      break;
    
    case DCYLINDER: // plot a cylinder
      if (memcmp(cmdptr, cmdptr+3*sizeof(float), 3*sizeof(float))) {
	cylinder((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr)[6],
		 int(((float *)cmdptr)[8]) != 0 );
      }
      break;
    case DCYLINDER_I: // more cylinder
      if (memcmp( dataBlock + ((int *)cmdptr)[0], 
		  dataBlock + ((int *)cmdptr)[1], 
		  3*sizeof(float))) {
	cylinder(dataBlock + (int)(((float *)cmdptr)[0]), dataBlock +
		 (int)(((float *)cmdptr)[1]), ((float *)cmdptr)[2] ,
		 int(((float *) cmdptr)[8]) != 0);
      }
      break;

    case DCONE:      // plot a cone  
      if (memcmp(cmdptr, cmdptr+3*sizeof(float), 3*sizeof(float))) {
	cone((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr)[6]);
      }
      break;
    case DCONE_I:    // this is getting boring
      if (memcmp( dataBlock + ((int *)cmdptr)[0], 
		  dataBlock + ((int *)cmdptr)[1], 
		  3*sizeof(float))) {
	cone(dataBlock + (int)(((float *)cmdptr)[0]), dataBlock +
	     (int)(((float *)cmdptr)[1]), ((float *)cmdptr)[2] );
      }
      break;
    
    case DTRIANGLE:    // plot a triangle
      triangle((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr) +  6,
	       ((float *)cmdptr) + 9, ((float *)cmdptr) + 12, 
	       ((float *)cmdptr) + 15);
      break;
    case DTRIANGLE_I:  // you are getting sleeeeeepyyyyy
      triangle( dataBlock + ((int *) cmdptr)[1],
		dataBlock + ((int *) cmdptr)[2],
		dataBlock + ((int *) cmdptr)[3],
		dataBlock + ((int *) cmdptr)[0],
		dataBlock + ((int *) cmdptr)[0],
		dataBlock + ((int *) cmdptr)[0] 
		);
      break;
    case DSQUARE:      // plot a square (norm, 4 verticies
      square((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr) + 6, 
	     ((float *)cmdptr) + 9, ((float *)cmdptr) + 12);
      break;
    case DSQUARE_I:    // square a plot
      square(dataBlock + ((int *)cmdptr)[0], dataBlock + ((int*)cmdptr)[1], 
	     dataBlock + ((int *)cmdptr)[2], dataBlock + ((int *)cmdptr)[3], 
	     dataBlock + ((int *)cmdptr)[4]);
      break;

      ///////////// keep track of state information as well

    case DLINEWIDTH:   //  set the line width
      super_set_line_width( ((int *) cmdptr)[0]);
      break;
    case DLINESTYLE:   // set the line style
      super_set_line_style( ((int *) cmdptr) [0]);
      break;
    case DSPHERERES:   // sphere resolution
      super_set_sphere_res( ((int *) cmdptr) [0]);
      break;
    case DSPHERETYPE:   // sphere resolution
      super_set_sphere_style( ((int *) cmdptr) [0]);
      break;
    case DMATERIALS:
      super_materials( ((int *) cmdptr) [0]);
      break;
    case DCOLORINDEX:  // change the color
      super_set_color(( (int *) cmdptr)[0]);
      break; 
    case DCOLORRGB:  {  // RGB color no longer implemented, but just in case
      float *p = (float *) cmdptr;   // find the 'nearest' color
      int n = colorList -> nearest_index(p[0], p[1], p[2]); 
      if (n != colorIndex) {
	colorIndex = n;
      }
      break;
    }
    case DCOLORDEF:
      // not implemented for file renderers -- get from the current display
      break;

      ////////// Change the viewing matrix stack
    case DPUSH:
      super_push();
      break;
    case DPOP:
      super_pop();
      break;
    case DLOAD:
      super_load( (float *) cmdptr);
      break;
    case DMULT:
      super_multmatrix( (float *) cmdptr);
      break;
    case DTRANS:
      super_translate( (float *) cmdptr);
      break;
    case DROT:
      super_rot( (float *) cmdptr);
      break;
    case DSCALE:
      super_scale( (float *) cmdptr);
      break;

      // text information
    case DTEXTPOS:
      super_text_position(((float *)cmdptr)[0], ((float *)cmdptr)[1],
			  ((float *)cmdptr)[2]);
      break;
    case DTEXT:
      text( (char *) cmdptr);
      break;

    case DCOMMENT:
      comment( (char *)cmdptr);
      break;

      // pick selections (only one implemented)
    case DPICKPOINT:
      pick_point( (float *) cmdptr, ((int *) cmdptr)[3]);
      break;
    case DPICKPOINT_I:
      pick_point( dataBlock + ((int *) cmdptr)[0], ((int *) cmdptr)[1]);
      break;

    case DMOREDATA:
    case DNEWDATA:
#ifdef VMDCAVE
      dataBlock = (float *)cmdptr;             // point to the current position in list
#else
      dataBlock = ((float **)cmdptr)[0];       // point to given memory loc
#endif
      break;
    
    } // end of switch statement

    // update position in array
    cmdptr += cmdsize;
    
  } // while (tok != DLASTCOMMAND)
}

////////////////////////////////////////////////////////////////////

void FileRenderer::super_set_line_width(int new_width)
{
  lineWidth = new_width;
  set_line_width(new_width);
}

void FileRenderer::super_set_line_style(int new_style)
{
  lineStyle = new_style;
  set_line_style(new_style);
}

void FileRenderer::super_set_sphere_res(int new_res)
{
  sphereResolution = new_res;
  set_sphere_res(new_res);
}

void FileRenderer::super_set_sphere_style(int new_style)
{
  sphereStyle = new_style;
  set_sphere_style(new_style);
}

void FileRenderer::super_set_color(int color_index)
{
  if (colorIndex != color_index) {
    colorIndex = color_index;
    set_color(color_index);
  }
}

// turn materials on or off
void FileRenderer::super_materials(int on_or_off)
{
  if (on_or_off) {
    materials_on = 1;
    activate_materials();
  } else {
    materials_on = 0;
    deactivate_materials();
  }
}


//////////////// change the viewing matrix array state ////////////////

void FileRenderer::super_push(void)
{
  transMat.push();
  push();
}
void FileRenderer::super_pop()
{
  transMat.pop();
  pop();
}
void FileRenderer::super_load(float *cmdptr)
{
  Matrix4 tmp(cmdptr);
  (transMat.top()).loadmatrix(tmp);
  load(tmp);
}
void FileRenderer::super_multmatrix(float *cmdptr)
{
  Matrix4 tmp(cmdptr);
  (transMat.top()).multmatrix(tmp);
  multmatrix(tmp);
}
void FileRenderer::super_translate(float *cmdptr)
{
  (transMat.top()).translate( cmdptr[0], cmdptr[1], cmdptr[2]);
  translate( cmdptr[0], cmdptr[1], cmdptr[2]);
}
void FileRenderer::super_rot(float * cmdptr)
{
  (transMat.top()).rot( cmdptr[0], 'x' + (int) (cmdptr[1]) );
  rot( cmdptr[0], 'x' + (int) (cmdptr[1]) );
}
void FileRenderer::super_scale(float *cmdptr)
{
  (transMat.top()).scale( cmdptr[0], cmdptr[1], cmdptr[2] );
  scale( cmdptr[0], cmdptr[1], cmdptr[2] );
}

// scale the radius a by the global scaling factor, return as b.
float FileRenderer::scale_radius(float r) {
  // of course, VMD does not have a direction-independent scaling
  // factor, so I'll fake it using an average of the scaling
  // factors in each direction.
  
  float scaleFactor;
  
  scaleFactor = (sqrtf( 
  (((transMat.top()).mat)[0][0])*(((transMat.top()).mat)[0][0]) +
  (((transMat.top()).mat)[1][0])*(((transMat.top()).mat)[1][0]) +
  (((transMat.top()).mat)[2][0])*(((transMat.top()).mat)[2][0]) ) +
  sqrtf( (((transMat.top()).mat)[0][1])*(((transMat.top()).mat)[0][1]) +
  (((transMat.top()).mat)[1][1])*(((transMat.top()).mat)[1][1]) +
  (((transMat.top()).mat)[2][1])*(((transMat.top()).mat)[2][1]) ) +
  sqrtf( (((transMat.top()).mat)[0][2])*(((transMat.top()).mat)[0][2]) +
  (((transMat.top()).mat)[1][2])*(((transMat.top()).mat)[1][2]) +
  (((transMat.top()).mat)[2][2])*(((transMat.top()).mat)[2][2]) ) ) / 3.0;
  
  if(r < 0.0) {
    msgErr << "FileRenderer: Error, Negative radius" << sendmsg;
    r = -r;
  } 
  
  r = r * scaleFactor;
  
  return r;
}


///////////////////////////// text
void FileRenderer::super_text_position(float x, float y, float z)
{
  textpos[0] = x;
  textpos[1] = y;
  textpos[2] = z;
  text_position(x, y, z);
}


