/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: FileRenderer.h,v $
 *	$Author: ulrich $	$Locker:  $		$State: Exp $
 *	$Revision: 1.14 $	$Date: 1997/03/18 20:29:34 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The FileRenderer class implements the data and functions needed to 
 * render a scene to a file in some format (postscript, raster3d, etc.)
 *
 ***************************************************************************/
#ifndef FILERENDERER_H
#define FILERENDERER_H

// FileRenderer is a class which allows rendering of graphics to
// files, in the form of commands for other programs (raytracers, etc.) 
// or image formats, or whatever.

#include <stdio.h>
#include "Stack.h"
#include "Matrix4.h"
#include "DisplayDevice.h"
#include "FileRenderer.h"

// This is the base class for all the renderers that go to a
// file and are on the render list.  There are five operations
// available to the outside world

class FileRenderer : public DisplayDevice {
protected:
  // default parameters for this instance
  char *publicName, *defaultFilename, *defaultCommandLine;

  // the current file (still not with streams ...)
  FILE *outfile;
  // is the file opened correctly
  int isOpened;
  // the current filename
  char *my_filename;
  
public:
  // create the renderer; set the 'visible' name for the renderer list
  FileRenderer(char *public_name, char *default_file_name,
	       char *default_command_line);
  virtual ~FileRenderer(void);

  const char *visible_name(void) const { return publicName;}
  const char *default_filename(void) const {return defaultFilename;}
  const char *default_exec_string(void) const {return defaultCommandLine;}

  // open the file; don't write the header info
  // return TRUE if opened okay
  // if file already opened, complain, and close previous file
  // this will also reset the state variables
  virtual int open_file(char *filename);
private:
  void reset_state(void);

protected:
  // write the header info.  This is an alias for prepare3D
  // Yes, you have to write a version for you - even if blank
  virtual void write_header(void);
public:
  virtual int prepare3D(int) {
    if (isOpened) {
      write_header();
    }
    return TRUE;
  }
  
public:
  // render the display list
  virtual void render(void *display_list);


protected:
  // write any trailer info.  This is called by update
  // Yes, you have to write a version for you - even if blank
  virtual void write_trailer(void);
  // close the file.  This is called by update, and exists
  // due to symmetry.  Also, is called for case when open is
  // called when a file was already open.
  virtual void close_file(void);

public:
  // don't need to override this (unless you want to do so)
  virtual void update(int) {
    if (isOpened) {
      write_trailer();
      close_file();
      isOpened = FALSE;
    }
  }

  //////////////////// rendering objects
protected:

  // pointer to data block (for indexed object types)
  float *dataBlock;
  
  ///////////// Information about the current state //////
  // (for those that do not want to take care of it themselves)
  // the 'super_' version is called by render to set the matrix.  It
  // then calls the non-super version
  Stack<Matrix4> transMat;
  void super_push(void);
  virtual void push(void) {}
  void super_pop(void);
  virtual void pop(void){}
  void super_load(float *cmdptr);
  virtual void load(Matrix4& /*mat*/) {}
  void super_multmatrix(float *cmdptr);
  virtual void multmatrix(Matrix4& /*mat*/) {}
  void super_translate(float *cmdptr);
  virtual void translate(float /*x*/, float /*y*/, float /*z*/) {}
  void super_rot(float *cmdptr);
  virtual void rot(float /*ang*/, char /*axis*/) {}
  void super_scale(float *cmdptr);
  virtual void scale(float /*scalex*/, float /*scaley*/, 
		     float /*scalez*/) {}
  float scale_radius(float);    // compute the current scaling factor

  // change the color definitions
  int colorIndex;
  void super_set_color(int index);     // ensures set_color is only called
  virtual void set_color(int /*index*/) {} // when the color index changes

  // change the line definitions
  int lineWidth, lineStyle;
  void super_set_line_width(int new_width);
  virtual void set_line_width(int /*new_width*/) {}  // called by super
  void super_set_line_style(int new_style);
  virtual void set_line_style(int /*new_style*/) {}  // called by super

  // change the sphere definitions
  int sphereResolution, sphereStyle;
  void super_set_sphere_res(int res);
  virtual void set_sphere_res(int /*res*/) {}        // called by super
  void super_set_sphere_style(int style);
  virtual void set_sphere_style(int /*style*/) {}       // called by super


  void define_light(int, float *, float *);
  void light_onoff(int light, int state);
//  virtual int do_define_material(int, float *) {}

  int materials_on;
  void super_materials(int on_or_off);
  virtual void activate_materials(void) {}       // if the previous is TRUE
  virtual void deactivate_materials(void) {}     //  if super is FALSE
  

  ////////////////////// various virtual generic graphics commands
  virtual void point(float * /*xyz*/) {}
  virtual void sphere(float * /*xyzr*/) {}

  virtual void line(float * /*xyz1*/, float * /*xyz2*/) {}
  virtual void cylinder(float * /*xyz1*/, float * /*xyz2*/, 
			float /* radius*/, int /*filled*/) {}
  virtual void cone(float * /*xyz1*/, float * /*xyz2*/, 
		    float /* radius*/) {}

  virtual void triangle(float * /*xyz1*/, float * /*xyz2*/, float * /*xyz3*/, 
			float * /*n1*/, float * /*n2*/, float * /*n3*/) {}
  virtual void square(float * /*n*/, float * /*xyz1*/, float * /*xyz2*/, 
		      float * /*xyz3*/, float * /*xyz4*/) {}
  virtual void text(char * /*str */) {}
  virtual void comment (char *) {}

  float textpos[3];
  void super_text_position(float x, float y, float z);
  virtual void text_position(float /* x*/, float /* y*/, float /* z*/) {}

  // these are here for completeness, but currently only the 'token'
  // one uses it
  virtual void pick_point(float * /*xyz*/, int /*id*/) {}

};

#endif

