/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: FilesFormsObj.C,v $
 *	$Author: ulrich $	$Locker:  $		$State: Exp $
 *	$Revision: 1.12 $	$Date: 1997/03/20 03:37:47 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to load a new molecule from a set of files.
 *
 ***************************************************************************/

#include <stdio.h>
#include "FilesFormsObj.h"
#include "CmdMenu.h"
#include "CmdMol.h"
#include "MoleculeFile.h"
#include "Global.h"
#include "utilities.h"

// different ways to read in a molecule
enum GUIMolFormats {
	NEWMOL_PSF_PDB,		NEWMOL_PSF_DCD,		NEWMOL_PDB,
	NEWMOL_PDB_DCD,		NEWMOL_RASTER3D,	NEWMOL_PARM_CRD,
	NEWMOL_EDM,
	NEWMOL_B_T,		NEWMOL_B_PREP,		NEWMOL_B_BS,
	NEWMOL_B_CAR,		NEWMOL_B_BOOG,		NEWMOL_B_CACCRT,
	NEWMOL_B_CADPAC,	NEWMOL_B_CHARMM,	NEWMOL_B_C3D1,
	NEWMOL_B_C3D2,		NEWMOL_B_CSSR,		NEWMOL_B_FDAT,
	NEWMOL_B_C,		NEWMOL_B_FEAT,		NEWMOL_B_F,
	NEWMOL_B_GAMOUT,	NEWMOL_B_G,		NEWMOL_B_GAUOUT,
	NEWMOL_B_HIN,		NEWMOL_B_ISIS,		NEWMOL_B_MACMOL,
	NEWMOL_B_K,		NEWMOL_B_MICRO,		NEWMOL_B_MI,
	NEWMOL_B_MO,		NEWMOL_B_MM3,		NEWMOL_B_MMADS,
	NEWMOL_B_MDL,		NEWMOL_B_MOLEN,		NEWMOL_B_AC,
	NEWMOL_B_AI,		NEWMOL_B_AO,		NEWMOL_B_PC,
	NEWMOL_B_QUANTA,	NEWMOL_B_SHELX,		NEWMOL_B_SPAR,
	NEWMOL_B_SEMI,		NEWMOL_B_SPMM,		NEWMOL_B_MOL,
	NEWMOL_B_MOL2,		NEWMOL_B_CON,		NEWMOL_B_MACCS2,
	NEWMOL_B_MACCS3,	NEWMOL_B_UNIXYZ,	NEWMOL_B_X,
	NEWMOL_B_XED,		NEWMOL_METHODS };

// where do the babel formats begin? They should be the last formats in the
// list
#define FIRST_BABEL_FORMAT	NEWMOL_B_T


// different file types supported here
enum GUIMolFileTypes {
	NEWMOL_TYPE_PDB,	NEWMOL_TYPE_PSF,	NEWMOL_TYPE_DCD,
	NEWMOL_TYPE_RASTER3D,	NEWMOL_TYPE_PARM,       NEWMOL_TYPE_CRD,
	NEWMOL_TYPE_EDM,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_B_T,	NEWMOL_TYPE_B_PREP,	NEWMOL_TYPE_B_BS,
	NEWMOL_TYPE_B_CAR,	NEWMOL_TYPE_B_BOOG,	NEWMOL_TYPE_B_CACCRT,
	NEWMOL_TYPE_B_CADPAC,	NEWMOL_TYPE_B_CHARMM,	NEWMOL_TYPE_B_C3D1,
	NEWMOL_TYPE_B_C3D2,	NEWMOL_TYPE_B_CSSR,	NEWMOL_TYPE_B_FDAT,
	NEWMOL_TYPE_B_C,	NEWMOL_TYPE_B_FEAT,	NEWMOL_TYPE_B_F,
	NEWMOL_TYPE_B_GAMOUT,	NEWMOL_TYPE_B_G,	NEWMOL_TYPE_B_GAUOUT,
	NEWMOL_TYPE_B_HIN,	NEWMOL_TYPE_B_ISIS,	NEWMOL_TYPE_B_MACMOL,
	NEWMOL_TYPE_B_K,	NEWMOL_TYPE_B_MICRO,	NEWMOL_TYPE_B_MI,
	NEWMOL_TYPE_B_MO,	NEWMOL_TYPE_B_MM3,	NEWMOL_TYPE_B_MMADS,
	NEWMOL_TYPE_B_MDL,	NEWMOL_TYPE_B_MOLEN,	NEWMOL_TYPE_B_AC,
	NEWMOL_TYPE_B_AI,	NEWMOL_TYPE_B_AO,	NEWMOL_TYPE_B_PC,
	NEWMOL_TYPE_B_QUANTA,	NEWMOL_TYPE_B_SHELX,	NEWMOL_TYPE_B_SPAR,
	NEWMOL_TYPE_B_SEMI,	NEWMOL_TYPE_B_SPMM,	NEWMOL_TYPE_B_MOL,
	NEWMOL_TYPE_B_MOL2,	NEWMOL_TYPE_B_CON,	NEWMOL_TYPE_B_MACCS2,
	NEWMOL_TYPE_B_MACCS3,	NEWMOL_TYPE_B_UNIXYZ,	NEWMOL_TYPE_B_X,
	NEWMOL_TYPE_B_XED,  	NEWMOL_TYPES };

//
// static strings and flags for these molecule data file combinations
//

// type of structure for the given method for reading in a file
static int newmol_struct_types[NEWMOL_METHODS] = {
	NEWMOL_TYPE_PSF,	NEWMOL_TYPE_PSF,	NEWMOL_TYPE_PDB,
	NEWMOL_TYPE_PDB,	NEWMOL_TYPE_RASTER3D,	NEWMOL_TYPE_PARM,
	NEWMOL_TYPE_EDM,
	NEWMOL_TYPE_B_T,	NEWMOL_TYPE_B_PREP,	NEWMOL_TYPE_B_BS,
	NEWMOL_TYPE_B_CAR,	NEWMOL_TYPE_B_BOOG,	NEWMOL_TYPE_B_CACCRT,
	NEWMOL_TYPE_B_CADPAC,	NEWMOL_TYPE_B_CHARMM,	NEWMOL_TYPE_B_C3D1,
	NEWMOL_TYPE_B_C3D2,	NEWMOL_TYPE_B_CSSR,	NEWMOL_TYPE_B_FDAT,
	NEWMOL_TYPE_B_C,	NEWMOL_TYPE_B_FEAT,	NEWMOL_TYPE_B_F,
	NEWMOL_TYPE_B_GAMOUT,	NEWMOL_TYPE_B_G,	NEWMOL_TYPE_B_GAUOUT,
	NEWMOL_TYPE_B_HIN,	NEWMOL_TYPE_B_ISIS,	NEWMOL_TYPE_B_MACMOL,
	NEWMOL_TYPE_B_K,	NEWMOL_TYPE_B_MICRO,	NEWMOL_TYPE_B_MI,
	NEWMOL_TYPE_B_MO,	NEWMOL_TYPE_B_MM3,	NEWMOL_TYPE_B_MMADS,
	NEWMOL_TYPE_B_MDL,	NEWMOL_TYPE_B_MOLEN,	NEWMOL_TYPE_B_AC,
	NEWMOL_TYPE_B_AI,	NEWMOL_TYPE_B_AO,	NEWMOL_TYPE_B_PC,
	NEWMOL_TYPE_B_QUANTA,	NEWMOL_TYPE_B_SHELX,	NEWMOL_TYPE_B_SPAR,
	NEWMOL_TYPE_B_SEMI,	NEWMOL_TYPE_B_SPMM,	NEWMOL_TYPE_B_MOL,
	NEWMOL_TYPE_B_MOL2,	NEWMOL_TYPE_B_CON,	NEWMOL_TYPE_B_MACCS2,
	NEWMOL_TYPE_B_MACCS3,	NEWMOL_TYPE_B_UNIXYZ,	NEWMOL_TYPE_B_X,
	NEWMOL_TYPE_B_XED };

// type of coordinate file for the given method of reading in a file. If
// NEWMOL_TYPE_NONE, the method only requires a single structure file.
static int newmol_coor_types[NEWMOL_METHODS] = {
	NEWMOL_TYPE_PDB,	NEWMOL_TYPE_DCD,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_DCD,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_CRD,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE };

// name listed in the menu for how to read in a molecule
static char *newmol_method_names[NEWMOL_METHODS] = {
	"psf and pdb",		"psf and dcd",		"pdb only",
	"pdb and dcd",		"Raster3D",		"parm and crd",
	"electron density map",
	"Alchemy",		"AMBER PREP",		"Ball and Stick",
	"Biosym .CAR",		"Boogie",		"Cacao Cartesian",
	"Cambridge CADPAC",	"CHARMm",		"Chem3D Cartesian 1",
	"Chem3D Cartesian 2",	"CSD CSSR",		"CSD FDAT",
	"CSD GSTAT",		"Feature",		"Free Form Fractional",
	"GAMESS Output",	"Gaussian Z-Matrix",	"Gaussian Output",
	"Hyperchem HIN",	"MDL Isis", 		"Mac Molecule",
	"Macromodel",		"Micro World",		"MM2 Input",
	"MM2 Output",		"MM3",			"MMADS",
	"MDL MOLfile",		"MOLIN",		"Mopac Cartesian",
	"Mopac Internal",	"Mopac Output",		"PC Model",
	"Quanta",		"ShelX",		"Spartan",
	"Spartan Semi-Empirical", "Spartan Molecular Mechanics", "Sybyl Mol",
	"Sybyl 2",		"Conjure",		"Maccs 2D",
	"Maccs 3D",		"UniChem XYZ",		"XYZ",
	"XED" };

// structure codes used when creating new molecule command
static int newmol_struct_codes[NEWMOL_METHODS] = {
	MoleculeFile::PSF,	MoleculeFile::PSF,	MoleculeFile::PDB,
	MoleculeFile::PDB,	MoleculeFile::RASTER3D,	MoleculeFile::PARM, 
	MoleculeFile::EDM,
	MoleculeFile::T,	MoleculeFile::PREP,	MoleculeFile::BS,
	MoleculeFile::CAR,	MoleculeFile::BOOG,	MoleculeFile::CACCRT,
	MoleculeFile::CADPAC,	MoleculeFile::CHARMM,	MoleculeFile::C3D1,
	MoleculeFile::C3D2,	MoleculeFile::CSSR,	MoleculeFile::FDAT,
	MoleculeFile::C,	MoleculeFile::FEAT,	MoleculeFile::F,
	MoleculeFile::GAMOUT,	MoleculeFile::G,	MoleculeFile::GAUOUT,
	MoleculeFile::HIN,	MoleculeFile::ISIS,	MoleculeFile::MACMOL,
	MoleculeFile::K,	MoleculeFile::MICRO,	MoleculeFile::MI,
	MoleculeFile::MO,	MoleculeFile::MM3,	MoleculeFile::MMADS,
	MoleculeFile::MDL,	MoleculeFile::MOLEN,	MoleculeFile::AC,
	MoleculeFile::AI,	MoleculeFile::AO,	MoleculeFile::PC,
	MoleculeFile::QUANTA,	MoleculeFile::SHELX,	MoleculeFile::SPAR,
	MoleculeFile::SEMI,	MoleculeFile::SPMM,	MoleculeFile::MOL,
	MoleculeFile::MOL2,	MoleculeFile::CON,	MoleculeFile::MACCS2,
	MoleculeFile::MACCS3,	MoleculeFile::UNIXYZ,	MoleculeFile::X,
	MoleculeFile::XED };

// coordinate file codes used when creating a new molecule.  If UNKNOWN,
// a coordinate file is not used by the particular molecule format
static int newmol_coor_codes[NEWMOL_METHODS] = {
	CoorFileData::PDB,	CoorFileData::DCD,	CoorFileData::UNKNOWN,
	CoorFileData::DCD,	CoorFileData::UNKNOWN,  CoorFileData::CRD,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN,
	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN };

// static strings and flags for these molecule data file types
static char *newmol_type_names[NEWMOL_TYPES] = {
	"pdb",			"psf",			"dcd",
	"r3D",			"parm",			"crd",
	"edm",			"",
	"t",			"prep",			"bs",
	"car",			"boog",			"caccrt",
	"cadpac",		"charmm",		"c3d1",
	"c3d2",			"cssr",			"fdat",
	"c",			"feat",			"f",
	"gamout",		"g",			"gauout",
	"hin",			"isis",			"macmol",
	"k",			"micro",		"mi",
	"mo",			"mm3",			"mmads",
	"mdl",			"molen",		"ac",
	"ai",			"ao",			"pc",
	"quanta",		"shelx",		"spar",
	"semi",			"spmm",			"mol",
	"mol2",			"con",			"maccs2",
	"maccs3",		"unixyz",		"x",
	"xed" };

static char *newmol_type_masks[NEWMOL_TYPES] = {
	"*.[pP][dD][bB]",	"*.[pP][sS][fF]",	"*.[dD][cC][dD]",	
	"*",			"*.[pP][aA][rR][mM]",   "*.[cC][rR][dD]",
	"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*",			"*",			"*",
	"*" };


/////////////////////////  constructor  
FilesFormsObj::FilesFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("files", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = filesMenu;
  lightButton = files_menu_button;
  offButton = files_menu_off;
  
  // turn on if required
  if(sh)
    On();
}

/////////////////////////////  private routines  

// select a filename
void  FilesFormsObj::choose_filename(FL_OBJECT *txt, int ftype, char *prompt) {
  freeze();

  // get file name
  char *mask = newmol_type_masks[ftype];
  char *fname = (char *) fl_show_fselector(prompt, "", mask, "");
  if(fname)
    fl_set_input(txt, fname);

  unfreeze();
}


void FilesFormsObj::reset_mol_type(int newtype) {
  char selectbuf[16];

  freeze();
    // set the structure file inputs properly
    strcpy(selectbuf, "Select ");
    strcat(selectbuf, newmol_type_names[newmol_struct_types[newtype]]);
    fl_set_object_label(struct_file_select, selectbuf);
    fl_set_input(struct_file_name, "");

    // set the coordinate file inputs properly, and show inputs if needed
    fl_set_input(coor_file_name, "");
    if(newmol_coor_types[newtype] != NEWMOL_TYPE_NONE) {
      fl_show_object(coor_file_text);
      fl_show_object(coor_file_select);
      fl_show_object(coor_file_name);
      strcpy(selectbuf, "Select ");
      strcat(selectbuf, newmol_type_names[newmol_coor_types[newtype]]);
      fl_set_object_label(coor_file_select, selectbuf);
    } else {
      fl_hide_object(coor_file_text);
      fl_hide_object(coor_file_select);
      fl_hide_object(coor_file_name);
    }
  unfreeze();
}


//////////////////////  protected virtual routines  

// routine to check the main form for use
int FilesFormsObj::check(FL_OBJECT *obj) {

  if(obj == struct_file_type) {
    reset_mol_type(fl_get_browser(obj) - 1);

  } else if(obj == struct_file_name || obj == coor_file_name) {
    // do nothing

  } else if(obj == struct_file_select) {
    // select was pressed, get new file name
    choose_filename(struct_file_name, 
	newmol_struct_types[fl_get_browser(struct_file_type) - 1],
      	"Select molecular structure file:");
    
  } else if(obj == coor_file_select) {
    // select was pressed, get new file name
    choose_filename(coor_file_name, 
	newmol_coor_types[fl_get_browser(struct_file_type) - 1],
      	"Select initial coordinate file:");
  
  } else if(obj == files_reset) {
    reset();
  
  } else if(obj == files_cancel) {
    reset();
    addcommand(new CmdMenuShow(id(),FALSE,id()));

  } else if(obj == files_load) {
    char *strfile = (char *) fl_get_input(struct_file_name);
    char *coorfile = (char *) fl_get_input(coor_file_name);
    int ftype = fl_get_browser(struct_file_type) - 1;

    if(strfile && strlen(strfile) < 1)
      strfile = NULL;
    if(coorfile && strlen(coorfile) < 1)
      coorfile = NULL;

    if(!strfile) {
      msgErr << "You must specify a structure file on the top line."<< sendmsg;
    } else if(newmol_coor_types[ftype] != NEWMOL_TYPE_NONE && !coorfile) {
      msgErr << "You must specify a coordinate file on the bottom line.";
      msgErr << sendmsg;
    } else {
      int sftype = newmol_struct_codes[ftype];
      int cftype = newmol_coor_codes[ftype];
      fl_hide_object(obj);
      fl_show_object(mol_wait_text);
      addcommand(new CmdMolNew(strfile, sftype, coorfile, cftype, id()));
    }

  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int FilesFormsObj::forms_act_on_command(int type, Command *cmd, int suc) {

  MSGDEBUG(3,"FilesFormsObj: acting on command " << type << sendmsg);

  // check all the possible commands that we look for ...
  // base class already takes care of new, del, and top
  if(type == Command::MOL_NEW && cmd->getUIid() == id() && suc) {
    // this UI did the loading, and it worked ... turn off the menu
    addcommand(new CmdMenuShow(id(),FALSE,id()));

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void FilesFormsObj::init(void) {
  int i;

  freeze();
    // set all the basic values for controllers, etc
    fl_clear_browser(struct_file_type);
    for(i=0; i < FIRST_BABEL_FORMAT; i++)
      fl_add_browser_line(struct_file_type, newmol_method_names[i]);

    // add babel formats, if babel is available
    if(vmdBabelBin)
      for(i=FIRST_BABEL_FORMAT; i < NEWMOL_METHODS; i++)
        fl_add_browser_line(struct_file_type, newmol_method_names[i]);

    // set the proper default method for reading in a file
    fl_select_browser_line(struct_file_type, NEWMOL_PSF_PDB + 1);
    
    // do a reset to set initial values
    reset();

  unfreeze();
}


// reset the user interface
void FilesFormsObj::reset(void) {
  freeze();
    fl_hide_object(mol_wait_text);
    fl_show_object(files_load);

    reset_mol_type(fl_get_browser(struct_file_type) - 1);
  unfreeze();
}

