/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: GeometryList.h,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.6 $      $Date: 1997/03/13 17:38:56 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * This maintains a set of lists of geometry monitors, and draws them to
 * the scene.  This is a Displayable which keeps the graphical representations
 * of the geometry monitors; this is also a Pickable, and creates and controls
 * the PickMode objects which create new geometry monitors via the pointers.
 * This object keeps a set of ResizeArray, each of which is a 'category' for
 * geometry monitors (i.e Atoms, Bonds, etc.) which contain Geometry objects. 
 *
 ***************************************************************************/
#ifndef GEOMETRYLIST_H
#define GEOMETRYLIST_H

#include "Displayable.h"
#include "NameList.h"
#include "ResizeArray.h"
#include "Geometry.h"
#include "DispCmds.h"
class MoleculeList;


// structure used to store data about each category
typedef ResizeArray<Geometry *>  *GeomListPtr;
typedef struct {
  GeomListPtr geomList;
  int defaultColor;
} GeomListStruct;


class GeometryList : public Displayable3D {

private:
  // list of ResizeArray's which hold the geometry objects
  NameList<GeomListStruct *> geomLists;
  
  // integer for color category with the colors to use.  If < 0, use
  // default colors
  int colorCat;

  // molecule list to use for data
  MoleculeList *molList;

  // useful display command objects
  DispCmdMaterials cmdMaterials;
  DispCmdColorIndex cmdColor;
  DispCmdLineType cmdLineType;
  DispCmdLineWidth cmdLineWidth;

  // keep track of the most recently selected 'pick'.  
  Geometry *top_geo;

protected:
  // do action when a new color list is provided
  virtual void do_use_colors(void);

  // do action due to the fact that a color for the given ColorList for
  // the specified category has changed
  virtual void do_color_changed(ColorList *, int);

  // create a new PickMode object when necesssary
  virtual PickMode *create_pick_mode(int);

public:
  // constructor: the Scene to register this with
  GeometryList(Scene *, MoleculeList * = NULL);
  
  // destructor: must clean up all lists
  virtual ~GeometryList(void);
  
  //
  // return information about this class
  //
  
  // number of geometry lists
  int num_lists(void) { return geomLists.num(); }
  
  // access the pointer to the Nth geometry list
  GeomListPtr geom_list(int n) { return (geomLists.data(n))->geomList; }
  
  // return pointer to the geometry list with the given name, NULL if not found
  GeomListPtr geom_list(char *nm) {
    int glistindex = geom_list_index(nm);
    return (glistindex >= 0 ? geom_list(glistindex) : (GeomListPtr) NULL);
  }

  // return the name of the Nth geometry list
  char *geom_list_name(int n) { return geomLists.name(n); }
  
  // return the index of the geom list with the given name, -1 if not found
  int geom_list_index(char *nm) { return geomLists.typecode(nm); }
  
  //
  // routines to add/delete/modify geometry categories
  //
  
  // add a new category: specify the name, and default color.  Return index of
  // new list.
  int add_geom_list(char *, int);
  
  // delete the Nth category.  Return success.
  int del_geom_list(int);
  
  // delete the category with the given name
  void del_geom_list(char *nm) { del_geom_list(geom_list_index(nm)); }

  //
  // routines to add/delete/modify geometry objects
  // NOTE: after Geometry objects have been added, they remain in the
  //	lists until either explicitely deleted, or their 'ok' routine
  //	no longer returns 'TRUE'.  If 'ok' returns FALSE, they are
  //	deleted.
  //

  // add a new geometry object to the list with the given index.  Return
  // success.
  int add_geometry(int, Geometry *);

  // add a new geometry object to the list with the given name.  Return
  // success.
  int add_geometry(char *nm, Geometry *g) {
    return add_geometry(geom_list_index(nm), g);
  }
  
  // delete the Nth geometry object in the list with the given index.
  // return success.
  int del_geometry(int, int);
  
  // delete the Nth geometry object in the list with the given name.
  // return success.
  int del_geometry(char *nm, int n) {
    return del_geometry(geom_list_index(nm), n);
  }
  
  // show/hide the Nth geometry monitor in the given category.  If
  // N < 0, hide ALL monitors in that category.  Return success.
  // args: category, geometry monitor index, show (T) or hide (F)
  int show_geometry(int, int, int);

  // same as above, but giving a name for the category instead of index
  int show_geometry(char *nm, int n, int s) {
    return show_geometry(geom_list_index(nm), n, s);
  }

  //
  // public virtual drawing routines
  //
  
  // prepare for drawing ... do any updates needed right before draw.
  virtual void prepare(DisplayDevice *);

};

#endif
  
