/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: GeometryMol.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.6 $      $Date: 1996/12/12 21:32:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A base class for all Geometry objects which measure information about
 * atoms in a molecule.  A molecule Geometry monitor is assumed to operate
 * on N atoms, and be able to calculate a single floating-point value for
 * those atoms.  (i.e. the angle formed by three atoms in space)
 *
 ***************************************************************************/

#include "GeometryMol.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "Displayable.h"
#include "DispCmds.h"
#include "utilities.h"

// useful drawing command used here
static DispCmdLine cmdLineGeom;
static DispCmdTextPos cmdTextPosGeom;



////////////////////////  constructor  /////////////////////////////////
GeometryMol::GeometryMol(int n, int *mols, int *atms, MoleculeList *mlist)
	: Geometry(n) {

  molList = mlist;
  gmName = NULL;
  uniquegmName = NULL;

  for(int i=0; i < numItems; i++) {
    objIndex[i] = mols[i];
    comIndex[i] = atms[i];
    
    // make sure an atom is not repeated in this list
    if(i > 0 && objIndex[i-1]==objIndex[i] && comIndex[i-1]==comIndex[i]) {
      // set a bogus value for the first atom index, to make the
      // check_mol routine fail
      comIndex[0] = (-1);
    }
  }

  // sort the items properly
  sort_items();

  // create the name for this object
  set_name();
}


////////////////////////  constructor  /////////////////////////////////
GeometryMol::~GeometryMol(void) {
  // delete name if necessary
  if(gmName)
    delete [] gmName;
  if(uniquegmName)
    delete [] uniquegmName;
}


////////////////////////  protected routines  //////////////////////////

// set the name of this item
void GeometryMol::set_name(void) {
  char namebuf[256];
  char namebuf2[256];
  Molecule *mol;
  register int i;

  if(items() < 1)
    return;

  // put first name in string
  if((mol = check_mol(objIndex[0], comIndex[0]))) {
    strcpy(namebuf, mol->atom_full_name(comIndex[0]));
    strcpy(namebuf2, mol->atom_short_name(comIndex[0]));
  } else
    return;

  // put rest of names in string in format: n1/n2/n3/.../nN
  for(i = 1; i < items(); i++) {
    if((mol = check_mol(objIndex[i], comIndex[i]))) {
      strcat(namebuf, "/");
      strcat(namebuf, mol->atom_full_name(comIndex[i]));
      strcat(namebuf2, "/");
      strcat(namebuf2, mol->atom_short_name(comIndex[i]));
    } else {
      return;
    }
  }
  
  // now make a copy of this name
  if(gmName)
    delete [] gmName;
  if(uniquegmName)
    delete [] uniquegmName;

  uniquegmName = stringdup(namebuf);
  gmName = stringdup(namebuf2);
}


// sort the elements in the list, so that the lowest atom index is first
// (but preserve the relative order, i.e. a-b-c or c-b-a)
void GeometryMol::sort_items(void) {
  register int i,j;

  // swap order if first component index > last component index
  if( (comIndex[0] > comIndex[items()- 1]) ||
      (comIndex[0] == comIndex[items()-1] &&
      		objIndex[0] > objIndex[items()-1]) ) {
    for(i=0, j=(items() - 1); i < j; i++, j--) {
      int tmpindex = comIndex[i];
      comIndex[i] = comIndex[j];
      comIndex[j] = tmpindex;
      tmpindex = objIndex[i];
      objIndex[i] = objIndex[j];
      objIndex[j] = tmpindex;
    }
  }
}


// check whether the given molecule & atom index is OK
// if OK, return Molecule pointer; otherwise, return NULL
Molecule *GeometryMol::check_mol(int m, int a) {

  Molecule *mol = molList->molecule(molList->mol_index_from_id(m));

  if(!mol || a < 0 || a >= mol->nAtoms)
    mol = NULL;
  
  return mol;
}


// for the given Molecule, find the TRANSFORMED coords for the given atom
// return Molecule pointer if successful, NULL otherwise.
Molecule *GeometryMol::transformed_atom_coord(int m, int a, float *pos) {
  Molecule *mol;

  // only return value if molecule is legal and atom is displayed
  if((mol = normal_atom_coord(m, a, pos)) && mol->atom_displayed(a)) {
    
    // now multiply it by the molecule's tranformation matrix
    (mol->tm).multpoint3d(pos, pos);

    // calculation was successful; return the molecule pointer
    return mol;
  }
  
  // if here, error (i.e. atom not displayed, or not proper mol id)
  return NULL;
}


// for the given Molecule, find the UNTRANSFORMED coords for the given atom
// return Molecule pointer if successful, NULL otherwise.
Molecule *GeometryMol::normal_atom_coord(int m, int a, float *pos) {
  Timestep *now;
  Molecule *mol;

  // get the molecule pointer, and get the coords for the current timestep
  if((mol = check_mol(m, a)) && (now = mol->current())) {
    memcpy((void *)pos, (void *)(now->pos + 3*a), 3*sizeof(float));
    return mol;
  }
  
  // if here, error (i.e. atom not displayed, or not proper mol id)
  return NULL;
}


// draws a line between the two given points
void GeometryMol::display_line(float *p1, float *p2, Displayable *d) {
  cmdLineGeom.putdata(p1, p2, d);
}


// print given text at current valuePos position
void GeometryMol::display_string(char *str, Displayable *d) {
  DispCmdText cmdTextGeom(str);
  cmdTextPosGeom.putdata(valuePos, d);
  cmdTextGeom.put(d);
}

////////////////////  public virtual routines  //////////////////////////

// return the name of this geometry marker; by default, just blank
char *GeometryMol::name(void) {
  return (gmName ? gmName : Geometry::name());
}


// return 'unique' name of the marker, which should be different than
// other names for different markers of this same type
char *GeometryMol::unique_name(void) {
  return (uniquegmName ? uniquegmName : Geometry::unique_name());
}


// check whether the geometry value can still be calculated
int GeometryMol::ok(void) {
  register int i;
  
  for(i=0; i < numItems; i++)
    if(!check_mol(objIndex[i], comIndex[i]))
      return FALSE;

  return TRUE;
}


// calculate a whole list of items, if this object can do so.  Return success.
int GeometryMol::calculate_all(ResizeArray<float> &valArray) {
  register int i;
  int num_ts, orig_ts;

  // use the first molecule to determine which frames to cycle through
  Molecule *mol = molList->molecule(molList->mol_index_from_id(objIndex[0]));
  if( ! has_value() || !mol )
    return FALSE;

  // get the max & current frame number, and make sure there are frames
  num_ts = mol->num();
  if((orig_ts = mol->frame()) < 0)
    return FALSE;

  // go through all the frames, calculating values
  for(i=0; i < num_ts; i++) {
    mol->override_current_frame(i);
    ///    valArray[2*i] = (float)i;
    ///    valArray[2*i + 1] = calculate();
    valArray[i] = calculate();
  }
  
  // reset the current frame and current value of this geometry monitor
  mol->override_current_frame(orig_ts);
  calculate();
  
  return TRUE;
}

