/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Inform.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1995/05/11 23:01:59 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Inform - takes messages and displays them to the given ostream.
 *	Also creates 4 instances: msgInfo, msgWarn, msgErr, and msgDebug.
 *	A message is sent to an Inform object by treating it as an ostream,
 *	then ending the message by sending the 'sendmsg' manipulator.
 *	Each message is assigned the current 'level of interest'; the lower
 *	the level, the more important it is.  Each Inform object is set for
 *	a current level; messages with a level <= the current level are
 *	displayed.  Levels run from 1 ... 10.
 *
 ***************************************************************************/
#ifndef INFORM_H
#define INFORM_H

// macro for sending debug messages to the specific debugging Inform object
// (msgDebug, defined at the end of this file).  The first version includes the
// send command in the compiled program, the second leaves it out of the
// executable (reducing the size and improving performance).
// 'informobj' is the Inform instance to use; level is the level to set the
// message to (MIN_MSG_LEVEL to MAX_MSG_LEVEL); msg is a single command
// to send the message (should NOT be terminated by a semicolon)

#ifdef VMDDEBUG
#define MSGDEBUG(level, msg) { msgDebug.msg_level(level) << msg; }
#else
#define MSGDEBUG(level, msg)
#endif

#include <strstream.h>

// largest message (in bytes) that can be kept
#define MAX_MSG_SIZE	(1024 * 8)
#define MIN_MSG_LEVEL	1
#define MAX_MSG_LEVEL	10


class Inform {

private:
  // name of this object; put at start of each message.
  char *name;

  // string holding the message to send, and output stream obj to format it
  char *msgBuf;
  ostrstream *msg;

  // where to put the messages; can be changed, by default = cout
  ostream *msgDest;

  // do we output the message?
  int On;

  // do we need a newline before the message?
  int needNewline;
  
  // output level of this Inform object; messages with a level <= the output
  // level are printed
  int outputLevel;
  
  // current message level; this is set by the 'levelN' manipulators, or
  // by the routine msg_level(int).  After a message is printed, the current
  // message level is reset to the minimum.
  int msgLevel;

  // print out the message in the given buffer.  Will modify the string,
  // so beware.  Arguments: string
  void display_message(char *);

public:
  // constructor: name, is it on or off by default
  Inform(char *, int ison=1);

  // destructor
  ~Inform(void);
  
  // turn messages on/off
  void on(int o) { On = o; }
  int on(void) { return On; }

  // get/set the current output and message levels
  int output_level(void) { return outputLevel; }
  Inform& output_level(int ol) {
    if(ol >= MIN_MSG_LEVEL && ol <= MAX_MSG_LEVEL)
      outputLevel = ol;
    return *this;
  }

  int msg_level(void) { return msgLevel; }
  Inform& msg_level(int ol) {
    if(ol >= MIN_MSG_LEVEL && ol <= MAX_MSG_LEVEL)
      msgLevel = ol;
    return *this;
  }

  // flag we need a newline before next message is displayed
  void need_newline(int nl) { needNewline = nl; }
  int need_newline(void) { return needNewline; }

  // change the destination for the messages
  void destination(ostream *os) { msgDest = os; }
  ostream *destination(void) { return msgDest; }

  // the signal has been given, process the message.  Return ref to object.
  Inform& send(void);

  // overload the '<<' operator to allow items to be output to this object.
  // This will then send the data to our message buffer.
  Inform& operator<<(const char *str) { *msg << str; return *this; }
  Inform& operator<<(char c) { *msg << c; return *this; }
  Inform& operator<<(short i) { *msg << i; return *this; }
  Inform& operator<<(int i) { *msg << i; return *this; }
  Inform& operator<<(long i) { *msg << i; return *this; }
  Inform& operator<<(float i) { *msg << i; return *this; }
  Inform& operator<<(double i) { *msg << i; return *this; }
  Inform& operator<<(Inform& (*f)(Inform&)) { return f(*this); }

};

// manipulator for signaling we want to send the message.
extern Inform& sendmsg(Inform&);

// manipulators for setting the current msg level
extern Inform& level1(Inform&);
extern Inform& level2(Inform&);
extern Inform& level3(Inform&);
extern Inform& level4(Inform&);
extern Inform& level5(Inform&);
extern Inform& level6(Inform&);
extern Inform& level7(Inform&);
extern Inform& level8(Inform&);
extern Inform& level9(Inform&);
extern Inform& level10(Inform&);

// external declarations of 4 inform objects
extern Inform msgInfo;
extern Inform msgWarn;
extern Inform msgErr;
extern Inform msgDebug;

#endif

