/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Measure.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1997/03/13 17:38:56 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  These are the various Tcl "measure" commands and the base (non-Tcl)
 * commands used to solve them.  These include center of mass (any weight)
 * radius of gyration, and best fit alignment (in the leqst squares sense).
 *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include "tcl.h"
#include "Global.h"
#include "AtomSel.h"
#include "TclCommands.h"
#include "MoleculeList.h"
#include "SymbolTable.h"
#include "Matrix4.h"
#include "UIText.h"

// Standard functions available to everyone
static char *measure_error_messages[] = {
  "no atom selection",                              // -1
  "no atoms in selection",                          // -2
  "incorrect number of weights for selection",      // -3
  "internal error: NULL pointer given",             // -4
  "divide by zero",                                 // -5
  "molecule was deleted(?)",                        // -6
  "cannot understand weight parameter",             // -7
  "non-number given as parameter",                  // -8
  "two selections don't have the same number of atoms", // -9
  "internal error: out of range",                   // -10
  "no coordinates in selection",                    // -11
  "couldn't compute eigenvalue/vectors",            // -12
  "unknown Tcl problem"                             // -13
};
  
const char *measure_error(int errno) {
  if (errno >= 0 || errno < -13) return "bad error number";
  return measure_error_messages[-errno - 1];
}

// Center of mass (various weights allowed)
// Takes an atom selection and one of two possible weights
//  1) if num == sel.selected ; assumes there is one weight per 
//           selected atom
//  2) if num == sel.num_atoms; assumes weight[i] is for atom[i]
// returns center coordinate in float com[3]
// returns 0 if valid data
// returns <0 if invalid data
int measure_center(AtomSel *sel, int num, float *weight, float *com)
{
  if (!sel) return -1;
  if (!sel->molecule()) return -6;
  if (num != sel -> num_atoms && num != sel -> selected) {
    printf("num %d num_atoms %d selected %d\n", num, sel->num_atoms, 
	   sel->selected);
    return -3;
  }
  if (!com || !weight) return -4;

  // compute the center of mass
  float x=0, y=0, z=0, w = 0;
  float *framepos = sel -> coordinates();
  if (!framepos) return -11;

  int flg = (num == sel -> num_atoms); // is there 1 weight per atom?
  int offset = 0;
  for (int i=0; i<sel -> num_atoms; i++) {
    if (sel -> on[i]) {
      w += weight[offset];
      x += weight[offset] * framepos[3*i+0];
      y += weight[offset] * framepos[3*i+1];
      z += weight[offset] * framepos[3*i+2];
      offset++;
    } else if (flg) {
      offset++;
    }
  }

  if (w == 0) {
    return -5;
  }
  com[0] = x / w;
  com[1] = y / w;
  com[2] = z / w;
  return 0;
}

// Function: measure_minmax(AtomSel *)
//  Returns: the min and max x,y,z values in the selection (in the
//     min_coord and max_coord arrays)
//  returns 0 if success
//  returns <0 if not
int measure_minmax(AtomSel *sel, float *min_coord, float *max_coord)
{
  if (!sel) return -1;
  if (!sel->molecule()) return -6;
  if (!min_coord || !max_coord) return -4;

  // sel is good, get coords
  float *framepos = sel -> coordinates();
  if (!framepos) return -11;

  // everything is good; get minmax values
  int flg = 0;
  for (int i=0; i<sel->num_atoms; i++) {
    if (sel->on[i]) {
      if (!flg) {
	flg = 1;
	min_coord[0] = max_coord[0] = framepos[i*3+0];
	min_coord[1] = max_coord[1] = framepos[i*3+1];
	min_coord[2] = max_coord[2] = framepos[i*3+2];
      } else {
	float tmp = framepos[i*3+0];
	if (tmp < min_coord[0]) min_coord[0] = tmp; else {
	  if (tmp > max_coord[0]) max_coord[0] = tmp;}
	tmp = framepos[i*3+1];
	if (tmp < min_coord[1]) min_coord[1] = tmp; else {
	  if (tmp > max_coord[1]) max_coord[1] = tmp;}
	tmp = framepos[i*3+2];
	if (tmp < min_coord[2]) min_coord[2] = tmp; else {
	  if (tmp > max_coord[2]) max_coord[2] = tmp;}
      }
    }
  }
  if (!flg) {
    return -2;
  }
  return 0;
}


/// measure the radius of gyration, including the given weights
//  rgyr := sqrt(sum (mass(n) ( r(n) - r(com) )^2)/sum(mass(n)))
//  The return value, a float, is put in 'float *rgyr'
//  The function return value is 0 if ok, <0 if not
int measure_rgyr(AtomSel *sel, int num, float *weight, float *rgyr)
{
  // compute the center of mass with the current weights
  float com[3];
  int ret_val;
  // this also checks the input parameters
  ret_val = measure_center(sel, num, weight, com);
  if (ret_val < 0) return ret_val;
  if (!rgyr) return -4;
  
  // I know a coord set exists (because measure_center returned OK)
  float *framepos = sel -> coordinates();

  // measure center of gyration
  float total_w = 0, w;

  int flg = (num == sel -> num_atoms); // is there 1 weight per atom?
  int offset = 0;
  float sum = 0;
  for (int i=0; i<sel -> num_atoms; i++) {
    if (sel -> on[i]) {
      w = weight[offset];
      total_w += w;
      sum += w * distance2(framepos + 3*i, com);
      offset++;
    } else if (flg) {
      offset++;
    }
  }

  if (total_w == 0) {
    return -5;
  }
  // and finalize the computation
  *rgyr = sqrt(sum/total_w);
  return 0;
}

/// measure the rmsd given a selection and weight term
//  1) if num == sel.selected ; assumes there is one weight per 
//           selected atom
//  2) if num == sel.num_atoms; assumes weight[i] is for atom[i]
//  returns 0 and value in rmsd if good
//   return < 0 if invalid
//  Function is::=  rmsd = 
//    sqrt(sum(weight(n) * sqr(r1(i(n))-r2(i(n))))/sum(weight(n)) / N
int measure_rmsd(AtomSel *sel1, AtomSel *sel2, int num, 
		 float *weight, float *rmsd)
{
  // check everything and get the coords
  if (!sel1) return -1;
  if (!sel1->molecule()) return -6;
  if (!sel2) return -1;
  if (!sel2->molecule()) return -6;

  if (!weight || !rmsd) return -4;
  int sel_flg;
  // the number of selected atoms must be the same
  if (sel1 -> selected != sel2 -> selected) {
    return -9;
  }
  // need to know how to traverse the list of weights
  // there could be 1 weight per atom (sel_flg == 1) or 
  // 1 weight per selected atom (sel_flg == 0)
  if (num == sel1 -> num_atoms) {
    sel_flg = 1; // using all elements
  } else {
    sel_flg = 0; // using elements from selection
  }
  float *framepos1 = sel1 -> coordinates();
  if (!framepos1) return -11;
  float *framepos2 = sel2 -> coordinates();
  if (!framepos2) return -11;

  // temporary variables
  *rmsd = 0;
  float w = 0, tmp_w;
  int w_index = 0;  // the term in the weight field to use
  int offset[2] = { 0, 0};
  int count = sel1 -> selected;

  // compute the gyrafion
  while (count-- > 0) {
    while (!sel1 -> on[offset[0]]) {
      offset[0]++;
      if (offset[0] >= sel1 -> num_atoms) return -10;
    }
    while (!sel2 -> on[offset[1]]) {
      offset[1]++;
      if (offset[1] >= sel2 -> num_atoms) return -10;
    }
    // the weight offset to use depends on how many terms there are
    if (sel_flg == 0) {
      tmp_w = weight[w_index++];
    } else {
      tmp_w = weight[offset[0]]; // use the first selection for the weights
    }
    w += tmp_w;
    // both pointing to usable data
    *rmsd += tmp_w * distance2(framepos1 + 3*offset[0], 
			       framepos2 + 3*offset[1]);
    // and advance to the next value
    offset[0]++;
    offset[1]++;
  }
  // finish the rmds calcs
  if (w == 0) {
    return -5;
  }
  *rmsd = sqrt(*rmsd / w);
  // and say rmsd is OK
  return 0;
}

// find the best fit alignment to take the first structure into the second
// Put the result in the matrix 'mat'
// Complication arise because the Tcl implementation of arrays
// is the transpose of that in VMD (which comes from GL).
//  This algorithm comes from Kabsch, Acta Cryst. (1978) A34, 827-828.
// Need the 2nd weight for the com calculation
int measure_fit(AtomSel *sel1, AtomSel *sel2, int num,
		  float *weight, int num2, float *weight2, Matrix4 *mat)
{
  float comx[3];
  float comy[3];
  int ret_val;
  ret_val = measure_center(sel1, num, weight, comx);
  if (ret_val < 0) {
    return ret_val;
  }
  ret_val = measure_center(sel2, num2, weight2, comy);
  if (ret_val < 0) {
    return ret_val;
  }
  if (!mat) return -4;

  int sel_flg;
  if (sel1 -> selected != sel2 -> selected) {
      return -9;
  }
  if (num == sel1 -> num_atoms) {
    sel_flg = 1;
  } else {
    sel_flg = 0;
  }

  float *x = sel1 -> coordinates();
  float *y = sel2 -> coordinates();
  // the Kabsch method won't work of the number of atoms is less than 4
  // (and won't work in some cases of n > 4; I think it works so long as
  // three or more planes are needed to intersect all the data points
  switch (sel1 -> selected) {
  case 1: { // simple center of mass alignment
    Matrix4 tmp;
    tmp.translate(-comx[0], -comx[1], -comx[2]);
    tmp.translate(comy[0], comy[1], comy[2]);
    memcpy(mat->mat, tmp.mat, 16*sizeof(float));
    return 0;
  }
  case 3:
  case 2: { // call a Tcl function
    // find the first (n-1) points (from each molecule)
    int pts[4], count = 0;
    int n;
    for (n=0; n<sel1 -> num_atoms; n++) {
      if (sel1 -> on[n]) {
	pts[count++] = n;
	if (sel1 -> selected == 2) {
	  count++;                   // will put y data in pts[3]
	  break;
	}
	if (count == 2) break;
      }
    }
    for (n=0; n<sel2 -> num_atoms; n++) {
      if (sel2 -> on[n]) {
	pts[count++] = n;
	if (sel1 -> selected == 2) {
	  count++;
	  break;
	}
	if (count == 4) break;
      }
    }
    if (count != 4) {
      msgErr << "fit: internal error: couldn't find the two points"
	     << sendmsg;
      return -9;
    }
    // have the points, now call the Tcl function to do the calc for me
    // (it is a lot easier that way)
    char s[(TCL_DOUBLE_SPACE+2) * 24 + 20];
    char *t;
    sprintf(s, "vmd_measure_fit_%dpoints {", sel1 -> selected);
    t = tcl_append_double(s, x[3*pts[0]+0], 0);   // x[1]
    t = tcl_append_double(t, x[3*pts[0]+1], 1);
    t = tcl_append_double(t, x[3*pts[0]+2], 1);
    strcat(t, "} {");
    if (sel1 -> selected == 3) {
      t = tcl_append_double(s, x[3*pts[1]+0], 0);   // x[2]
      t = tcl_append_double(t, x[3*pts[1]+1], 1);
      t = tcl_append_double(t, x[3*pts[1]+2], 1);
      strcat(t, "} {");
    }
    t = tcl_append_double(t, y[3*pts[2]+0], 0);   // y[1]
    t = tcl_append_double(t, y[3*pts[2]+1], 1);
    t = tcl_append_double(t, y[3*pts[2]+2], 1);
    strcat(t, "} {");
    if (sel1 -> selected == 3) {
      t = tcl_append_double(t, y[3*pts[3]+0], 0);   // y[2]
      t = tcl_append_double(t, y[3*pts[3]+1], 1);
      t = tcl_append_double(t, y[3*pts[3]+2], 1);
      strcat(t, "} {");
    }
    t = tcl_append_double(t, comx[0], 0);   // comx
    t = tcl_append_double(t, comx[1], 1);
    t = tcl_append_double(t, comx[2], 1);
    strcat(t, "} {");
    t = tcl_append_double(t, comy[0], 0);   // comy
    t = tcl_append_double(t, comy[1], 1);
    t = tcl_append_double(t, comy[2], 1);
    strcat(t, "}");

    if (Tcl_Eval(uiText -> tclInterp, s) != TCL_OK) {  // call the command
      msgErr << uiText -> tclInterp -> result << sendmsg;
      uiText -> tclInterp -> result[0] = 0;
      return -13;
    }
    // get the matrix from the result
    char res[16*(TCL_DOUBLE_SPACE+4)+10];
    strcpy(res, uiText -> tclInterp -> result);
    int ret_val = tcl_get_matrix("fit: ", uiText -> tclInterp, res, mat);
    Tcl_ResetResult(uiText -> tclInterp);
    if (ret_val != TCL_OK) {
      msgErr << uiText -> tclInterp -> result << sendmsg;
      return -13;
    }

    return 0;
  }
  default:
    break;
  }
  // at this point I know all the data values are good
  
  // a) compute R = r(i,j) = sum( w(n) * (y(n,i)-comy(i)) * (x(n,j)-comx(j)))
  Matrix4 R;
  int i,j;
  float scale = num * num;
  for (i=0; i<3; i++) {
    for (j=0; j<3; j++) {
      float *tmp = &(R.mat[i][j]);
      *tmp = 0.0;
      int nx = sel1 -> num_atoms - 1, ny = sel2 -> num_atoms - 1;
      while (nx >= 0 || ny >= 0) {
	// find the x atom
	while (nx >= 0 && !sel1 -> on[nx]) {
	  nx--;
	}
	// find the y atom
	while (ny >= 0 && !sel2 -> on[ny]) {
	  ny--;
	}
	// found both, so get data
	*tmp += weight[nx] * (y[3*ny+i] - comy[i]) * (x[3*nx+j] - comx[j]) /
	  scale;
	// and count down one
	nx--;
	ny--;
      }
    }
  }

  // b) 1) form R~R
  Matrix4 Rt;
  for (i=0; i<3; i++) {
    for (j=0; j<3; j++) {
      Rt.mat[i][j] = R.mat[j][i];
    }
  }
  Matrix4 RtR(R);
  RtR.multmatrix(Rt);

  // b) 2) find the eigenvalues and eigenvectors
  {
    char s[9 * TCL_DOUBLE_SPACE + 15];
    char *tmp;
    s[0] = '{'; s[1] = 0;
    tmp = tcl_append_double(  s, RtR.mat[0][0], 0);
    tmp = tcl_append_double(tmp, RtR.mat[0][1], 1);
    tmp = tcl_append_double(tmp, RtR.mat[0][2], 1);
    *tmp++ = '}'; *tmp++ = ' '; *tmp++ = '{'; *tmp = 0;
    tmp = tcl_append_double(tmp, RtR.mat[1][0], 0);
    tmp = tcl_append_double(tmp, RtR.mat[1][1], 1);
    tmp = tcl_append_double(tmp, RtR.mat[1][2], 1);
    *tmp++ = '}'; *tmp++ = ' '; *tmp++ = '{'; *tmp = 0;
    tmp = tcl_append_double(tmp, RtR.mat[2][0], 0);
    tmp = tcl_append_double(tmp, RtR.mat[2][1], 1);
    tmp = tcl_append_double(tmp, RtR.mat[2][2], 1);
    *tmp++ = '}'; *tmp = 0;

    if (Tcl_VarEval(uiText -> tclInterp,
		    "measure eigen symmetric ", s, NULL) != TCL_OK) {
      return -12;
    }
  }
  // get the values/vectors
  float evalue[3];
  float evector[3][3];
  int scan_count = 
  sscanf(uiText->tclInterp->result, "{ %f %f %f } { %f %f %f } { %f %f %f } "
	 "{ %f %f %f }", evalue+0, evalue+1, evalue+2,
	 evector[0]+0, evector[0]+1, evector[0]+2,
	 evector[1]+0, evector[1]+1, evector[1]+2,
	 evector[2]+0, evector[2]+1, evector[2]+2);
  if ( scan_count != 12) {
    msgErr << "WRONG NUMBER OF PARAMETERS: " << scan_count << sendmsg;
    msgErr << uiText -> tclInterp -> result << sendmsg;
  }
  Tcl_ResetResult(uiText->tclInterp);

  // b) 4) sort so that the eigenvalues are from largest to smallest
  //      (or rather so a[0] is eigenvector with largest eigenvalue, ...)
  float *a[3];
  a[0] = evector[0];
  a[1] = evector[1];
  a[2] = evector[2];
#define SWAP(qq,ww) {                                           \
    float v; float *v1;                                         \
    v = evalue[qq]; evalue[qq] = evalue[ww]; evalue[qq] = v;    \
    v1 = a[qq]; a[qq] = a[ww]; a[ww] = v1;                      \
}
  if (evalue[0] < evalue[1]) {
    SWAP(0, 1);
  }
  if (evalue[0] < evalue[2]) {
    SWAP(0, 2);
  }
  if (evalue[1] < evalue[2]) {
    SWAP(1, 2);
  }

  // b) 3) normalize the eigenvectors, and a3 = a1 x a2
  normalize(a[0]);
  normalize(a[1]);
  cross_prod(a[2], a[0], a[1]);

  // c) determine b(i) = R*a(i)
  float b[3][3];
  Rt.multpoint3d(a[0], b[0]);
  normalize(b[0]);

  Rt.multpoint3d(a[1], b[1]);
  normalize(b[1]);
  // to be on the safe side, take out the projection along b[0]
  {
    float dp = dot_prod(b[0], b[1]);
    float tmp[3];
    tmp[0] = b[0][0] * dp;
    tmp[1] = b[0][1] * dp;
    tmp[2] = b[0][2] * dp;
    subtract(b[1], b[1], tmp);
    normalize(b[1]);
  }

  cross_prod(b[2], b[0], b[1]);

  // d) compute U = u(i,j) = sum(b(k,i) * a(k,j))
  Matrix4 U;
  for (i=0; i<3; i++) {
    for (j=0; j<3; j++) {
      float *tmp = &(U.mat[j][i]);
      *tmp = 0;
      for (int k=0; k<3; k++) {
	*tmp += b[k][i] * a[k][j];
      }
    }
  }

  // e) apply the offset for com
  Matrix4 tx;
  tx.translate(-comx[0], -comx[1], -comx[2]);
  Matrix4 ty;
  ty.translate(comy[0], comy[1], comy[2]);
  //  U.multmatrix(com);
  ty.multmatrix(U);
  ty.multmatrix(tx);
  memcpy(mat->mat, ty.mat, 16*sizeof(float));
  return 0;
}


//////////////////////////// The following are used for Tcl
/// get the array of weights given a string

// the string can be one of
// the terms from the atom selection symbol table (eg, "mass",
// "occupancy", "x", ...)  or a Tcl list of values.  If a selection is
// given, the symbol table is checked first, though that will only be a
// problem if there is a symbol whose "name" is a number. If the result
// is good, the data returned is "0", num == number of elements given,
// and *data is not NULL (but you will have to delete [] it yourself If
// the data is bad, the value returned is < 0
// CAUTION: this returns success with a string with number of elements
//   != sel -> selected and != sel -> num_atoms.
// I need the selection to get the molecule id; perhaps I need another
// version which takes the molecule id directly?
//  If the weight_string is NULL or "none", an array of 1s is returned
int get_weights(AtomSel *sel, char *weight_string, int *num, float **data)
{
  *data = NULL;
  *num = 0;
  if (!weight_string || !strcmp(weight_string, "none")) {
    if (!sel) {
      return -1;
    }
    *num = sel -> num_atoms;
    *data = new float[*num];
    for (int i=(*num)-1; i>=0; i--) {
      (*data)[i] = 1.0;
    }
    return 0;
  }
  // if a selection string was given, check the symbol table
  while (1) {
    if (!sel) break;
    if (moleculeList -> mol_index_from_id(sel -> molid) < 0) {
      break;
    }
    sel->use();
    int fctn;
    fctn = atomSelParser.find_attribute(weight_string);
    if (fctn < 0) break;

    // the keyword exists, so get the data
    *num = sel -> num_atoms;
    *data = new float[*num];
    double *tmp_data = new double[*num];
    atomSelParser.extract_keyword_info(fctn, *num, tmp_data, sel->on);
    for (int i=(*num) - 1; i>=0; i--) {
      (*data)[i] = tmp_data[i];
    }
    delete [] tmp_data;
    return 0;
  }
  //#ifdef VMDTCL
  {
    // and see if this is a Tcl list with the right number of atoms
    Tcl_Interp *tmp_interp = NULL;
    tmp_interp = Tcl_CreateInterp();
    int list_num;
    char **list_data;
    if (Tcl_SplitList(tmp_interp, weight_string, &list_num, &list_data) !=
	TCL_OK) {
      return -7;
    }
    // convert the numbers to a list
    *data = new float[list_num];
    double tmp_data;
    for (int i=0; i<list_num; i++) {
      if (Tcl_GetDouble(tmp_interp, list_data[i], &tmp_data) != TCL_OK) {
	delete [] (*data);
	*data = NULL;
	free(list_data);
	return -8;
      }
      (*data)[i] = tmp_data;
    }
    *num = list_num;
    Tcl_DeleteInterp(tmp_interp);
    return 0;
  }
  //#else
  //return -7;
  //#endif // VMDTCL
}


/*
Function:  vmd_measure_center
Parameters:  <selection>               // computes with weight == 1
Parameters:  <selection> weight [none|atom value|string array] 
  computes with the weights based on the following:
      none   => weights all 1
      atom value => value from atomSelParser (eg
         mass  => use weight based on mass
         index => use weight based on atom index (0 to n-1)
      string => use string to get weights for each atom.  The string can
         have number == number of selected atoms or total number of atoms

 Examples: 
    vmd_measure_center atomselect12
    vmd_measure_center atomselect12 weight mass
    vmd_measure_center atomselect12 {12 13} [atomselect top "index 2 3"]
 If no weight is given, no weight term is used (computes center of number)
 */
int vmd_measure_center(ClientData, Tcl_Interp *interp, int argc, 
		       char *argv[])
{
  if (argc != 2 && argc != 4 ) {
    interp -> result = "measure: wrong options for 'center': "
      "measure center <selection> ?weight <weightstring>?";
    return TCL_ERROR;
  }
  if (argc == 4 && strcmp(argv[2], "weight")) {
    interp -> result = "measure: center parameter can only be 'weight'";
    return TCL_ERROR;
  }
  
  // get the selection
  AtomSel *sel = tcl_commands_get_sel(argv[1]);
  if (!sel) {
    interp -> result = "measure: center: no atom selection";
    return TCL_ERROR;
  }

  // get the weight
  float *weight = NULL;
  int num;
  {
    int ret_val;
    if (argc == 2) {          // only from atom selection, so weight is 1
      ret_val = get_weights(sel, NULL, &num, &weight);
    } else {
      ret_val = get_weights(sel, argv[3], &num, &weight);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure: center: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
  }

  // compute the center of "mass"
  {
    float com[3];
    int ret_val = measure_center(sel, num, weight, com);
    delete [] weight;
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure: center: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
    char tmp[TCL_DOUBLE_SPACE];
    Tcl_PrintDouble(interp, com[0], tmp);
    Tcl_AppendElement(interp, tmp);
    Tcl_PrintDouble(interp, com[1], tmp);
    Tcl_AppendElement(interp, tmp);
    Tcl_PrintDouble(interp, com[2], tmp);
    Tcl_AppendElement(interp, tmp);
  }

  return TCL_OK;
}

/*
Function:   vmd_measure_rgyr
Parameters: <selection> {weight [none|atom value|string array]}      
// Function: vmd_measure_minmax <selection>
//  Returns: the cartesian range of a selection (min/max){x,y,z}
//  Example: vmd_measure_minmax atomselect76
//     {-5 0 0} {15 10 11.2}
*/
int vmd_measure_minmax(ClientData, Tcl_Interp *interp, int argc, 
		       char *argv[])
{
  if (argc != 2) {
    interp -> result = "measure: 'minmax' takes one parameter, the selection";
    return TCL_ERROR;
  }
  AtomSel *sel = tcl_commands_get_sel(argv[1]);
  float min_coord[3], max_coord[3];
  int ret_val = measure_minmax(sel, min_coord, max_coord);
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure: minmax: ", measure_error(ret_val));
    return TCL_ERROR;
  }
  
  {
    char *tmp = interp -> result;
    tmp[0] = '{'; tmp[1] = 0; tmp++;
    tmp = tcl_append_double(tmp, min_coord[0], 0);
    tmp = tcl_append_double(tmp, min_coord[1], 1);
    tmp = tcl_append_double(tmp, min_coord[2], 1);
    *tmp++ = '}'; *tmp++ = ' '; *tmp++ = '{'; *tmp = 0;
    tmp = tcl_append_double(tmp, max_coord[0], 0);
    tmp = tcl_append_double(tmp, max_coord[1], 1);
    tmp = tcl_append_double(tmp, max_coord[2], 1);
    *tmp++ = '}'; *tmp++ = 0;
  }

  return TCL_OK;
}


/* Function: vmd_measure_rmsd <selection1> <selection2> 
                       {weight [none|atom value|string array]}

   Returns the RMSD between the two selection, taking the weight (if
any) into account.  If number of elements in the weight != num_atoms
in sel1 then (num in weight = num selected in sel1 = num selected in
sel2) else (num in weight = total num in sel1 = total num in sel2).
The weights are taken from the FIRST selection, if needed

Examples:
  set sel1 [atomselect 0 all]
  set sel2 [atomselect 1 all]
  measure rmsd $sel1 $sel2
  measure rmsd $sel1 $sel2 weight mass
  set sel3 [atomselect 0 "index 3 4 5"]
  set sel4 [atomselect 1 "index 8 5 9"]    # gets turned to 5 8 9
  measure rmsd $sel3 $sel4 weight occupancy
*/
int vmd_measure_rmsd(ClientData, Tcl_Interp *interp, int argc, 
		       char *argv[])
{
  if (argc !=3 && argc != 5) {
    interp -> result = "measure: wrong options for 'rmsd': "
      "measure rmsd <selection1> <selection2> ?weight <weightstring>?";
    return TCL_ERROR;
  }
  // get the selections
  AtomSel *sel1 = tcl_commands_get_sel(argv[1]);
  AtomSel *sel2 = tcl_commands_get_sel(argv[2]);
  // get the weight
  float *weight = NULL;
  int num;
  {
    int ret_val;
    if (argc == 3) {
      ret_val = get_weights(sel1, NULL, &num, &weight);
    } else {
      ret_val = get_weights(sel1, argv[4], &num, &weight);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure: rmsd: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
  }
  // compute the rmsd
  {
    float rmsd;
    int ret_val = measure_rmsd(sel1, sel2, num, weight, &rmsd);
    delete [] weight;
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure: rmsd: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
    sprintf(interp -> result, "%f", rmsd);
  }
  return TCL_OK;
}

//////////////////////////////////////////////
// measure fit $sel1 $sel2 [weight <weights>][
int vmd_measure_fit(ClientData, Tcl_Interp *interp, int argc,
		    char *argv[])
{
  if (argc != 3 && argc != 5) {
    interp -> result = "measure: wrong options for 'fit': "
      "measure fit <selection1> <selection2> ?weight <weightstring>?";
    return TCL_ERROR;
  }
  // get the selections
  AtomSel *sel1 = tcl_commands_get_sel(argv[1]);
  AtomSel *sel2 = tcl_commands_get_sel(argv[2]);
  // get the weight
  float *weight = NULL, *weight2 = NULL;
  int num1, num2, ret_val;
  {
    if (argc == 3) {
      ret_val = get_weights(sel1, NULL, &num1, &weight);
    } else {
      ret_val = get_weights(sel1, argv[4], &num1, &weight);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure: fit1: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
  }
  {
    if (argc == 3) {
      ret_val = get_weights(sel2, NULL, &num2, &weight2);
    } else {
      ret_val = get_weights(sel2, argv[4], &num2, &weight2);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure: fit2: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
  }
  // compute the transformation matrix
  Matrix4 T;
  ret_val = measure_fit(sel1, sel2, num1, weight, num2, weight2, &T);
  delete [] weight;
  delete [] weight2;
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure: fit3: ", measure_error(ret_val),
		     NULL);
    return TCL_ERROR;
  }

  // and return the matrix
  tcl_append_matrix(interp, T);
  return TCL_OK;
}

/********************************* init the commands ******************/
void init_measure_commands(Tcl_Interp *interp)
{
  Tcl_CreateCommand(interp, "vmd_measure_center", vmd_measure_center,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "vmd_measure_minmax", vmd_measure_minmax,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "vmd_measure_rmsd", vmd_measure_rmsd,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "vmd_measure_fit", vmd_measure_fit,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
}

